package com.artfess.application.controller;

import com.artfess.application.model.MessageType;
import com.artfess.application.model.MsgTemplate;
import com.artfess.application.persistence.manager.MessageTypeManager;
import com.artfess.application.persistence.manager.MsgTemplateManager;
import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.util.StringUtil;
import com.artfess.uc.api.impl.util.ContextUtil;
import com.artfess.uc.api.model.IUser;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

/**
 * <pre>
 * 描述：portal_message_log 控制器类
 * 构建组：x7
 * 作者:zhangxw
 * 邮箱:zhangxw@jee-soft.cn
 * 日期:2019-05-30 21:34:36
 * 版权：广州宏天软件有限公司
 * </pre>
 */
@RestController
@RequestMapping(value = "/msg/messageType/v1")
@Api(tags = "消息分类")
@ApiGroup(group = {ApiGroupConsts.GROUP_APPLICATION})
public class MessageTypeController extends BaseController<MessageTypeManager, MessageType> {
    @Resource
    MessageTypeManager messageTypeManager;
    @Resource
    MsgTemplateManager msgTemplateManager;

    /**
     * 消息分类列表(分页条件查询)数据
     *
     * @param queryFilter
     * @return
     * @throws Exception PageJson
     * @throws
     */
    @PostMapping("/list")
    @ApiOperation(value = "消息分类列表", httpMethod = "POST", notes = "获取消息分类列表")
    public PageList<MessageType> list(@ApiParam(name = "queryFilter", value = "查询对象") @RequestBody QueryFilter<MessageType> queryFilter) throws Exception {
        return messageTypeManager.query(queryFilter);
    }

    /**
     * 根据ID获取分类
     *
     * @param id
     * @return
     */
    @RequestMapping(value = "getById", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "根据ID获取内容", httpMethod = "GET", notes = "根据ID获取内容")
    public MessageType getById(
            @ApiParam(name = "id", value = "id", required = true) @RequestParam String id) {
        MessageType messageType = messageTypeManager.getById(id);
        return messageType;
    }

    /**
     * 新增消息分类
     *
     * @param
     * @return
     * @throws Exception
     * @throws
     */
    @PostMapping(value = "/save")
    @ApiOperation(value = "新增,更新消息分类", httpMethod = "POST", notes = "新增,更新消息分类")
    public CommonResult<String> save(@ApiParam(name = "MessageType", value = "消息分类", required = true) @RequestBody MessageType messageType) throws Exception {
        String msg = "添加消息分类成功";
        boolean isExist = false;
        if (ObjectUtils.isNotEmpty(messageType)) {
            QueryFilter queryFilter = QueryFilter.build().withDefaultPage();
            String id = messageType.getId();
            String code = messageType.getCode();
            if (StringUtils.isNotBlank(id)) {
                queryFilter.addFilter("id_", id, QueryOP.NOT_IN);
            }
            if (StringUtils.isNotBlank(code)) {
                queryFilter.addFilter("code_", code, QueryOP.EQUAL);
            }
            PageList<MessageType> query = baseService.query(queryFilter);
            isExist = query.getRows().size() > 0;
        }
        if (isExist) {
            return new CommonResult<>(false, "分类编号已经存在，添加失败", null);
        } else {
            if (StringUtil.isEmpty(messageType.getId())) {
                HashMap<String, Object> map = new HashMap<>();
                IUser user= ContextUtil.getCurrentUser();
                if(ObjectUtils.isNotEmpty(user)){
                    map.put("tenant_id_", user.getTenantId());
                }
               int nextSequence =baseService.getNextSequence(map);
                messageType.setSn((short)nextSequence);
                baseService.create(messageType);

            } else {
                baseService.update(messageType);
                msg = "更新消息分类成功";
            }
            return new CommonResult<String>(msg);
        }
    }

    /**
     * 删除消息分类
     *
     * @param id
     * @return
     * @throws Exception
     * @throws
     */
    @DeleteMapping(value = "/remove/{id}")
    @ApiOperation(value = "删除消息分类", httpMethod = "DELETE", notes = "删除消息分类")
    public CommonResult<String> remove(@ApiParam(name = "id", value = "业务主键", required = true) @PathVariable String id) throws Exception {
        String code = messageTypeManager.get(id).getCode();
        MsgTemplate msgTemplate = msgTemplateManager.gettypeKey(code);
        if (msgTemplate == null) {
            messageTypeManager.remove(id);
            return new CommonResult<String>(true, "删除成功");
        } else {
            return new CommonResult<String>(false, "该分类已被应用，删除失败！");
        }
    }


    /**
     * 批量删除消息分类
     *
     * @param ids
     * @return
     * @throws Exception
     * @throws
     */
    @DeleteMapping(value = "/removes")
    @ApiOperation(value = "批量删除消息分类", httpMethod = "DELETE", notes = "批量删除消息分类")
    public CommonResult<String> removes(@ApiParam(name = "ids", value = "多个主键之间用逗号分隔", required = true) @RequestParam String... ids) throws Exception {
        String msg = "删除消息分类成功";
        for (int i = 0; i < ids.length; i++) {
            String name = messageTypeManager.get(ids[i]).getName();
            String code = messageTypeManager.get(ids[i]).getCode();
            MsgTemplate msgTemplate = msgTemplateManager.gettypeKey(code);
            if (msgTemplate != null) {
                msg = name+"删除消息分类失败";
            }else{
                messageTypeManager.removeByIds(Arrays.asList(ids[i]));
            }
        }
        return new CommonResult<String>(msg);
   }

    @Override
    @RequestMapping(value = "updateSequence", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "批量修改排序号", httpMethod = "POST", notes = "批量修改排序号")
    public CommonResult<String> updateSequence(@ApiParam(name = "params", value = "排序参数：Key：ID，Value：排序号") @RequestBody HashMap<String, Integer> params) throws Exception {
        messageTypeManager.updateSequence(params);
        return new CommonResult<>(true, "排序完成");
    }

    @GetMapping("/getHasRightMessageType")
    @ApiOperation(value = "获取当前用户有权限的分类", httpMethod = "GET", notes = "获取当前用户有权限的分类")
    public List<MessageType> getHasRightMeetingRoom() throws Exception {
        QueryFilter queryFilter = QueryFilter.build();
        return messageTypeManager.query(queryFilter).getRows();
    }
}
