package com.artfess.examine.manager.impl;

import cn.hutool.core.util.NumberUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.artfess.base.constants.CodePrefix;
import com.artfess.base.enums.DelStatusEnum;
import com.artfess.base.enums.PaperStatusEnum;
import com.artfess.base.enums.PaperTypeEnum;
import com.artfess.base.enums.QuestionStateEnum;
import com.artfess.base.enums.QuestionTypeEnum;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.enums.ScoreLevelEnum;
import com.artfess.base.exception.BaseException;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.query.FieldRelation;
import com.artfess.base.query.PageBean;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.util.AuthenticationUtil;
import com.artfess.base.util.time.DateUtil;
import com.artfess.examine.dao.ExamPaperBaseDao;
import com.artfess.examine.dao.ExamQuestionsInfoDao;
import com.artfess.examine.dao.ExamQuestionsOptionDao;
import com.artfess.examine.dao.ExamSubjectInfoDao;
import com.artfess.examine.dao.ExamUserRecordDao;
import com.artfess.examine.dao.ExamYearReportDao;
import com.artfess.examine.manager.ExamPaperSettingManager;
import com.artfess.examine.manager.ExamUserEvaluationManager;
import com.artfess.examine.manager.ExamUserRecordDetailManager;
import com.artfess.examine.manager.ExamUserRecordManager;
import com.artfess.examine.model.ExamPaperBase;
import com.artfess.examine.model.ExamPaperSetting;
import com.artfess.examine.model.ExamQuestionsInfo;
import com.artfess.examine.model.ExamQuestionsOption;
import com.artfess.examine.model.ExamSubjectInfo;
import com.artfess.examine.model.ExamUserRecord;
import com.artfess.examine.model.ExamUserRecordDetail;
import com.artfess.examine.model.ExamYearReport;
import com.artfess.examine.vo.*;
import com.artfess.redis.util.RedisUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * 考生考试记录（人员考试成绩） 服务实现类
 *
 * @author min.wu
 * @company 阿特菲斯信息技术有限公司
 * @since 2022-10-19
 */
@Slf4j
@Service
public class ExamUserRecordManagerImpl extends BaseManagerImpl<ExamUserRecordDao, ExamUserRecord> implements ExamUserRecordManager {

    @Autowired
    private ExamUserRecordDetailManager userRecordDetailManager;

    @Autowired
    private ExamPaperSettingManager paperSettingManager;

    @Autowired
    private ExamUserRecordManager userRecordManager;

    @Resource
    private ExamQuestionsOptionDao questionsOptionDao;

    @Autowired
    private RedisUtil redisUtil;

    @Resource
    private ExamPaperBaseDao paperBaseDao;

    @Resource
    private ExamQuestionsInfoDao questionsInfoDao;

    @Resource
    private ExamSubjectInfoDao subjectInfoDao;

    @Resource
    private ExamUserEvaluationManager userEvaluationManager;

    @Resource(name = "bmpExecutorService")
    private ExecutorService executorService;

    @Resource
    private ExamYearReportDao yearReportDao;


    @Override
    public List<QuestionsInfoVo> getQuestionList(ExamReqVo reqVo) {

        return this.baseMapper.getQuestionList(reqVo);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveScore(SubmitAnswerReqVo reqVo) {
        Assert.hasText(reqVo.getRecordId(), "答题明细id不能为空");
        ExamUserRecord userRecord = baseMapper.selectById(reqVo.getRecordId());
        List<ExamUserRecordDetail> detailList = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(reqVo.getQuestionOptionList())) {
            BigDecimal totalScore = BigDecimal.ZERO;
            for (QuestionOptionReqVo option : reqVo.getQuestionOptionList()) {
                if (null == option.getActualScore()) {
                    continue;
                }

                QueryWrapper<ExamUserRecordDetail> queryWrapper = new QueryWrapper<>();
                queryWrapper.eq("record_id_", reqVo.getRecordId());
                queryWrapper.eq("question_id_", option.getQuestionId());
                ExamUserRecordDetail userRecordDetail = userRecordDetailManager.getBaseMapper().selectOne(queryWrapper);
                if (null != userRecordDetail) {
                    if (QuestionTypeEnum.fillIn.getType().equals(userRecordDetail.getQuestionType())
                            || QuestionTypeEnum.shortAnswer.getType().equals(userRecordDetail.getQuestionType())
                            || QuestionTypeEnum.operation.getType().equals(userRecordDetail.getQuestionType())
                            || QuestionTypeEnum.lst.getType().equals(userRecordDetail.getQuestionType())) {
                        if(null == userRecordDetail.getActualScore()) {
                            userRecordDetail.setActualScore(BigDecimal.ZERO);
                        }
                        if(StringUtils.isEmpty(userRecordDetail.getResult()) && option.getActualScore().longValue() > 0) {
                            throw new BaseException("题目未填写内容分数不能大于0，请检查评卷分数。");
                        }
                        BigDecimal actualScore = BigDecimal.ZERO;
                        if(null != option.getActualScore()) {
                            actualScore = option.getActualScore();
                        }
                        if(actualScore.longValue() == userRecordDetail.getScore().longValue()) {
                            userRecordDetail.setIsRight("1");
                        }else if(actualScore.longValue() < userRecordDetail.getScore().longValue()){
                            userRecordDetail.setIsRight("2");
                        }else if(actualScore.longValue() == 0){
                            userRecordDetail.setIsRight("0");
                        }
                        userRecordDetail.setActualScore(actualScore);
                        totalScore = totalScore.add(option.getActualScore());

                        detailList.add(userRecordDetail);
                    }else {
                        totalScore = totalScore.add(userRecordDetail.getActualScore());
                    }
                }
            }
            userRecord.setSysScore(totalScore);
        }
        userRecord.setLevel(ScoreLevelEnum.getLevel(userRecord.getSysScore()));
        userRecord.setMarkTime(LocalDateTime.now());
        userRecord.setMarkUser(AuthenticationUtil.getCurrentUsername());
        userRecord.setStatus(QuestionStateEnum.finish.getType());
        int i = baseMapper.updateById(userRecord);
        if(i > 0) {
            userRecordDetailManager.updateBatchById(detailList);
            executorService.execute(() -> {
                try {
                    userEvaluationManager.userEvaluation(userRecord);
                } catch (Exception e) {
                    log.error("考生年度综合评定失败:{}", e);
                }
            });
        }

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MyExamInfoVo startExam(ExamReqVo reqVo) {
        Assert.hasText(reqVo.getRecordId(), "考试试卷id不能为空");
        ExamUserRecord examUserRecord = userRecordManager.getById(reqVo.getRecordId());
        if(!examUserRecord.getUserId().equals(reqVo.getUserId())) {
            throw new RuntimeException("当前试卷发生错误！");
        }
        Assert.notNull(examUserRecord, "考试任务不存在，请联系管理员");

        QueryWrapper<ExamPaperSetting> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("paper_id_", examUserRecord.getPaperId());
        ExamPaperSetting paperSetting = paperSettingManager.getOne(queryWrapper);
        boolean dateEquals = DateUtil.isDateLittle(LocalDateTime.now(), paperSetting.getPlanStartTime());
        if (dateEquals) {
            throw new RuntimeException("当前试卷还未开始");

        }
        ExamPaperBase examPaperBase = paperBaseDao.selectById(examUserRecord.getPaperId());
        if(null == examPaperBase) {
            throw new RuntimeException("当前试卷已删除，不能进行考试");
        }
        if (QuestionStateEnum.haveTest.getType().equals(examUserRecord.getStatus())) {
            throw new RuntimeException("当前试卷您已交卷，不能再次考试");
        }

        if (QuestionStateEnum.finish.getType().equals(examUserRecord.getStatus())) {
            throw new RuntimeException("当前试卷已阅卷，不能再次考试");
        }

        if (QuestionStateEnum.zuobi.getType().equals(examUserRecord.getStatus())) {
            throw new RuntimeException("考试中有作弊行为，已被强制交卷");
        }

        MyExamInfoVo myExamInfoVo = new MyExamInfoVo();
        BeanUtils.copyProperties(examUserRecord, myExamInfoVo);
        myExamInfoVo.setTotalNumber(examPaperBase.getTotalNumber());
        myExamInfoVo.setTotalScore(examPaperBase.getTotalScore());
        if(null == examUserRecord.getSwitchCount()) {
            myExamInfoVo.setInSwitchCount(0);
        }else{
            myExamInfoVo.setInSwitchCount(examUserRecord.getSwitchCount());
        }
        myExamInfoVo.setSwitchCount(paperSetting.getSwitchCount());
        reqVo.setRecordId(examUserRecord.getId());
        //总时长（毫秒）
        BigDecimal totalTime = paperSetting.getTimeLength().multiply(new BigDecimal(60)).multiply(new BigDecimal(1000));
        if (QuestionStateEnum.inTest.getType().equals(examUserRecord.getStatus())) {
            //继续考试
            //判断考试是否结束
            getBalanceAwswerTime(reqVo);

            Long useTime = System.currentTimeMillis() - myExamInfoVo.getStartTime().toInstant(ZoneOffset.of("+8")).toEpochMilli();
            //获得剩余毫秒数
            myExamInfoVo.setTimeLength(totalTime.subtract(new BigDecimal(useTime)));

        } else {
            //第一次考试
            //1.创建答题redis定时任务
            //2.发送延时消息
            //3.修改考试任务状态，更新开始考试时间
            examUserRecord.setStatus(QuestionStateEnum.inTest.getType());
            //考虑接口延迟
            examUserRecord.setStartTime(LocalDateTime.now());

            //修改或创建考试活动任务并修改剩余考试时间
            myExamInfoVo.setTimeLength(totalTime);
            createUserTask(reqVo, paperSetting.getTimeLength());
        }
        userRecordManager.updateById(examUserRecord);
        List<QuestionsInfoVo> questionList = userRecordManager.getQuestionList(reqVo);
        if(CollectionUtils.isEmpty(questionList)) {
            throw new RuntimeException("当前试卷沒有题目信息");
        }
        List<String> questionIds = questionList.stream().map(QuestionsInfoVo::getQuestionId).collect(Collectors.toList());
        QueryWrapper<ExamQuestionsOption> optionQueryWrapper = new QueryWrapper<>();
        optionQueryWrapper.in("question_id_", questionIds);
        optionQueryWrapper.orderByAsc("option_key_");
        List<ExamQuestionsOption> examQuestionsOptions = questionsOptionDao.selectList(optionQueryWrapper);

        Map<String, List<ExamQuestionsOption>> map = examQuestionsOptions.stream().collect(Collectors.groupingBy(ExamQuestionsOption::getQuestionId));
        questionList.forEach(question -> {
            question.setRightOption(null);
            if (!map.containsKey(question.getQuestionId())) {
                return;
            }
            question.setOptions(map.get(question.getQuestionId()));
        });
        myExamInfoVo.setQuestionsInfoVos(questionList);
        myExamInfoVo.setRecordId(examUserRecord.getId());
        return myExamInfoVo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void submitAnswer(SubmitAnswerReqVo reqVo) {

        ExamUserRecord examUserRecord = userRecordManager.getBaseMapper().selectById(reqVo.getRecordId());
        Assert.notNull(examUserRecord, "考试任务不存在，请联系管理员");
        QueryWrapper<ExamPaperSetting> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("paper_id_", examUserRecord.getPaperId());
        ExamPaperSetting paperSetting = paperSettingManager.getOne(queryWrapper);
        Assert.notNull(paperSetting, "考试信息不存在");
//        boolean dateEquals = DateUtil.isDateLittle(LocalDateTime.now(), paperSetting.getPlanStartTime());
//        if (dateEquals) {
//            throw new RuntimeException("当前试卷还未开始");
//
//        }

        if (QuestionStateEnum.haveTest.getType().equals(examUserRecord.getStatus())) {
            throw new RuntimeException("当前试卷您已交卷，不能再次考试");
        }

        if (QuestionStateEnum.finish.getType().equals(examUserRecord.getStatus())) {
            throw new RuntimeException("当前试卷已阅卷，不能再次考试");
        }

        if (QuestionStateEnum.zuobi.getType().equals(examUserRecord.getStatus())) {
            throw new RuntimeException("考试中有作弊行为，已被强制交卷");
        }
        boolean b = DateUtil.belongCalendar(LocalDateTime.now(), paperSetting.getPlanStartTime(), paperSetting.getPlanEndTime());

        if (!b && paperSetting.getStatus().equals("2")) {
            throw new RuntimeException("考试任务已结束");
        }

        //已答的不做修改
        List<QuestionOptionReqVo> options = reqVo.getQuestionOptionList();
        //将用户回答的问题以及问题答案放入map中，跟正确答案作对比
        Map<String, String> userOptionNumber = Maps.newHashMap();
        options.forEach(option -> {
            if (userOptionNumber.containsKey(option.getQuestionId())) {
                userOptionNumber.put(option.getQuestionId(), userOptionNumber.get(option.getQuestionId()) + "," + option.getResult());
            } else {
                userOptionNumber.put(option.getQuestionId(), option.getResult());
            }
        });
        //判断是否交卷
        if (null != reqVo.getStatus() && reqVo.getStatus() == 1) {
            examUserRecord.setStatus(QuestionStateEnum.haveTest.getType());
            long time = System.currentTimeMillis() - examUserRecord.getStartTime().toInstant(ZoneOffset.of("+8")).toEpochMilli();
            //初始化Formatter的转换格式。
            SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
            formatter.setTimeZone(TimeZone.getTimeZone("GMT+00:00"));
            String answerTime = formatter.format(time);
            examUserRecord.setAnswerTime(answerTime);
            examUserRecord.setEndTime(LocalDateTime.now());
            if(null != paperSetting.getTrainModel() && 1 == paperSetting.getTrainModel()){
                examUserRecord.setStatus(QuestionStateEnum.finish.getType());
            }
        }
        //如果是试卷则给答题内容打分
        calculateScore(userOptionNumber, examUserRecord);

    }

    @Override
    public List<UserInfoVo> findByPaperId(String paperId) {
        return this.baseMapper.findByPaperId(paperId);
    }

    @Override
    public PageList<ExamUserRecord> myPaper(QueryFilter<ExamUserRecord> queryFilter) {
        String year = String.valueOf(LocalDateTime.now().getYear());
        QueryWrapper<ExamYearReport> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("year_", year);
        Integer count = yearReportDao.selectCount(queryWrapper);

        PageBean pageBean = queryFilter.getPageBean();
        AtomicReference<String> paperStatus = new AtomicReference<>();
        queryFilter.getQuerys().forEach(queryField -> {
            //如果是模拟训练则只能查看自己创建的试卷
            if("paperStatus".equals(queryField.getProperty())){
                paperStatus.set(queryField.getValue() + "");
            }
        });
        if(PaperStatusEnum.notRelease.getType().equals(paperStatus.get())) {
            queryFilter.addFilter("sp.plan_start_time_", LocalDateTime.now(), QueryOP.GREAT, FieldRelation.AND);
        } else if(PaperStatusEnum.inTets.getType().equals(paperStatus.get())) {
            queryFilter.addFilter("sp.plan_start_time_", LocalDateTime.now(), QueryOP.LESS, FieldRelation.AND);
            queryFilter.addFilter("sp.plan_end_time_", LocalDateTime.now(), QueryOP.GREAT_EQUAL, FieldRelation.AND);
        } else if (PaperStatusEnum.finish.getType().equals(paperStatus.get())) {
            queryFilter.addFilter("sp.plan_end_time_", LocalDateTime.now(), QueryOP.LESS, FieldRelation.AND);
        }
        if(!StringUtils.isEmpty(paperStatus.get())) {
            queryFilter.getQuerys().removeIf(s -> "paperStatus".equals(s.getProperty()));
        }
        queryFilter.addFilter("b.is_dele_",DelStatusEnum.N.getType(), QueryOP.EQUAL);
        IPage<ExamUserRecord> result = baseMapper.myPaper(convert2IPage(pageBean), convert2Wrapper(queryFilter, currentModelClass()));
        result.getRecords().forEach(userRecord -> {
            boolean dateEquals = DateUtil.isDateLittle(LocalDateTime.now(), userRecord.getPlanEndTime());
            if (dateEquals) {
                userRecord.setPaperStatus(PaperStatusEnum.inTets.getType());
            } else {
                userRecord.setPaperStatus(PaperStatusEnum.finish.getType());
            }

            boolean dateLittle = DateUtil.isDateLittle(LocalDateTime.now(), userRecord.getPlanStartTime());
            if (dateLittle) {
                userRecord.setPaperStatus(PaperStatusEnum.notRelease.getType());
            }

            if (QuestionStateEnum.notTo.getType().equals(userRecord.getStatus()) && dateEquals) {
                userRecord.setStatus(QuestionStateEnum.toBeAnswer.getType());
            }
            if(count > 0) {
                userRecord.setReportStatus("1");
            }else {
                userRecord.setReportStatus("0");
            }
        });

        return new PageList<ExamUserRecord>(result);
    }

    @Override
    public MyExamInfoVo getUserRecord(String id) {
        ExamUserRecord examUserRecord = userRecordManager.getById(id);
        Assert.notNull(examUserRecord, "考试任务不存在，请联系管理员");
        ExamReqVo reqVo = new ExamReqVo();
        reqVo.setRecordId(id);
        List<QuestionsInfoVo> questionList = userRecordManager.getQuestionList(reqVo);
        if(CollectionUtils.isEmpty(questionList)) {
            return new MyExamInfoVo();
        }
        List<String> questionIds = questionList.stream().map(QuestionsInfoVo::getQuestionId).collect(Collectors.toList());
        QueryWrapper<ExamQuestionsOption> optionQueryWrapper = new QueryWrapper<>();
        optionQueryWrapper.in("question_id_", questionIds);
        optionQueryWrapper.orderByAsc("option_key_");
        List<ExamQuestionsOption> examQuestionsOptions = questionsOptionDao.selectList(optionQueryWrapper);

        Map<String, List<ExamQuestionsOption>> map = examQuestionsOptions.stream().collect(Collectors.groupingBy(ExamQuestionsOption::getQuestionId));
        questionList.forEach(question -> {
            if(StringUtils.isEmpty(question.getResult())) {
                question.setActualScore(BigDecimal.ZERO);
            }
            if (!map.containsKey(question.getQuestionId())) {
                return;
            }
            question.setOptions(map.get(question.getQuestionId()));
        });
        MyExamInfoVo myExamInfoVo = new MyExamInfoVo();
        BeanUtils.copyProperties(examUserRecord, myExamInfoVo);
        reqVo.setRecordId(examUserRecord.getId());
        myExamInfoVo.setQuestionsInfoVos(questionList);
        myExamInfoVo.setRecordId(examUserRecord.getId());
        return myExamInfoVo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void switchCount(String id) {

        ExamUserRecord examUserRecord = userRecordManager.getBaseMapper().selectById(id);
        Assert.notNull(examUserRecord, "考试任务不存在，请联系管理员");
        QueryWrapper<ExamPaperSetting> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("paper_id_", examUserRecord.getPaperId());
        ExamPaperSetting paperSetting = paperSettingManager.getOne(queryWrapper);
        Assert.notNull(paperSetting, "考试信息不存在");
        if (null == paperSetting.getSwitchCount()) {
            return;
        }
        if (null == examUserRecord.getSwitchCount()) {
            examUserRecord.setSwitchCount(1);
        } else {
            examUserRecord.setSwitchCount(examUserRecord.getSwitchCount() + 1);
        }

        if (paperSetting.getSwitchCount() <= examUserRecord.getSwitchCount()) {
            throw new BaseException(ResponseErrorEnums.Compulsory_PAPER_EXCEPTION);
        }
        this.baseMapper.updateById(examUserRecord);

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void applyExamPaper(String id) {
        Assert.hasText(id, "请选择试卷");
        ExamUserRecord examUserRecord = userRecordManager.getBaseMapper().selectById(id);
        examUserRecord.setStatus("6");
        examUserRecord.setApplyTime(LocalDateTime.now());
        this.baseMapper.updateById(examUserRecord);

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean modifyStatus(ApplyPaperReqVo reqVo) {
        Assert.hasText(reqVo.getId(), "请选择申请记录");
        ExamUserRecord examUserRecord = userRecordManager.getBaseMapper().selectById(reqVo.getId());
        Assert.notNull(examUserRecord, "申请记录不存在，请联系管理员");
        if ("0" == reqVo.getStatus()) {
            examUserRecord.setStatus("8");
        } else {
            examUserRecord.setStatus("7");
        }
        int i = this.baseMapper.updateById(examUserRecord);
        if (i > 0) {
            return true;
        }
        return false;
    }

    @Override
    public PageList<PaperStatisticalVo> paperStatistical(QueryFilter<ExamPaperBase> queryFilter) {

        PageBean pageBean = queryFilter.getPageBean();
        IPage<PaperStatisticalVo> result = baseMapper.paperStatistical(convert2IPage(pageBean), convert2Wrapper(queryFilter, currentModelClass()));

        result.getRecords().forEach(paperStatisticalVo -> {
            paperStatisticalVo.setTotalNumber(paperStatisticalVo.getActualCount() + paperStatisticalVo.getLackCount());
            if(null == paperStatisticalVo.getPassCount() || 0 == paperStatisticalVo.getPassCount().doubleValue()) {
                paperStatisticalVo.setPassPercentage(0+"%");
            }else{
                BigDecimal multiply = paperStatisticalVo.getPassCount().divide(new BigDecimal(paperStatisticalVo.getTotalNumber()), 2, BigDecimal.ROUND_HALF_UP).multiply(new BigDecimal("100"));
                paperStatisticalVo.setPassPercentage(multiply+"%");
            }



            boolean dateEquals = DateUtil.isDateLittle(LocalDateTime.now(), paperStatisticalVo.getPlanEndTime());
            if (dateEquals) {
                paperStatisticalVo.setPaperStatus(PaperStatusEnum.inTets.getType());
            } else {
                paperStatisticalVo.setPaperStatus(PaperStatusEnum.finish.getType());
            }

            boolean dateLittle = DateUtil.isDateLittle(LocalDateTime.now(), paperStatisticalVo.getPlanStartTime());
            if (dateLittle) {
                paperStatisticalVo.setPaperStatus(PaperStatusEnum.notRelease.getType());
            }
            paperStatisticalVo.setAvgScore(new BigDecimal(paperStatisticalVo.getAvgScore()).setScale(2, RoundingMode.HALF_UP).toString());

        });
        return new PageList<PaperStatisticalVo>(result);
    }

    @Override
    public PaperAnalysisVo paperAnalysis() {
        QueryWrapper<ExamQuestionsInfo> questionsInfoQueryWrapper = new QueryWrapper<>();
        questionsInfoQueryWrapper.eq("is_dele_", DelStatusEnum.N.getType());
        Integer questionCount = questionsInfoDao.selectCount(questionsInfoQueryWrapper);

        QueryWrapper<ExamPaperBase> paperBaseQueryWrapper = new QueryWrapper<>();
        paperBaseQueryWrapper.eq("is_dele_", DelStatusEnum.N.getType());
        paperBaseQueryWrapper.eq("type_", PaperTypeEnum.formal.getType());
        Integer paperCount = paperBaseDao.selectCount(paperBaseQueryWrapper);

        QueryWrapper<ExamSubjectInfo> subjectInfoQueryWrapper = new QueryWrapper<>();
        subjectInfoQueryWrapper.eq("is_dele_", DelStatusEnum.N.getType());
        Integer subjectCount = subjectInfoDao.selectCount(subjectInfoQueryWrapper);

        Integer userCount = this.baseMapper.selectUserCount();
        List<SubjectQuestionTypeVo> questionTypeList = questionsInfoDao.selectTypeCount();

        PaperAnalysisVo paperAnalysisVo = new PaperAnalysisVo();
        paperAnalysisVo.setQuestionCount(questionCount);
        paperAnalysisVo.setPaperCount(paperCount);
        paperAnalysisVo.setSubjectCount(subjectCount);
        paperAnalysisVo.setUserCount(userCount);
        paperAnalysisVo.setQuestionTypeList(questionTypeList);

        List<JSONObject> difficulty = questionsInfoDao.selectDifficulty();
        paperAnalysisVo.setDifficulty(difficulty);
        return paperAnalysisVo;
    }

    @Override
    public JobStatisticalVo jobStatistical() {
        List<ExamUserRecord> records = this.baseMapper.myUserRecord(AuthenticationUtil.getCurrentUserId());
        JobStatisticalVo jobStatisticalVo = new JobStatisticalVo();
        Integer recordCount = 0;
        Integer passCount = 0;
        Integer lackCount = 0;
        Integer zbCount = 0;
        for(ExamUserRecord userRecord : records) {
            if(QuestionStateEnum.ksStatusList().contains(userRecord.getStatus())) {
                recordCount ++;
            }
            if(null != userRecord.getSysScore() && null != userRecord.getTotalNumber()) {
                boolean equals = NumberUtil.equals(userRecord.getTotalScore(),userRecord.getSysScore());
                if(equals) {
                    passCount ++;
                }
            }

            boolean dateEquals = DateUtil.isDateLittle(LocalDateTime.now(), userRecord.getPlanEndTime());
            if (!dateEquals && QuestionStateEnum.qkStatusList().contains(userRecord.getStatus())) {
                lackCount ++;
            }
            if(QuestionStateEnum.zuobi.getType().equals(userRecord.getStatus())) {
                zbCount ++;
            }
        }
        jobStatisticalVo.setRecordCount(recordCount);
        jobStatisticalVo.setLackCount(lackCount);
        jobStatisticalVo.setPassCount(passCount);
        jobStatisticalVo.setZbCount(zbCount);
        return jobStatisticalVo;
    }

    @Override
    public void updateStatus(String id) {
        Assert.hasText(id, "请选择考试id");
        ExamUserRecord examUserRecord = baseMapper.selectById(id);
        Assert.notNull(examUserRecord, "考试记录不存在，请联系管理员");
        examUserRecord.setStatus(QuestionStateEnum.zuobi.getType());
        this.baseMapper.updateById(examUserRecord);
    }

    @Override
    public PageList<ExamUserRecord> detailPage(QueryFilter<ExamUserRecord> queryFilter) {
        PageBean pageBean = queryFilter.getPageBean();
        AtomicReference<String> paperStatus = new AtomicReference<>();
        queryFilter.getQuerys().forEach(queryField -> {
            //如果是模拟训练则只能查看自己创建的试卷
            if("paperStatus".equals(queryField.getProperty())){
                paperStatus.set(queryField.getValue() + "");
            }
        });
        if(PaperStatusEnum.notRelease.getType().equals(paperStatus.get())) {
            queryFilter.addFilter("sp.plan_start_time_", LocalDateTime.now(), QueryOP.GREAT, FieldRelation.AND);
        } else if(PaperStatusEnum.inTets.getType().equals(paperStatus.get())) {
            queryFilter.addFilter("sp.plan_start_time_", LocalDateTime.now(), QueryOP.LESS, FieldRelation.AND);
            queryFilter.addFilter("sp.plan_end_time_", LocalDateTime.now(), QueryOP.GREAT_EQUAL, FieldRelation.AND);
        } else if (PaperStatusEnum.finish.getType().equals(paperStatus.get())) {
            queryFilter.addFilter("sp.plan_end_time_", LocalDateTime.now(), QueryOP.LESS, FieldRelation.AND);
        }
        if(!StringUtils.isEmpty(paperStatus.get())) {
            queryFilter.getQuerys().removeIf(s -> "paperStatus".equals(s.getProperty()));
        }

        IPage<ExamUserRecord> result = baseMapper.detailPage(convert2IPage(pageBean), convert2Wrapper(queryFilter, currentModelClass()));
        result.getRecords().forEach(userRecord -> {
            boolean dateEquals = DateUtil.isDateLittle(LocalDateTime.now(), userRecord.getPlanEndTime());
            if (dateEquals) {
                userRecord.setPaperStatus(PaperStatusEnum.inTets.getType());
            } else {
                userRecord.setPaperStatus(PaperStatusEnum.finish.getType());
            }

            boolean dateLittle = DateUtil.isDateLittle(LocalDateTime.now(), userRecord.getPlanStartTime());
            if (dateLittle) {
                userRecord.setPaperStatus(PaperStatusEnum.notRelease.getType());
            }

            if (QuestionStateEnum.notTo.getType().equals(userRecord.getStatus()) && dateEquals) {
                userRecord.setStatus(QuestionStateEnum.toBeAnswer.getType());
            }
        });

        return new PageList<ExamUserRecord>(result);
    }

    @Override
    public MyExamInfoVo myPaperInfo(String id) {
        ExamUserRecord examUserRecord = userRecordManager.getById(id);
        Assert.notNull(examUserRecord, "考试任务不存在，请联系管理员");
        ExamReqVo reqVo = new ExamReqVo();
        reqVo.setRecordId(id);
        List<QuestionsInfoVo> questionList = userRecordManager.getQuestionList(reqVo);
        if(CollectionUtils.isEmpty(questionList)) {
            return new MyExamInfoVo();
        }
        List<String> questionIds = questionList.stream().map(QuestionsInfoVo::getQuestionId).collect(Collectors.toList());
        QueryWrapper<ExamQuestionsOption> optionQueryWrapper = new QueryWrapper<>();
        optionQueryWrapper.in("question_id_", questionIds);
        optionQueryWrapper.orderByAsc("option_key_");
        List<ExamQuestionsOption> examQuestionsOptions = questionsOptionDao.selectList(optionQueryWrapper);

        Map<String, List<ExamQuestionsOption>> map = examQuestionsOptions.stream().collect(Collectors.groupingBy(ExamQuestionsOption::getQuestionId));
        questionList.forEach(question -> {
            if(StringUtils.isEmpty(question.getResult())) {
                question.setActualScore(BigDecimal.ZERO);
            }
            if (!map.containsKey(question.getQuestionId())) {
                return;
            }
            question.setOptions(map.get(question.getQuestionId()));
        });
        MyExamInfoVo myExamInfoVo = new MyExamInfoVo();
        BeanUtils.copyProperties(examUserRecord, myExamInfoVo);
        reqVo.setRecordId(examUserRecord.getId());
        myExamInfoVo.setQuestionsInfoVos(questionList);
        myExamInfoVo.setRecordId(examUserRecord.getId());
        myExamInfoVo.setUserScore(examUserRecord.getSysScore());
        return myExamInfoVo;
    }

    @Override
    public List<YearSubjectScoreVO> getYearSubjectScore(String userId, Integer year) {
        return baseMapper.getYearSubjectScoreByUserId(userId, year);
    }


    @Override
    public ExamPaperBase findByMyRecord(String id) {
        Assert.hasText(id, "请选择考试id");
        ExamUserRecord examUserRecord = userRecordManager.getBaseMapper().selectById(id);
        Assert.notNull(examUserRecord, "考试任务不存在，请联系管理员");
        QueryWrapper<ExamPaperSetting> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("paper_id_", examUserRecord.getPaperId());
        ExamPaperSetting paperSetting = paperSettingManager.getOne(queryWrapper);
        Assert.notNull(paperSetting, "考试信息不存在");
        ExamPaperBase examPaperBase = paperBaseDao.selectById(examUserRecord.getPaperId());
        if(null == examPaperBase) {
            return new ExamPaperBase();
        }
        examPaperBase.setPassScore(paperSetting.getPassScore());
        examPaperBase.setTimeLength(paperSetting.getTimeLength());
        return examPaperBase;
    }


    /**
     * 给用户回答的问题进行打分
     *
     * @param userOptionNumber
     * @param record
     */
    private void calculateScore(Map<String, String> userOptionNumber, ExamUserRecord record) {
        log.info("问题选项：{}", userOptionNumber);
        //如果已经生成试卷题目则直接返回生成的记录
        QueryWrapper<ExamUserRecordDetail> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("record_id_", record.getId());
        List<ExamUserRecordDetail> list = userRecordDetailManager.getBaseMapper().selectList(queryWrapper);
        BigDecimal totalScore = BigDecimal.ZERO;
        for (ExamUserRecordDetail question : list) {
            String result = userOptionNumber.get(question.getQuestionId());
            question.setResult(result);
            //未回答题目直接跳过
            if (!userOptionNumber.containsKey(question.getQuestionId())) {
                continue;
            }
            if(QuestionTypeEnum.multi.getType().equals(question.getQuestionType())) {
                String[] split = result.split(",");
                Arrays.sort(split);
                result = String.join(",", split);
                String[] rightKeys = question.getRightKey().split(",");
                Arrays.sort(rightKeys);
                question.setRightKey(String.join(",", rightKeys));
            }
            if (QuestionTypeEnum.radio.getType().equals(question.getQuestionType())
                    || QuestionTypeEnum.multi.getType().equals(question.getQuestionType())
                    || QuestionTypeEnum.judge.getType().equals(question.getQuestionType())) {
                if (result.equals(question.getRightKey())) {
                    question.setActualScore(question.getScore());
                    question.setIsRight("1");
                } else {
                    question.setActualScore(BigDecimal.ZERO);
                    question.setIsRight("0");
                }
                totalScore = totalScore.add(question.getActualScore());
            }
        }
        userRecordDetailManager.updateBatchById(list);
        log.info("问题详情：{}", JSON.toJSONString(list));
        record.setSysScore(totalScore);
        userRecordManager.updateById(record);
    }


    private void createUserTask(ExamReqVo reqVo, BigDecimal timeLength) {
        String key = CodePrefix.PAPER_TASK_KEY.getKey() + ":" + reqVo.getRecordId();
        Long time = timeLength.longValue() * 60;
        redisUtil.set(key, JSON.toJSON(reqVo), time);
    }

    private void getBalanceAwswerTime(ExamReqVo reqVo) {

        String key = CodePrefix.PAPER_TASK_KEY.getKey() + ":" + reqVo.getRecordId();
        Object taskInfo = redisUtil.get(key);
        Assert.notNull(taskInfo, "考试已结束");
    }
}
