package com.artfess.device.base.controller;


import com.alibaba.fastjson.JSON;
import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.annotation.PowerLogInfo;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.enums.LogType;
import com.artfess.base.enums.OperationType;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.model.CommonResult;
import com.artfess.base.valid.AddGroup;
import com.artfess.base.valid.UpdateGroup;
import com.artfess.device.base.manager.DeviceProductTypeManager;
import com.artfess.device.base.model.DeviceProductType;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

/**
 * 设备产品类型信息 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author min.wu
 * @since 2022-07-15
 */
@Slf4j
@RestController
@Api(tags = "设施设备-产品分类管理")
@RequestMapping("/device/product/type/")
@ApiGroup(group = {ApiGroupConsts.DEVICE_BIZ})
public class DeviceProductTypeController extends BaseController<DeviceProductTypeManager, DeviceProductType> {

    @Override
    @PostMapping("/")
    @ApiOperation("添加实体的接口")
    public CommonResult<String> create(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({AddGroup.class}) DeviceProductType t) {
        String id = baseService.insertTree(t);
        if(!StringUtils.isNotBlank(id)) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
        return new CommonResult<>();
    }

    @Override
    @PutMapping("/")
    @ApiOperation("更新实体")
    public CommonResult<String> updateById(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({UpdateGroup.class})  DeviceProductType t) {
        String id = baseService.updateProductType(t);
        if(!StringUtils.isNotBlank(id)) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "更新实体失败");
        }
        return new CommonResult<>();
    }

    @PostMapping("/getTree")
    @ApiOperation("获取产品分类下拉树")
    @PowerLogInfo(logType = LogType.BIZ, operaionType = OperationType.QUERY, description = "获取产品分类下拉树")
    public CommonResult<String> getTree(@ApiParam(name = "model", value = "获取产品分类下拉树") @RequestBody DeviceProductType entity) {
        log.info("获取产品分类下拉树请求参数:{}", JSON.toJSONString(entity));
        List<DeviceProductType> list = baseService.getTree(entity);
        return CommonResult.success(list, null);
    }

    @PostMapping("/getAllTree")
    @ApiOperation("获取产品分类（包括摄像头）下拉树")
    @PowerLogInfo(logType = LogType.BIZ, operaionType = OperationType.QUERY, description = "获取产品分类（包括摄像头）下拉树")
    public CommonResult<String> getAllTree(@ApiParam(name = "model", value = "获取产品分类（包括摄像头）下拉树") @RequestBody DeviceProductType entity) {
        log.info("获取产品分类（包括摄像头）下拉树:{}", JSON.toJSONString(entity));
        List<DeviceProductType> list = baseService.getAllTree(entity);
        return CommonResult.success(list, null);
    }

}
