package net.zstong.awump.base.web;


import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import net.zstong.awump.base.entity.AwRegionEntity;
import net.zstong.awump.base.service.AwRegionService;
import org.hswebframework.web.api.crud.entity.QueryNoPagingOperation;
import org.hswebframework.web.api.crud.entity.QueryParamEntity;
import org.hswebframework.web.api.crud.entity.TreeSupportEntity;
import org.hswebframework.web.authorization.Authentication;
import org.hswebframework.web.authorization.annotation.Authorize;
import org.hswebframework.web.authorization.annotation.Resource;
import org.hswebframework.web.authorization.annotation.SaveAction;
import org.hswebframework.web.crud.service.ReactiveCrudService;
import org.hswebframework.web.crud.web.reactive.ReactiveServiceCrudController;
import org.hswebframework.web.exception.NotFoundException;
import org.jetlinks.community.device.entity.DeviceCategoryEntity;
import org.springframework.web.bind.annotation.*;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

@RestController
@RequestMapping({"/aw/base/region"})
@Slf4j
@Tag(name = "行政区划管理")
@AllArgsConstructor
@Resource(id = "aw-base-region", name = "库房行政区划管理")
public class AwRegionController implements ReactiveServiceCrudController<AwRegionEntity, String> {


    private final AwRegionService awRegionService;


    @Override
    public ReactiveCrudService<AwRegionEntity, String> getService() {
        return awRegionService;
    }


    @GetMapping
    @QueryNoPagingOperation(summary = "获取全部区划")
    @Authorize(merge = false)
    public Flux<AwRegionEntity> getAllCategory(@Parameter(hidden = true) QueryParamEntity query) {
        return this
            .awRegionService
            .createQuery()
            .setParam(query)
            .fetch();
    }

    @GetMapping("/_tree")
    @QueryNoPagingOperation(summary = "获取全部区划(树结构)")
    @Authorize(merge = false)
    public Flux<AwRegionEntity> getAllCategoryTree(@Parameter(hidden = true) QueryParamEntity query) {
        return this
            .awRegionService
            .createQuery()
            .setParam(query)
            .fetch()
            .collectList()
            .flatMapMany(all -> {
                System.out.println("========GetMapping===========" + all);
                return Flux.fromIterable(TreeSupportEntity.list2tree(all, AwRegionEntity::setChildren));
            });
    }

    @PostMapping("/_tree")
    @QueryNoPagingOperation(summary = "获取全部区划(树结构)")
    @Authorize(merge = false)
    public Flux<AwRegionEntity> getAllCategoryTreeByQueryParam(@RequestBody Mono<QueryParamEntity> query) {
        return this.awRegionService.getAllCategoryTreeByQueryParam(query);
    }


    @PutMapping({"/{id}"})
    @SaveAction
    @Operation(
        summary = "根据ID修改数据"
    )
    public Mono<Boolean> update(@PathVariable String id, @RequestBody Mono<AwRegionEntity> payload) {
        System.out.println(id + "===========payload========" + payload);
        return Authentication.currentReactive().flatMap((auth) -> {
            return payload.map((entity) -> {
                return this.applyAuthentication(entity, auth);
            });
        }).switchIfEmpty(payload).flatMap((entity) -> {
            System.out.println(id + "===========entity========" + entity);

            return this.getService().updateById(id, Mono.just(entity));
        }).doOnNext((i) -> {
            if (i == 0) {
                throw new NotFoundException();
            }
        }).thenReturn(true);
    }

}
