package com.artfess.form.persistence.manager.impl;

import com.artfess.base.datasource.DatabaseContext;
import com.artfess.base.datasource.DatabaseSwitchResult;
import com.artfess.base.exception.BaseException;
import com.artfess.base.handler.MultiTenantHandler;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.util.FileUtil;
import com.artfess.base.util.JAXBUtil;
import com.artfess.base.util.SQLUtil;
import com.artfess.base.util.StringUtil;
import com.artfess.base.util.ThreadMsgUtil;
import com.artfess.base.util.UniqueIdUtil;
import com.artfess.base.util.ZipUtil;
import com.artfess.form.model.ViewManage;
import com.artfess.form.model.ViewManageList;
import com.artfess.form.persistence.dao.ViewManageDao;
import com.artfess.form.persistence.manager.ViewManageManager;
import com.artfess.table.operator.IViewOperator;
import com.artfess.table.util.MetaDataUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.io.File;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * <pre>
 *
 * 描述：视图管理 处理实现类
 * 构建组：x7
 * 作者:pangq
 * 邮箱:pangq@jee-soft.cn
 * 日期:2020-04-30 17:01:50
 * 版权：广州宏天软件股份有限公司
 * </pre>
 */
@Service("viewManageManager")
public class ViewManageManagerImpl extends BaseManagerImpl<ViewManageDao, ViewManage> implements ViewManageManager {
    @Resource
    DatabaseContext databaseContext;

    /**
     * 保存视图
     *
     * @param saveType 保存类型：0仅保存，1保存并创建视图
     */
    @Override
    @Transactional
    public void savePub(ViewManage viewManage, Integer saveType) {
        if (saveType == 1) {
            this.createPhysicalView(viewManage);
            viewManage.setStatus(ViewManage.Generated);
        }
        //取消租户隔离
        MultiTenantHandler.removeThreadLocalIgnore();

        if (StringUtil.isEmpty(viewManage.getId())) {
            QueryWrapper wrapper = new QueryWrapper();
            wrapper.eq("DS_ALIAS_", viewManage.getDsAlias());
            wrapper.eq("VIEW_NAME_", viewManage.getViewName());
            List<ViewManage> viewManageList = baseMapper.selectList(wrapper);
            if (viewManageList != null && !viewManageList.isEmpty()) {
                throw new BaseException("视图：\"" + viewManage.getViewName() + "\"已存在，不要重复添加！");
            }
            this.create(viewManage);
        } else {
            this.update(viewManage);
        }
    }

    @Override
    @Transactional
    public void createPhysicalView(String id) {
        ViewManage viewManage = this.get(id);
        this.createPhysicalView(viewManage);
        viewManage.setStatus(ViewManage.Generated);
        this.update(viewManage);
    }

    public void createPhysicalView(ViewManage viewManage) {
        if (SQLUtil.containsSqlInjection(viewManage.getSql())) {
            throw new BaseException("sql语句含有非法注入！");
        }
        try (DatabaseSwitchResult dResult = databaseContext.setDataSource(viewManage.getDsAlias())) {
            IViewOperator iViewOperator = MetaDataUtil.getIViewOperatorAfterSetDT(dResult.getDbType());
            iViewOperator.createOrRep(viewManage.getViewName(), viewManage.getSql());
        } catch (Exception e) {
            String msg = e.getMessage();
            if (msg != null) {
                if (msg.contains("is not VIEW")) {
                    throw new BaseException("视图名称在数据库已存在，请调整！");
                } else if (msg.contains("bad SQL grammar")) {
                    throw new BaseException("SQL语句存在语法错误，请调整！");
                }
            }
            throw new BaseException(e.getMessage(), e);
        }
    }

    @Override
    public Map<String, String> exportData(List<String> ids) throws Exception {
        Map<String, String> map = new HashMap<>();
        ViewManageList viewManageList = new ViewManageList();
        for (String id : ids) {
            ViewManage viewManage = this.get(id);
            viewManageList.addViewManage(viewManage);
        }
        try {
            String xml = JAXBUtil.marshall(viewManageList, ViewManageList.class);
            map.put("viewManages.form.xml", xml);
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException("导出失败！" + e.getMessage(), e);
        }
        return map;
    }

    @Override
    @Transactional
    public void importData(MultipartFile file) {
        String unZipFilePath = null;
        try {
            String rootRealPath = Paths.get(FileUtil.getIoTmpdir(), "attachFiles/unZip").toString();
            FileUtil.createFolder(rootRealPath, true);
            ZipUtil.unZipFile(file, rootRealPath);
            String name = file.getOriginalFilename();
            String fileDir = StringUtil.substringBeforeLast(name, ".");
            unZipFilePath = Paths.get(rootRealPath, fileDir).toString();
            String xml = FileUtil.readFile(Paths.get(unZipFilePath, "viewManages.form.xml").toString());
            if (StringUtil.isEmpty(xml)) {
                throw new BaseException("导入的xml文件没有内容！");
            }
            ViewManageList list = (ViewManageList) JAXBUtil.unmarshall(xml, ViewManageList.class);
            List<ViewManage> viewManageList = list.getViewManageList();
            for (ViewManage viewManage : viewManageList) {
                importData(viewManage);
            }
        } catch (Exception e) {
            throw new BaseException(e.getMessage(), e);
        } finally {
            if (StringUtil.isNotEmpty(unZipFilePath)) {
                File formDir = new File(unZipFilePath);
                if (formDir.exists()) {
                    FileUtil.deleteDir(formDir);
                }
            }
        }
    }

    private void importData(ViewManage viewManage) {
        QueryWrapper wrapper = new QueryWrapper();
        wrapper.eq("DS_ALIAS_", viewManage.getDsAlias());
        wrapper.eq("VIEW_NAME_", viewManage.getViewName());
        List<ViewManage> viewManageList = baseMapper.selectList(wrapper);
        if (viewManageList != null && !viewManageList.isEmpty()) {
            ThreadMsgUtil.addMapMsg2("viewsSkipped", String.format("<div style='margin-left:12px; margin-top:6px;'>%s [%s]</div>", viewManage.getViewName() + "", viewManage.getDsAlias() + ""));
            return;
        }
        String id = UniqueIdUtil.getSuid();
        viewManage.setId(id);
        if (SQLUtil.containsSqlInjection(viewManage.getSql())) {
            throw new BaseException("SQL语句含有非法注入！");
        }
        try (DatabaseSwitchResult dResult = databaseContext.setDataSource(viewManage.getDsAlias())) {
            IViewOperator iViewOperator = MetaDataUtil.getIViewOperatorAfterSetDT(dResult.getDbType());
            iViewOperator.createOrRep(viewManage.getViewName(), viewManage.getSql());
        } catch (Exception e) {
            throw new BaseException("SQL校验不通过，不支持导入！");
        }
        viewManage.setStatus(ViewManage.Generated);
        this.create(viewManage);
        ThreadMsgUtil.addMapMsg2("viewsSaved", String.format("<div style='margin-left:12px; margin-top:6px;'>%s [%s]</div>", viewManage.getViewName() + "", viewManage.getDsAlias() + ""));
    }
}
