package com.artfess.cqxy.projectApproval.manager.impl;

import cn.afterturn.easypoi.excel.ExcelExportUtil;
import cn.afterturn.easypoi.excel.ExcelImportUtil;
import cn.afterturn.easypoi.excel.entity.ExportParams;
import cn.afterturn.easypoi.excel.entity.ImportParams;
import com.artfess.base.context.BaseContext;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.query.PageBean;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.BeanUtils;
import com.artfess.cqxy.projectApproval.dao.DecisionBasisDao;
import com.artfess.cqxy.projectApproval.manager.DecisionAnnualPlanManager;
import com.artfess.cqxy.projectApproval.manager.DecisionBasisManager;
import com.artfess.cqxy.projectApproval.model.DecisionAnnualPlan;
import com.artfess.cqxy.projectApproval.model.DecisionBasis;
import com.artfess.cqxy.projectManagement.enums.ProjectStatusEnum;
import com.artfess.cqxy.projectManagement.manager.ProjectManagementManager;
import com.artfess.cqxy.projectManagement.model.ProjectManagement;
import com.artfess.cqxy.search.enums.FunctionEnum;
import com.artfess.cqxy.search.manager.GlobalRetrievalManager;
import com.artfess.cqxy.search.model.GlobalRetrieval;
import com.artfess.cqxy.universal.manager.AccessoryManager;
import com.artfess.cqxy.universal.model.Accessory;
import com.artfess.cqxy.utils.BizUtils;
import com.artfess.poi.util.ExcelUtil;
import com.artfess.sysConfig.persistence.manager.SysDictionaryManager;
import com.artfess.sysConfig.persistence.param.DictModel;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.Workbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStream;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * 决策依据表(BizDecisionBasis)表服务实现类
 *
 * @author 黎沐华
 * @since 2022-02-14 13:57:39
 */
@Service
public class DecisionBasisManagerImpl extends BaseManagerImpl<DecisionBasisDao, DecisionBasis> implements DecisionBasisManager {

    @Autowired
    private AccessoryManager accessoryManager;

    @Autowired
    private SysDictionaryManager sdm;

    @Autowired
    ProjectManagementManager pmm;

    @Autowired
    private BaseContext baseContext;

    @Autowired
    private GlobalRetrievalManager grm;

    @Autowired
    private DecisionAnnualPlanManager decisionAnnualPlanManager;

    @Override
    public boolean saveOrUpdate(DecisionBasis entity) {
        boolean save = StringUtils.isEmpty(entity.getId());
        boolean savedMain = super.saveOrUpdate(entity);
        // 处理附件信息
        List<Accessory> flag = entity.getAccessoryInfo();//处理空指针
        List<Accessory> accessoryList = null == flag ? new ArrayList<>() : flag;
        for (Accessory ele : accessoryList) {
            ele.setSourceId(entity.getId());
            ele.setProjectId(entity.getProjectId());
            ele.setDirectory(ProjectStatusEnum.one.getCode());
            ele.setGroup("DecisionBasis");
            ele.setNode(ProjectStatusEnum.one.getCode());
            ele.setCreateBy(baseContext.getCurrentUserId());
            ele.setCreateName(baseContext.getCurrentUserName());
            ele.setCreateTime(LocalDateTime.now());
        }
        // 先清空该ID下的所有附件信息再添加，否则当修改时删除附件就不知道删掉哪一个了，传已经删除的ID过来改动大
        accessoryManager.removeBySourceId(entity.getId());
        // 如果没有附件则返回true，不进入添加
        boolean saveAcc = accessoryList.size() == 0 || accessoryManager.saveAccess(accessoryList);
        // 同步开竣工时间
        ProjectManagement projectManagement = pmm.getById(entity.getProjectId());
        String decisionAnnualPlanId = entity.getDecisionAnnualPlanId();
        if(null != projectManagement && StringUtils.isNotBlank(decisionAnnualPlanId)){
            DecisionAnnualPlan decisionAnnualPlan = decisionAnnualPlanManager.getById(decisionAnnualPlanId);
            // 开工时间
            if(projectManagement.getCommencementTime() == null && decisionAnnualPlan != null){
                projectManagement.setCommencementTime(decisionAnnualPlan.getCommencementTime());
            }
            //完工时间
            if(null == projectManagement.getCompletionTime() && decisionAnnualPlan != null){
                projectManagement.setCompletionTime(decisionAnnualPlan.getCompletionTime());
            }
            pmm.updateById(projectManagement);
        }
        // 同步到检索表
        GlobalRetrieval globalRetrieval= grm.getByBizId(entity.getId());
        handleRetrieval(save || BeanUtils.isEmpty(globalRetrieval) ?new GlobalRetrieval():globalRetrieval,entity);

        return savedMain && saveAcc;
    }

    private void handleRetrieval(GlobalRetrieval globalRetrieval, DecisionBasis entity){
        ProjectManagement projectManagement = pmm.getById(entity.getProjectId());
        globalRetrieval.setProjectId(entity.getProjectId());
        globalRetrieval.setProjectName(projectManagement.getProjectName());
        globalRetrieval.setPersonCharge(projectManagement.getProjectManager());
        globalRetrieval.setArchivesType(1);
        globalRetrieval.setBizDataId(entity.getId());
        globalRetrieval.setFunctionCode(FunctionEnum.two.getCode());
        globalRetrieval.setFunctionName(FunctionEnum.two.getName());
        globalRetrieval.setBizTableName(FunctionEnum.two.getTableName());
        globalRetrieval.setDetailsRoteUrl(FunctionEnum.two.getTableRoteUrl());
        globalRetrieval.setTableRoteUrl(FunctionEnum.two.getTableRoteUrl());
        globalRetrieval.setTableApiUrl(FunctionEnum.two.getTableApiUrl());
        globalRetrieval.setDetailsApiUrl(FunctionEnum.two.getDetailsApiUrl());
        globalRetrieval.setFunctionPath(FunctionEnum.two.getFunctionPath());
        globalRetrieval.setSearchTitle(entity.getDecisionBasisPersion()+"_"+entity.getDecisionBasisDescribe()+"_"+entity.getRemarks());
        grm.saveOrUpdate(globalRetrieval);
    }

    @Override
    public boolean deleteByIds(List<String> ids) {
        // 同步删除检索表中的信息
        for(String ele:ids){
            grm.remove(1,ele);
        }
        return removeByIds(ids);
    }

    @Override
    public DecisionBasis getById(String id) {
        DecisionBasis result = baseMapper.getById(id);
        result.setAccessoryInfo(accessoryManager.getAccessoryBySourceId(id));
        return result;
    }

    @Override
    public PageList<DecisionBasis> queryAllByPage(QueryFilter<DecisionBasis> queryFilter) {
        BizUtils.handleFilter(queryFilter, "bdb", "pm");
        IPage<DecisionBasis> result = baseMapper.queryAllByPage(
                convert2IPage(queryFilter.getPageBean()),
                convert2Wrapper(queryFilter, currentModelClass()));
        // 附件信息处理
        List<DecisionBasis> records = result.getRecords();
        for (DecisionBasis ele : records) {
            ele.setAccessoryInfo(accessoryManager.getAccessoryBySourceId(ele.getId()));
        }
        return new PageList<>(result);

    }

    @Override
    public void importExcelData(MultipartFile file, String projectId) {
        Assert.notNull(projectId,"项目ID不能为空");
        // 获取字典
        List<DictModel> decisionType = sdm.queryDictListItemsByCode("xmlx-jcyj-jcwjlx");
        try(InputStream inputStream = file.getInputStream();) {
            // 获取数据，翻译字典
            List<DecisionBasis> data = ExcelImportUtil.importExcel(inputStream,DecisionBasis.class,new ImportParams());
            for(DecisionBasis ele:data){
                ele.setProjectId(projectId);
                ele.setDecisionBasisType(BizUtils.getDicCodeByValue(decisionType, ele.getDecisionBasisType()));
                // 保存数据 & 同步到检索表
                save(ele);
                handleRetrieval(new GlobalRetrieval(),ele);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void exportDatatoExcel(QueryFilter<DecisionBasis> queryFilter, HttpServletResponse response) throws IOException {
        // 字典
        List<DictModel> decisionType = sdm.queryDictListItemsByCode("xmlx-jcyj-jcwjlx");
        // 文件名
        String fileName = "决策依据-导出结果";

        // 获取数据
        BizUtils.handleFilter(queryFilter, "bdb", "pm");
        List<DecisionBasis> data = baseMapper.queryAllByPage(
                    convert2IPage(new PageBean(0,-1,false)),
                    convert2Wrapper(queryFilter, currentModelClass()))
                .getRecords();
        if(null==data||data.size()==0){
            throw new RuntimeException("没有要导出的的数据！");
        }

        // 翻译字典
        for(DecisionBasis ele : data){
            ele.setDecisionBasisType(BizUtils.getDicValueByCode(decisionType, ele.getDecisionBasisType()));
        }

        // EasyPoi 导出参数、样式、表格格式设置
        ExportParams exportParams = BizUtils.getExportParams(fileName);
        fileName += ".xlsx";

        // 导出下载excel文件
        Workbook workbook = ExcelExportUtil.exportExcel(exportParams, DecisionBasis.class, data);
        ExcelUtil.downloadExcel(workbook, fileName, response);
    }

    @Override
    public void updateProjectIdByProiectId(String oldProjectIds, String newProjectId) {
        UpdateWrapper<DecisionBasis> wrapper = new UpdateWrapper<DecisionBasis>()
                .set(StringUtils.isNotBlank(newProjectId), "PROJECT_ID_", newProjectId)
                .in("PROJECT_ID_", Arrays.asList(oldProjectIds.split(",")));
        this.update(wrapper);
    }

}

