package com.artfess.cqxy.designEstimate.manager.impl;

import cn.afterturn.easypoi.excel.ExcelExportUtil;
import cn.afterturn.easypoi.excel.ExcelImportUtil;
import cn.afterturn.easypoi.excel.entity.ExportParams;
import cn.afterturn.easypoi.excel.entity.ImportParams;
import com.artfess.base.context.BaseContext;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.query.PageBean;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.BeanUtils;
import com.artfess.cqxy.bidManagement.model.BiddingPriceLimit;
import com.artfess.cqxy.designEstimate.dao.EstimateDao;
import com.artfess.cqxy.designEstimate.manager.EstimateManager;
import com.artfess.cqxy.designEstimate.model.Estimate;
import com.artfess.cqxy.projectApproval.model.DecisionAnnualPlan;
import com.artfess.cqxy.projectApproval.model.DecisionBasis;
import com.artfess.cqxy.projectManagement.enums.ProjectStatusEnum;
import com.artfess.cqxy.projectManagement.manager.ProjectManagementManager;
import com.artfess.cqxy.projectManagement.model.ProjectManagement;
import com.artfess.cqxy.search.enums.FunctionEnum;
import com.artfess.cqxy.search.manager.GlobalRetrievalManager;
import com.artfess.cqxy.search.model.GlobalRetrieval;
import com.artfess.cqxy.universal.manager.AccessoryManager;
import com.artfess.cqxy.universal.model.Accessory;
import com.artfess.cqxy.utils.BizUtils;
import com.artfess.poi.util.ExcelUtil;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * 设计概算 - 概算表(BizEstimate)表服务实现类
 * @author 黎沐华
 * @since 2022-02-15 17:29:48
 */
@Service
public class EstimateManagerImpl extends BaseManagerImpl<EstimateDao, Estimate> implements EstimateManager {

    @Autowired
    private AccessoryManager accessoryManager;

    @Autowired
    private BaseContext baseContext;

    @Autowired
    private ProjectManagementManager pmm;

    @Autowired
    private GlobalRetrievalManager grm;

    @Override
    public boolean saveOrUpdate(Estimate entity) {
        boolean save = StringUtils.isEmpty(entity.getId());
        boolean savedMain = super.saveOrUpdate(entity);
        // 处理附件信息
        List<Accessory> flag = entity.getAccessoryInfo();//处理空指针
        List<Accessory> accessoryList = null == flag ? new ArrayList<>():flag;
        for (Accessory ele : accessoryList) {
            ele.setSourceId(entity.getId());
            ele.setProjectId(entity.getProjectId());
            ele.setDirectory(ProjectStatusEnum.six.getCode());
            ele.setGroup("Estimate");
            ele.setNode(ProjectStatusEnum.six.getCode());
            ele.setCreateBy(baseContext.getCurrentUserId());
            ele.setCreateName(baseContext.getCurrentUserName());
            ele.setCreateTime(LocalDateTime.now());
        }
        // 先清空该ID下的所有附件信息再添加
        accessoryManager.removeBySourceId(entity.getId());
        // 如果没有附件则返回true，不进入添加
        boolean saveAcc = accessoryList.size() == 0 || accessoryManager.saveAccess(accessoryList);
        //更新项目状态
        pmm.updateStatusById(entity.getProjectId(), Integer.valueOf(ProjectStatusEnum.six.getCode()));
        // 同步到检索表
        GlobalRetrieval globalRetrieval= grm.getByBizId(entity.getId());
        handleRetrieval(save || BeanUtils.isEmpty(globalRetrieval) ?new GlobalRetrieval():globalRetrieval,entity);

        return savedMain && saveAcc;
    }

    private void handleRetrieval(GlobalRetrieval globalRetrieval, Estimate entity){
        ProjectManagement projectManagement = pmm.getById(entity.getProjectId());
        globalRetrieval.setProjectId(entity.getProjectId());
        globalRetrieval.setProjectName(projectManagement.getProjectName());
        globalRetrieval.setPersonCharge(projectManagement.getProjectManager());
        globalRetrieval.setArchivesType(1);
        globalRetrieval.setBizDataId(entity.getId());
        globalRetrieval.setFunctionCode(FunctionEnum.ten.getCode());
        globalRetrieval.setFunctionName(FunctionEnum.ten.getName());
        globalRetrieval.setBizTableName(FunctionEnum.ten.getTableName());
        globalRetrieval.setDetailsRoteUrl(FunctionEnum.ten.getTableRoteUrl());
        globalRetrieval.setTableRoteUrl(FunctionEnum.ten.getTableRoteUrl());
        globalRetrieval.setTableApiUrl(FunctionEnum.ten.getTableApiUrl());
        globalRetrieval.setDetailsApiUrl(FunctionEnum.ten.getDetailsApiUrl());
        globalRetrieval.setFunctionPath(FunctionEnum.ten.getFunctionPath());
        globalRetrieval.setDocumentNumber(entity.getDocumentNumber());
        globalRetrieval.setName(entity.getName());
        globalRetrieval.setDocumentUnit(entity.getEstablishmentApprovalUnit());
        globalRetrieval.setSearchTitle(entity.getName()+"_"+entity.getDocumentNumber()+"_"+entity.getEstablishmentApprovalUnit()+"_"+
                entity.getProposedLocation()+"_"+entity.getConstructionScale()+"_"+entity.getEstablishmentRemarks());
        grm.saveOrUpdate(globalRetrieval);
    }

    @Override
    public boolean deleteByIds(List<String> ids) {
        // 同步删除检索表中的信息
        for(String ele:ids){
            grm.remove(1,ele);
        }
        return removeByIds(ids);
    }

    @Override
    public Estimate getById(String id) {
        Estimate result = baseMapper.getById(id);
        result.setAccessoryInfo(accessoryManager.getAccessoryBySourceId(id));
        return result;
    }

    @Override
    public PageList<Estimate> queryAllByPage(QueryFilter<Estimate> queryFilter) {
        BizUtils.handleFilter(queryFilter,"be" ,"pm");
        IPage<Estimate> result =
                baseMapper.queryAllByPage(
                        convert2IPage(queryFilter.getPageBean()),
                        convert2Wrapper(queryFilter, currentModelClass()));
        // 附件信息处理
        List<Estimate> records = result.getRecords();
        for (Estimate ele: records ) {
            ele.setAccessoryInfo(accessoryManager.getAccessoryBySourceId(ele.getId()));
        }

        return new PageList<>(result);
    }

    @Override
    public void importExcelData(MultipartFile file, String projectId) {
        Assert.notNull(projectId,"项目ID不能为空");
        try(InputStream inputStream = file.getInputStream();) {
            // 获取数据
            List<Estimate> data = ExcelImportUtil.importExcel(inputStream,Estimate.class,new ImportParams());
            for(Estimate ele:data){
                ele.setProjectId(projectId);
                // 保存数据 & 同步到检索表
                save(ele);
                handleRetrieval(new GlobalRetrieval(),ele);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void exportDatatoExcel(QueryFilter<Estimate> queryFilter, HttpServletResponse response) throws IOException {
        String fileName = "设计概算-概算-导出结果";

        // 获取数据
        BizUtils.handleFilter(queryFilter, "be", "pm");
        List<Estimate> data = baseMapper.queryAllByPage(
                        convert2IPage(new PageBean(0,-1,false)),
                        convert2Wrapper(queryFilter, currentModelClass())).
                getRecords();
        if(null==data||data.size()==0){
            throw new RuntimeException("没有要导出的的数据！");
        }

        // EasyPoi 导出参数、样式、表格格式设置
        ExportParams exportParams = BizUtils.getExportParams(fileName);
        fileName += ".xlsx";

        // 导出下载excel文件
        Workbook workbook = ExcelExportUtil.exportExcel(exportParams, Estimate.class, data);
        ExcelUtil.downloadExcel(workbook, fileName, response);
    }

    @Override
    public void updateProjectIdByProiectId(String oldProjectIds, String newProjectId) {
        UpdateWrapper<Estimate> wrapper = new UpdateWrapper<Estimate>()
                .set(StringUtils.isNotBlank(newProjectId), "PROJECT_ID_", newProjectId)
                .in("PROJECT_ID_", Arrays.asList(oldProjectIds.split(",")));
        this.update(wrapper);
    }

    @Override
    public BigDecimal getEstimate(String pid, String year) {
        return baseMapper.getEstimate(pid,year);
    }

}

