package com.artfess.form.controller;

import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.datasource.DatabaseContext;
import com.artfess.base.datasource.DatabaseSwitchResult;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.exception.BaseException;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageBean;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.Base64;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.FileUtil;
import com.artfess.base.util.HttpUtil;
import com.artfess.base.util.JsonUtil;
import com.artfess.base.util.SQLUtil;
import com.artfess.base.util.StringUtil;
import com.artfess.base.util.UniqueIdUtil;
import com.artfess.base.util.ZipUtil;
import com.artfess.base.util.time.DateFormatUtil;
import com.artfess.base.util.time.DateUtil;
import com.artfess.form.model.CustomDialog;
import com.artfess.form.model.CustomQuery;
import com.artfess.form.persistence.manager.CustomDialogManager;
import com.artfess.form.persistence.manager.CustomQueryManager;
import com.artfess.form.vo.CustomQueryControllerVo;
import com.artfess.table.meta.impl.BaseTableMeta;
import com.artfess.table.model.Column;
import com.artfess.table.model.Table;
import com.artfess.table.operator.IViewOperator;
import com.artfess.table.util.MetaDataUtil;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Optional;

/**
 * 自定义查询Controller
 *
 * @company 阿特菲斯信息技术有限公司
 * @author:lj
 * @date:2018年6月13日
 */
@RestController
@RequestMapping("/form/customQuery/v1")
@Api(tags = "关联数据")
@ApiGroup(group = {ApiGroupConsts.GROUP_FORM})
@SuppressWarnings({"unchecked", "rawtypes"})
public class CustomQueryController extends BaseController<CustomQueryManager, CustomQuery> {
    @Resource
    CustomDialogManager customDialogManager;
    @Resource
    DatabaseContext databaseContext;

    @RequestMapping(value = "list", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "自定义查询信息列表(分页条件查询)", httpMethod = "POST", notes = "自定义查询信息列表(分页条件查询)")
    public PageList listJson(@ApiParam(name = "queryFilter", value = "通用查询对象") @RequestBody Optional<QueryFilter> queryFilter) throws Exception {
        QueryFilter filter = queryFilter.orElse(QueryFilter.build().withPage(new PageBean(1, PageBean.WITHOUT_PAGE)));
        if (BeanUtils.isEmpty(filter.getPageBean())) {
            filter.withPage(new PageBean(1, PageBean.WITHOUT_PAGE));
        }
        return baseService.query(filter);
    }

    @RequestMapping(value = "getByAlias", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "通过别名获取自定义查询信息", httpMethod = "POST", notes = "通过别名获取自定义查询信息")
    public CustomQuery getByAlias(@ApiParam(name = "alias", value = "别名") @RequestBody String alias) {
        return baseService.getByAlias(alias);
    }

    @RequestMapping(value = "customQueryGet", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "获取自定义查询信息", httpMethod = "POST", notes = "获取自定义查询信息")
    public CustomQuery get(@ApiParam(name = "id", value = "主键") @RequestBody String id) throws Exception {
        return baseService.get(id);
    }

    @RequestMapping(value = "saveDialogByQuery", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "关联查询添加为对话框", httpMethod = "POST", notes = "关联查询添加为对话框")
    public CommonResult saveDialogByQuery(@ApiParam(name = "json", value = "自定义查询信息json对象") @RequestBody String json) throws Exception {
        CustomDialog customDialog = null;
        String resultMsg;
        if (!StringUtil.isEmpty(json)) {
            ObjectNode node = (ObjectNode) JsonUtil.toJsonNode(json);
            String displayfield = node.get("resultfield") + "";//因把关联查询数据信息添加为对话框，关联查询没有显示字段，所有对话框的显示字段也用返回字段
            String conditionfield = node.get("conditionfield") + "";
            String resultfield = node.get("resultfield") + "";
            String sortfield = node.get("sortfield") + "";
            node.remove("displayfield");
            node.remove("conditionfield");
            node.remove("resultfield");
            node.remove("sortfield");
            customDialog = JsonUtil.toBean(node, CustomDialog.class);
            customDialog.setDisplayfield(displayfield);
            customDialog.setConditionfield(conditionfield);
            customDialog.setResultfield(resultfield);
            customDialog.setSortfield(sortfield);
            customDialog.setStyle(new Short("0"));//默认列表形式
            customDialog.setSelectNum(1);//默认单选
        }
        CustomDialog entity = new CustomDialog();
        if (StringUtil.isNotEmpty(customDialog.getAlias())) {
            entity = customDialogManager.getByAlias(customDialog.getAlias());
        }
        if (BeanUtils.isEmpty(entity)) {
            if (customDialogManager.getByAlias(customDialog.getAlias()) != null) {
                return new CommonResult(false, customDialog.getAlias() + "，已存在", null);
            }
            customDialog.setId(UniqueIdUtil.getSuid());
            customDialog.setCreateTime(LocalDateTime.now());
            customDialog.setUpdateTime(DateUtil.getCurrentDate());
            customDialogManager.create(customDialog);
            resultMsg = "添加成功";
        } else {
            customDialog.setId(entity.getId());
            customDialog.setUpdateTime(LocalDateTime.now());
            customDialogManager.update(customDialog);
            resultMsg = "更新成功";
        }
        return new CommonResult(true, resultMsg, null);
    }

    @RequestMapping(value = "saveQueryByDialog", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "对话框添加为关联查询", httpMethod = "POST", notes = "对话框添加为关联查询")
    public CommonResult saveQueryByDialog(@ApiParam(name = "json", value = "自定义查询信息json对象") @RequestBody String json) throws Exception {
        CustomQuery customQuery = queryByDialog(json);
        String resultMsg = null;
        CustomQuery entity = new CustomQuery();
        if (StringUtil.isNotEmpty(customQuery.getAlias())) {
            entity = baseService.getByAlias(customQuery.getAlias());
        }
        if (BeanUtils.isEmpty(entity)) {
            if (baseService.getByAlias(customQuery.getAlias()) != null) {
                return new CommonResult(false, customQuery.getAlias() + "，已存在", null);
            }
            customQuery.setId(UniqueIdUtil.getSuid());
            baseService.create(customQuery);
            resultMsg = "添加成功";
        } else {
            customQuery.setId(entity.getId());
            baseService.update(customQuery);
            resultMsg = "更新成功";
        }
        return new CommonResult(true, resultMsg, null);
    }

    @RequestMapping(value = "save", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "保存自定义查询信息", httpMethod = "POST", notes = "保存自定义查询信息")
    public CommonResult save(@ApiParam(name = "json", value = "自定义查询信息json对象") @RequestBody String json) throws Exception {
        CustomQuery customQuery = getCustomQuery(json);
        String resultMsg = null;
        if (StringUtil.isEmpty(customQuery.getId())) {
            if (baseService.getByAlias(customQuery.getAlias()) != null) {
                return new CommonResult(false, "别名" + customQuery.getAlias() + "，已存在", null);
            }
            customQuery.setId(UniqueIdUtil.getSuid());
            baseService.create(customQuery);
            resultMsg = "添加成功";
        } else {
            baseService.update(customQuery);
            resultMsg = "更新成功";
        }
        return new CommonResult(true, resultMsg, null);
    }

    @RequestMapping(value = "removes", method = RequestMethod.DELETE, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "批量删除自定义查询信息", httpMethod = "DELETE", notes = "批量删除自定义查询信息")
    public CommonResult removes(@ApiParam(name = "id", value = "主键") @RequestParam String ids) throws Exception {
        String[] aryIds = StringUtil.getStringAryByStr(ids);
        baseService.removeByIds(aryIds);
        return new CommonResult(true, "删除自定义查询信息成功", null);
    }

    @RequestMapping(value = "getByDsObjectName", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "获取表或视图列表", httpMethod = "POST", notes = "获取表或视图列表")
    public ArrayNode getByDsObjectName(@ApiParam(name = "vo", value = "") @RequestBody CustomQueryControllerVo vo) throws Exception {
        if (SQLUtil.containsSqlInjection(vo.getObjName())) {
            throw new BaseException(ResponseErrorEnums.BAD_REQUEST);
        }
        return baseService.getTableOrViewByDsName(vo);
    }

    @RequestMapping(value = "getTable", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "获得表对象", httpMethod = "POST", notes = "获得表对象")
    public ObjectNode getTable(@ApiParam(name = "vo", value = "") @RequestBody CustomQueryControllerVo vo,
                               @ApiParam(name = "flag", value = "表主键类型长度是否允许小于19") @RequestParam Optional<Boolean> flag) throws Exception {

        ObjectNode result = JsonUtil.getMapper().createObjectNode();

        try (DatabaseSwitchResult dResult = databaseContext.setDataSource(vo.getDsalias())) {
            Table table = null;
            // 表
            if (vo.getIsTable().equals("1")) {
                BaseTableMeta baseTableMeta = MetaDataUtil.getBaseTableMetaAfterSetDT(dResult.getDbType());// 获取表操作元

                table = baseTableMeta.getTableByName(vo.getObjName());
            } else {
                IViewOperator iViewOperator = MetaDataUtil.getIViewOperatorAfterSetDT(dResult.getDbType());
                table = iViewOperator.getModelByViewName(vo.getObjName());
            }
            boolean isFlag = flag.orElse(true);
            if (BeanUtils.isNotEmpty(table.getColumnList())) {
                for (Column column : table.getColumnList()) {
                    if (column.getIsPk()) {
                        if (!isFlag && (Column.COLUMN_TYPE_INT.equals(column.getColumnType()) || Column.COLUMN_TYPE_NUMBER.equals(column.getColumnType())) && column.getIntLen() < 19) {
                            throw new BaseException("当前表主键类型为数字，且长度小于19，无法应用系统主键生成规则，添加失败！ ");
                        } else {
                            break;
                        }
                    }
                }
            }
            result.set("table", JsonUtil.toJsonNode(table));
            return result;
        } catch (Exception e) {
            throw new BaseException(e.getMessage(), e);
        }
    }

    @RequestMapping(value = "doQuery", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "", httpMethod = "POST", notes = "")
    public PageList doQuery(@ApiParam(name = "alias", value = "别名") @RequestParam Optional<String> alias,
                            @ApiParam(name = "queryData", value = "") @RequestBody Optional<String> queryData,
                            @ApiParam(name = "page", value = "") @RequestParam Optional<Integer> page) throws Exception {
        CustomQuery customQuery = baseService.getByAlias(alias.orElse(null));
        if (customQuery == null) {
            return null;
        }
        try (DatabaseSwitchResult dResult = databaseContext.setDataSource(customQuery.getDsalias())) {
            PageList data = baseService.getData(customQuery, queryData.orElse(null), dResult.getDbType(), page.orElse(null), customQuery.getPageSize());
            return data;
        } catch (Exception e) {
            throw new BaseException(e.getMessage(), e);
        }
    }

    @RequestMapping(value = "doQueryBase64", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "", httpMethod = "GET", notes = "")
    public PageList doQueryBase64(@ApiParam(name = "alias", value = "别名") @RequestParam String alias,
                                  @ApiParam(name = "queryData", value = "") @RequestParam String queryData,
                                  @ApiParam(name = "page", value = "") @RequestParam Integer page) throws Exception {
        CustomQuery customQuery = baseService.getByAlias(alias);
        if (customQuery == null) {
            return null;
        }

        if (StringUtil.isNotEmpty(queryData)) {
            queryData = Base64.getFromBase64(queryData);
            queryData = Base64.getFromBase64(queryData);
        }
        // 切换这次进程的数据源
        try (DatabaseSwitchResult dResult = databaseContext.setDataSource(customQuery.getDsalias())) {
            PageList data = baseService.getData(customQuery, queryData, dResult.getDbType(), page, customQuery.getPageSize());
            return data;
        } catch (Exception e) {
            throw new BaseException(e.getMessage(), e);
        }
    }

    @RequestMapping(value = "getAll", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "取得所有的表对象", httpMethod = "GET", notes = "取得所有的表对象")
    public List<CustomQuery> getAll() throws Exception {
        List<CustomQuery> customQuerys = baseService.list();
        return customQuerys;
    }

    @RequestMapping(value = "getQueryPage", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "", httpMethod = "GET", notes = "")
    public PageList getQueryPage(@ApiParam(name = "alias", value = "别名") @RequestParam String alias) throws Exception {

        PageList pageList = new PageList<>();
        CustomQuery customQuery = baseService.getByAlias(alias);
        if (customQuery == null) {
            return null;
        }
        try (DatabaseSwitchResult dResult = databaseContext.setDataSource(customQuery.getDsalias())) {
            customQuery.setNeedPage(0);
            pageList = (PageList) baseService.getData(customQuery, null, dResult.getDbType(), 1, 20);
        } catch (Exception e) {
            throw new BaseException(e.getMessage(), e);
        }
        return pageList;
    }

    private CustomQuery queryByDialog(String json) throws Exception {
        CustomQuery customQuery = null;
        ObjectNode node = (ObjectNode) JsonUtil.toJsonNode(json);
        String conditionfield = node.get("conditionfield").toString();
        String resultfield = node.get("displayfield").toString();//拿对话框的显示字段作为关联数据的返回字段
        String sortfield = node.get("sortfield").toString();
        String header = JsonUtil.getString(node, "header");
        node.remove("conditionfield");
        node.remove("resultfield");
        node.remove("sortfield");
        customQuery = JsonUtil.toBean(node, CustomQuery.class);
        customQuery.setConditionfield(conditionfield);
        customQuery.setResultfield(resultfield);
        customQuery.setSortfield(sortfield);
        if (StringUtil.isNotEmpty(header)) {
            customQuery.setHeader(header);
        }
        customQuery.setPageSize(10);//从对话框添加关联数据默认显示的数据为10个
        return customQuery;
    }

    private CustomQuery getCustomQuery(String json) throws Exception {
        CustomQuery customQuery = null;
        ObjectNode node = (ObjectNode) JsonUtil.toJsonNode(json);
        String conditionfield = node.get("conditionfield").toString();
        String resultfield = node.get("resultfield").toString();
        String sortfield = node.get("sortfield").toString();
        String header = JsonUtil.getString(node, "header");
        node.remove("conditionfield");
        node.remove("resultfield");
        node.remove("sortfield");
        customQuery = JsonUtil.toBean(node, CustomQuery.class);
        customQuery.setConditionfield(conditionfield);
        customQuery.setResultfield(resultfield);
        customQuery.setSortfield(sortfield);
        if (StringUtil.isNotEmpty(header)) {
            customQuery.setHeader(header);
        }
        return customQuery;
    }

    @RequestMapping(value = "export", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "导出数据", httpMethod = "GET", notes = "导出数据")
    public void export(HttpServletResponse response, HttpServletRequest request,
                       @ApiParam(name = "ids", value = "ids", required = true) @RequestParam String ids) throws Exception {
        String[] idList = ids.split(",");
        String fileName = "customQuery_" + DateFormatUtil.format(LocalDateTime.now(), "yyyy_MMdd_HHmm");
        String json = baseService.export(idList);
        HttpUtil.downLoadFile(request, response, json, "customQuery.json", fileName);
    }

    @RequestMapping(value = "import", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "导入自定义查询", httpMethod = "POST", notes = "导入自定义查询")
    public CommonResult<String> importCustom(MultipartHttpServletRequest request, HttpServletResponse response) throws Exception {
        MultipartFile file = request.getFile("file");
        String unZipFilePath = "";
        try {
            String rootRealPath = (FileUtil.getIoTmpdir() + "/attachFiles/unZip/").replace("/", File.separator);
            FileUtil.createFolder(rootRealPath, true);
            String name = file.getOriginalFilename();
            String fileDir = StringUtil.substringBeforeLast(name, ".");
            ZipUtil.unZipFile(file, rootRealPath); // 解压文件
            unZipFilePath = rootRealPath + File.separator + fileDir; // 解压后文件的真正路径
            baseService.importFile(unZipFilePath);
            return new CommonResult<>(true, "导入成功");
        } catch (Exception e) {
            return new CommonResult<>(false, "导入失败：" + e.getMessage());
        } finally {
            if (StringUtil.isNotEmpty(unZipFilePath)) {
                File zipFile = new File(unZipFilePath);
                if (zipFile.exists()) {
                    zipFile.delete();
                }
            }
        }
    }
}
