package com.artfess.application.job;


import com.artfess.base.exception.BaseException;
import com.artfess.base.feign.ApplicationFeignService;
import com.artfess.base.feign.dto.MessageConfigDto;
import com.artfess.base.feign.dto.MsgTemplateDto;
import com.artfess.base.jms.JmsActor;
import com.artfess.base.jms.Notice;
import com.artfess.base.jms.NoticeMessageType;
import com.artfess.base.util.AppUtil;
import com.artfess.base.util.StringUtil;
import com.artfess.job.model.BaseJob;
import com.artfess.uc.api.model.IUser;
import com.artfess.uc.api.service.IUserService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

@Slf4j
public class TestJob extends BaseJob {

    @Override
    public void executeJob(JobExecutionContext context) {
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        String id = jobDataMap.getString("id");
        String logId = (String) jobDataMap.get("logId");
        //短信模板id
        String msgId = jobDataMap.getString("msgId");
        log.info("定时计划测试正常com.artfess.application.job.TestJob，msgId:{}，id:{}，logId:{}", msgId, id, logId);
        if (StringUtil.isEmpty(msgId)) {
            log.error("定时计划执行失败:定时器：{}，失败原因：短信模板id为空！", this.getClass().getName());
            throw new BaseException("定时计划执行失败:，失败原因：短信模板id为空");
        }
        ApplicationFeignService applicationFeignService = AppUtil.getBean(ApplicationFeignService.class);
        MsgTemplateDto msgTemplate = applicationFeignService.getMsgTemplate(msgId);
        if (null == msgTemplate) {
            throw new BaseException("定时计划执行失败:，失败原因：短信模板未找到，消息模板ID：" + msgId);
        }
        processMsg(msgTemplate);
    }

    /**
     * 根据任务中模板id获取模版配置信息
     * @param msgTemplate
     */
    private void processMsg(MsgTemplateDto msgTemplate) {
        String key = msgTemplate.getKey();
        ApplicationFeignService applicationFeignService = AppUtil.getBean(ApplicationFeignService.class);
        List<MessageConfigDto> list = applicationFeignService.getMessageConfigList(key);
        Map<String, Object> vars = Maps.newHashMap();
        vars.put("name","2023年1月排查方案");
        list.forEach(messageConfig -> {
            String userIds = messageConfig.getUserId();
            String orgIds = messageConfig.getOrgId();
            List<JmsActor> reciever = Lists.newArrayList();
            if (StringUtils.isNotBlank(userIds) || StringUtils.isNotBlank(orgIds)) {
                //设置发送用户
                reciever.addAll(getUserList(userIds, orgIds));
            }
            sendMsg(msgTemplate, reciever, messageConfig.getNotifyType(), vars);
        });
    }

    /**
     * @param reciever   接收人
     * @param notifyType 支持的提醒类型
     * @param vars       消息动态参数
     */
    private void sendMsg(MsgTemplateDto msgTemplate, List<JmsActor> reciever, String notifyType, Map<String, Object> vars) {
        NoticeMessageType[] messageTypes = NoticeMessageType.parseNotifyType(notifyType);
        for (NoticeMessageType type : messageTypes) {
            Notice notice = new Notice();
            notice.setSubject(msgTemplate.getSubject());
            notice.setReceiver(reciever);
            notice.setUseTemplate(true);
            notice.setMessageTypes(new NoticeMessageType[]{type});
            notice.setTemplateType(msgTemplate.getTypeKey());
            notice.setTemplateKey(msgTemplate.getKey());
            notice.setVars(vars);
            if (type.isPlain()) {
                notice.setContent(msgTemplate.getPlain());
            } else {
                notice.setContent(msgTemplate.getHtml());
            }
            ApplicationFeignService applicationFeignService = AppUtil.getBean(ApplicationFeignService.class);
            applicationFeignService.sendNoticeToQueue(notice);
        }
    }

    /**
     * 根据模板配置的组织id和用户id信息 组织用户信息
     *
     * @param userIds
     * @param orgIds
     * @return
     */
    public List<JmsActor> getUserList(String userIds, String orgIds) {
        List<IUser> userList = new ArrayList<>();
        IUserService userService = AppUtil.getBean(IUserService.class);
        List<IUser> listByUser = userService.getUserByIds(userIds);
        if (!CollectionUtils.isEmpty(listByUser)) {
            userList.addAll(listByUser);
        }
        if(!StringUtil.isEmpty(orgIds)){
            List<IUser> listByOrg = userService.getUserListByGroups("org", orgIds);
            if (!CollectionUtils.isEmpty(listByOrg)) {
                userList.addAll(listByOrg);
            }
        }

        List<JmsActor> reciever = JmsActor.parseJmsActor(userList);
        return reciever;
    }
}
