package com.artfess.ljzc.land.manager.impl;

import com.alibaba.fastjson.JSONObject;
import com.artfess.base.enums.DelStatusEnum;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.query.PageBean;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.ljzc.business.model.BizAssetBusinessInfo;
import com.artfess.ljzc.land.dao.BizAssetLandInfoDao;
import com.artfess.ljzc.land.manager.BizAssetLandInfoManager;
import com.artfess.ljzc.land.manager.BizAssetLandUseManager;
import com.artfess.ljzc.land.manager.BizAssetLandWarrentsManager;
import com.artfess.ljzc.land.model.BizAssetLandInfo;
import com.artfess.ljzc.land.model.BizAssetLandUse;
import com.artfess.ljzc.land.model.BizAssetLandWarrents;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.google.common.collect.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.List;

/**
 * 土地资产的基础信息 服务实现类
 *
 * @author min.wu
 * @company 阿特菲斯信息技术有限公司
 * @since 2023-11-27
 */
@Service
public class BizAssetLandInfoManagerImpl extends BaseManagerImpl<BizAssetLandInfoDao, BizAssetLandInfo> implements BizAssetLandInfoManager {

    @Autowired
    private BizAssetLandUseManager assetLandUseManager;

    @Autowired
    private BizAssetLandWarrentsManager assetLandWarrentsManager;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean audit(List<String> ids, String status, String assetMsg) {
        Assert.notEmpty(ids, "请选择要审核的资产");
        Assert.hasText(status, "请选择审核状态");
        List<BizAssetLandInfo> list = listByIds(ids);
        Assert.notEmpty(list, "资产信息不存在");
        list.forEach(info -> {
            info.setAssetAudit(status);
            info.setAssetMsg(assetMsg);
        });
        boolean b = updateBatchById(list);
        return b;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean createInfo(BizAssetLandInfo t) {
        boolean save = this.save(t);
        if (save) {
            processInfo(t);
        }
        return save;
    }

    @Override
    public PageList<BizAssetLandInfo> findByPage(QueryFilter<BizAssetLandInfo> queryFilter) {
        queryFilter.addFilter("is_dele_", DelStatusEnum.N.getType(), QueryOP.EQUAL);
        PageBean pageBean = queryFilter.getPageBean();
        Class<BizAssetLandInfo> currentModelClass = currentModelClass();
        IPage<BizAssetLandInfo> result = baseMapper.findByPage(convert2IPage(pageBean), convert2Wrapper(queryFilter, currentModelClass));
        return new PageList<BizAssetLandInfo>(result);
    }

    @Override
    public JSONObject landStatistics(String fullId) {

        JSONObject jsonObject = this.baseMapper.landStatistics(fullId);
        Integer year = LocalDate.now().getYear();
        Integer lastYear = LocalDate.now().getYear() - 1;
        List<Integer> yearList = Lists.newArrayList();
        yearList.add(year);
        yearList.add(lastYear);
        List<JSONObject> yearUseArea = this.baseMapper.yearUseArea(fullId, yearList);
        BigDecimal thisYearUnusedArea = BigDecimal.ZERO;
        BigDecimal lastYearUnusedArea = BigDecimal.ZERO;
        for (JSONObject area : yearUseArea) {
            if (area.containsKey("year") && year.equals(area.getInteger("year"))) {
                thisYearUnusedArea = thisYearUnusedArea.add(area.getBigDecimal("unusedArea"));
            }
            if (area.containsKey("year") && lastYear.equals(area.getInteger("year"))) {
                lastYearUnusedArea = lastYearUnusedArea.add(area.getBigDecimal("unusedArea"));
            }
        }
        //增减率
        String decreaseRate = null;
        if(lastYearUnusedArea.doubleValue() == 0){
            decreaseRate = "100%";
        }else{
            decreaseRate = thisYearUnusedArea.subtract(lastYearUnusedArea).divide(lastYearUnusedArea, 2, BigDecimal.ROUND_HALF_UP).multiply(new BigDecimal("100")) + "%";
        }

        jsonObject.put(year.toString(), thisYearUnusedArea);
        jsonObject.put("decreaseRate", decreaseRate);
        List<JSONObject> assetStatusList = this.baseMapper.assetStatusList(fullId);
        JSONObject data = new JSONObject();
        data.put("areaInfo", jsonObject);
        data.put("bookValueInfo", assetStatusList);
        return data;
    }

    private void processInfo(BizAssetLandInfo t) {
        BizAssetLandWarrents assetLandWarrents = t.getAssetLandWarrents();
        assetLandWarrents.setLandId(t.getId());
        assetLandWarrentsManager.saveOrUpdate(assetLandWarrents);

        BizAssetLandUse assetLandUse = t.getAssetLandUse();
        assetLandUse.setLandId(t.getId());
        assetLandUseManager.saveOrUpdate(assetLandUse);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean updateInfo(BizAssetLandInfo t) {
        boolean b = this.updateById(t);
        if (b) {
            processInfo(t);
        }
        return b;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean removeInfo(List<String> ids) {
        boolean b = this.removeByIds(ids);
        if (b) {
            QueryWrapper<BizAssetLandWarrents> warrentsQueryWrapper = new QueryWrapper<>();
            warrentsQueryWrapper.in("land_id_", ids);
            assetLandWarrentsManager.remove(warrentsQueryWrapper);

            QueryWrapper<BizAssetLandUse> useQueryWrapper = new QueryWrapper<>();
            useQueryWrapper.in("land_id_", ids);
            assetLandUseManager.remove(useQueryWrapper);
        }
        return b;
    }

    @Override
    public BizAssetLandInfo findById(String id) {
        Assert.hasText(id, "请选择要查看的资产");
        BizAssetLandInfo businessInfo = this.get(id);
        QueryWrapper<BizAssetLandWarrents> warrentsQueryWrapper = new QueryWrapper<>();
        warrentsQueryWrapper.eq("land_id_", id);
        BizAssetLandWarrents warrents = assetLandWarrentsManager.getOne(warrentsQueryWrapper);
        businessInfo.setAssetLandWarrents(warrents);
        QueryWrapper<BizAssetLandUse> useQueryWrapper = new QueryWrapper<>();
        useQueryWrapper.eq("land_id_", id);
        BizAssetLandUse assetLandUse = assetLandUseManager.getOne(useQueryWrapper);
        businessInfo.setAssetLandUse(assetLandUse);
        return businessInfo;
    }
}
