package com.artfess.data.controller;


import com.alibaba.fastjson.JSON;
import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.annotation.PowerLogInfo;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.enums.LogType;
import com.artfess.base.enums.OperationType;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.enums.TeachingTypelEnum;
import com.artfess.base.exception.RequiredException;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.PinyinUtil;
import com.artfess.base.valid.AddGroup;
import com.artfess.base.valid.UpdateGroup;
import com.artfess.data.manager.BizTeachingResourcesTypeManager;
import com.artfess.data.model.BizTeachingResourcesType;
import com.artfess.poi.util.ExcelUtils;
import com.artfess.poi.util.FileDownloadUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.core.io.ClassPathResource;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Arrays;
import java.util.List;

/**
 * 训练知识分类 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author min.wu
 * @since 2024-09-02
 */
@Slf4j
@RestController
@Api(tags = "训练数据-训练知识分类")
@RequestMapping("/biz/teachingResources/type/")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class BizTeachingResourcesTypeController extends BaseController<BizTeachingResourcesTypeManager, BizTeachingResourcesType> {

    @Override
    @PostMapping("/")
    @ApiOperation("添加实体的接口")
    public CommonResult<String> create(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({AddGroup.class}) BizTeachingResourcesType t) {
        String id = baseService.insertTree(t);
        if(!StringUtils.isNotBlank(id)) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
        return new CommonResult<>();
    }

    @Override
    @PutMapping("/")
    @ApiOperation("更新实体")
    public CommonResult<String> updateById(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({UpdateGroup.class})  BizTeachingResourcesType t) {
        BizTeachingResourcesType byId = baseService.getById(t.getId());
        String id = baseService.updateTree(t, byId.getName());
        if(!StringUtils.isNotBlank(id)) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "更新实体失败");
        }
        return new CommonResult<>();
    }

    @PostMapping("/getTree")
    @ApiOperation("获取训练知识分类下拉树")
    @PowerLogInfo(logType = LogType.BIZ, operaionType = OperationType.QUERY, description = "获取训练知识分类下拉树")
    public CommonResult<String> getTree(@ApiParam(name = "model", value = "获取训练知识分类下拉树") @RequestBody BizTeachingResourcesType entity) {
        log.info("获取训练知识分类下拉树请求参数:{}", JSON.toJSONString(entity));
        List<BizTeachingResourcesType> list = baseService.getTree(entity);
        return CommonResult.success(list, null);
    }

    @ApiOperation(value = "导入")
    @PostMapping("/importExcel")
    public CommonResult<String> importExcel(@RequestParam("file") MultipartFile file) {
        try {
            ExcelUtils<BizTeachingResourcesType> excelUtil = new ExcelUtils<>(BizTeachingResourcesType.class);
            List<BizTeachingResourcesType> list = excelUtil.importExcel(null, file.getInputStream());
            list.forEach(data -> {
                data.setType(TeachingTypelEnum.getType(data.getType()));
                data.setCode(PinyinUtil.getPinYinHeadChar(data.getName()));
            });
            baseService.saveBatch(list);
            return new CommonResult<>();
        } catch (Exception e) {
            throw new IllegalArgumentException("导入失败," + e.getMessage());
        }
    }

    @ApiOperation(value = "导出")
    @PostMapping("/export")
    public void export(HttpServletResponse response, HttpServletRequest request,
                       @ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<BizTeachingResourcesType> queryFilter) throws Exception {
        PageList<BizTeachingResourcesType> page = baseService.query(queryFilter);
        ExcelUtils<BizTeachingResourcesType> util = new ExcelUtils<BizTeachingResourcesType>(BizTeachingResourcesType.class);
        util.exportExcel(response, request, page.getRows(), "训练知识分类");
    }

    /**
     * 下载导入模板
     *
     * @param response
     * @return
     */
    @ApiOperation(value = "下载导入模板")
    @GetMapping("/downModel")
    public void downTemplate(HttpServletResponse response) {
        try {
            ClassPathResource classPathResource = new ClassPathResource("model/训练知识数据/训练知识分类.xlsx");
            FileDownloadUtil.fileDownload(response, classPathResource.getInputStream(), "训练知识分类模板.xlsx");
        } catch (Exception e) {
            response.setCharacterEncoding("utf-8");
            throw new RequiredException("你所下载的资源不存在");
        }
    }

    @Override
    @DeleteMapping("/")
    @ApiOperation("根据id集合批量删除")
    public CommonResult<String> deleteByIds(@ApiParam(name="ids", value="实体集合") @RequestParam String...ids) {
        boolean result = baseService.deleteByIds(Arrays.asList(ids));
        if(!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "删除实体失败");
        }
        return new CommonResult<>();
    }

}
