package com.artfess.examine.controller;


import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.annotation.DataPermission;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.context.BaseContext;
import com.artfess.base.controller.BaseController;
import com.artfess.base.enums.PaperStatusEnum;
import com.artfess.base.enums.PaperTypeEnum;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.exception.BaseException;
import com.artfess.base.exception.RequiredException;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.FieldRelation;
import com.artfess.base.query.PageBean;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.util.time.DateUtil;
import com.artfess.base.valid.AddGroup;
import com.artfess.base.valid.UpdateGroup;
import com.artfess.examine.manager.ExamPaperBaseManager;
import com.artfess.examine.manager.ExamUserRecordManager;
import com.artfess.examine.model.ExamPaperBase;
import com.artfess.examine.model.ExamUserRecord;
import com.artfess.poi.util.ExcelUtils;
import com.artfess.poi.util.FileDownloadUtil;
import com.artfess.uc.api.impl.util.ContextUtil;
import com.artfess.uc.api.model.IUser;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ClassPathResource;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;

/**
 * 试卷基础信息 前端控制器
 *
 * @author min.wu
 * @company 阿特菲斯信息技术有限公司
 * @since 2022-10-19
 */
@Slf4j
@RestController
@Api(tags = "基础数据-试卷管理")
@RequestMapping("/exam/paper/base/")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class ExamPaperBaseController extends BaseController<ExamPaperBaseManager, ExamPaperBase> {

    @Resource
    BaseContext baseContext;

    @Autowired
    private ExamUserRecordManager userRecordManager;

    @Override
    @DataPermission
    @PostMapping(value = "/query", produces = {"application/json; charset=utf-8"})
    @ApiOperation("分页查询结果")
    public PageList<ExamPaperBase> query(@ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<ExamPaperBase> queryFilter) {
        IUser currentUser = ContextUtil.getCurrentUser();
        AtomicBoolean flag = new AtomicBoolean(false);

        AtomicReference<String> paperStatus = new AtomicReference<>();
        queryFilter.getQuerys().forEach(queryField -> {
            //如果是模拟训练则只能查看自己创建的试卷
            if ("type_".equals(queryField.getProperty()) && PaperTypeEnum.simulation.getType().equals(queryField.getValue())) {
                flag.set(true);
            }
            if ("paperStatus".equals(queryField.getProperty())) {
                paperStatus.set(queryField.getValue() + "");
            }
        });
        if (PaperStatusEnum.notRelease.getType().equals(paperStatus.get())) {
            queryFilter.addFilter("s.status_", PaperStatusEnum.notRelease.getType(), QueryOP.EQUAL);
        } else if (PaperStatusEnum.inTets.getType().equals(paperStatus.get())) {
            queryFilter.addFilter("s.plan_start_time_", LocalDateTime.now(), QueryOP.LESS, FieldRelation.AND);
            queryFilter.addFilter("s.plan_end_time_", LocalDateTime.now(), QueryOP.GREAT_EQUAL, FieldRelation.AND);
        } else if (PaperStatusEnum.finish.getType().equals(paperStatus.get())) {
            queryFilter.addFilter("s.plan_end_time_", LocalDateTime.now(), QueryOP.LESS, FieldRelation.AND);
        }
        if (!org.springframework.util.StringUtils.isEmpty(paperStatus.get())) {
            queryFilter.getQuerys().removeIf(s -> "paperStatus".equals(s.getProperty()));
        }
        if (flag.get() && !currentUser.isAdmin()) {
            queryFilter.addFilter("b.create_by_", currentUser.getUserId(), QueryOP.EQUAL);
        }
        queryFilter.addFilter("b.IS_DELE_", '0', QueryOP.EQUAL);
        List<String> currentAndChildOrgIds = baseContext.getCurrentAndChildOrgIds();
        queryFilter.addFilter("b.create_org_id_", currentAndChildOrgIds, QueryOP.IN);
        return baseService.findByPage(queryFilter);
    }

    @Override
    @PostMapping("/")
    @ApiOperation("添加实体的接口")
    public CommonResult<String> create(@ApiParam(name = "model", value = "实体信息") @RequestBody @Validated({AddGroup.class}) ExamPaperBase t) {
        t.setType(PaperTypeEnum.formal.getType());
        String id = baseService.createInfo(t);
        if (!StringUtils.isNotBlank(id)) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
        return new CommonResult<>();
    }

    @Override
    @PutMapping("/")
    @ApiOperation("更新实体（未发布之前显示该按钮）")
    public CommonResult<String> updateById(@ApiParam(name = "model", value = "实体信息") @RequestBody @Validated({UpdateGroup.class}) ExamPaperBase t) {
        t.setType(PaperTypeEnum.formal.getType());
        String id = baseService.updateInfo(t);
        if (!StringUtils.isNotBlank(id)) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "更新实体失败");
        }
        return new CommonResult<>();
    }

    @Override
    @GetMapping("/{id}")
    @ApiOperation("根据id查询实体")
    public ExamPaperBase getById(@ApiParam(name = "id", value = "实体id") @PathVariable String id) {
        return baseService.findById(id);
    }

    @GetMapping("viewPaper/{id}")
    @ApiOperation("预览试卷")
    public ExamPaperBase viewPaper(@ApiParam(name = "id", value = "实体id") @PathVariable String id) {
        return baseService.viewPaper(id);
    }

    @ApiOperation("发布考试")
    @GetMapping("startPaper/{id}")
    public CommonResult<String> startTask(@ApiParam(name = "id", value = "实体id") @PathVariable String id) {
        baseService.startPaper(id);
        return new CommonResult<>();
    }


    @PostMapping("/addUser")
    @ApiOperation("试卷添加考生（未发布不显示该功能、只提交新增的考生信息）")
    public CommonResult<String> addUser(@ApiParam(name = "model", value = "实体信息") @RequestBody ExamPaperBase t) {
        baseService.addUser(t);
        return new CommonResult<>();
    }

    @ApiOperation(value = "导入")
    @PostMapping("/importExcel")
    public CommonResult<String> importExcel(@RequestParam("file") MultipartFile file) {
        try {
            try {
                ExcelUtils<ExamPaperBase> excelUtil = new ExcelUtils<>(ExamPaperBase.class);
                List<ExamPaperBase> list = excelUtil.importExcel(null, file.getInputStream());
                boolean b = baseService.saveBatch(list);
                if (!b) {
                    return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
                }
            } catch (Exception e) {
                return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
            }
            return new CommonResult<>();
        } catch (Exception e) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
    }

    @ApiOperation(value = "导出")
    @PostMapping("/export")
    public void export(HttpServletResponse response, HttpServletRequest request,
                       @ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<ExamPaperBase> queryFilter) throws Exception {
        PageList<ExamPaperBase> page = baseService.query(queryFilter);
        ExcelUtils<ExamPaperBase> util = new ExcelUtils<ExamPaperBase>(ExamPaperBase.class);
        util.exportExcel(response, request, page.getRows(), "试卷信息");
    }

    /**
     * 下载导入模板
     *
     * @param response
     * @return
     */
    @ApiOperation(value = "下载导入模板")
    @GetMapping("/downModel")
    public void downTemplate(HttpServletResponse response) {
        try {
            ClassPathResource classPathResource = new ClassPathResource("model/导入模板.xlsx");
            FileDownloadUtil.fileDownload(response, classPathResource.getInputStream(), "导入模板.xlsx");
        } catch (Exception e) {
            response.setCharacterEncoding("utf-8");
            throw new RequiredException("你所下载的资源不存在");
        }
    }

    @Override
    @DeleteMapping("/")
    @ApiOperation("根据id集合批量删除")
    public CommonResult<String> deleteByIds(@ApiParam(name = "ids", value = "实体集合") @RequestParam String... ids) {
        QueryFilter queryFilter = QueryFilter.build();
        queryFilter.addFilter("b.id_", Arrays.asList(ids), QueryOP.IN);
        PageBean pageBean = new PageBean(PageBean.NO_PAGE, PageBean.WITHOUT_PAGE);
        queryFilter.setPageBean(pageBean);
        PageList<ExamPaperBase> byPage = baseService.findByPage(queryFilter);
        byPage.getRows().forEach(examPaperBase -> {
            boolean dateEquals = DateUtil.isDateLittle(LocalDateTime.now(), examPaperBase.getPlanEndTime());
            if (dateEquals && !PaperStatusEnum.notRelease.getType().equals(examPaperBase.getStatus())) {
                examPaperBase.setStatus(PaperStatusEnum.inTets.getType());
            } else if (!dateEquals && !PaperStatusEnum.notRelease.getType().equals(examPaperBase.getStatus())) {
                examPaperBase.setStatus(PaperStatusEnum.finish.getType());
            }
            if (!PaperStatusEnum.notRelease.getType().equals(examPaperBase.getStatus())) {
                throw new BaseException(examPaperBase.getName() + "已发布，不能进行删除！");
            }
        });
        boolean result = baseService.removeByIds(Arrays.asList(ids));
        if (!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "删除实体失败");
        }
        LambdaQueryWrapper<ExamUserRecord> lambdaQueryWrapper = new LambdaQueryWrapper<>();
        lambdaQueryWrapper.in(ExamUserRecord::getPaperId, Arrays.asList(ids));
        userRecordManager.remove(lambdaQueryWrapper);
        return new CommonResult<>();
    }

    @ApiOperation("获取今年该课目是否已经设置过最终成绩")
    @GetMapping("getEndStatus/{subjectId}")
    public CommonResult<String> getEndStatus(@ApiParam(name = "subjectId", value = "课目id") @PathVariable String subjectId) {
        Long count = baseService.getEndStatus(subjectId);
        if (count > 0) {
            return CommonResult.success(null, "今年该课目已设置过最终成绩");
        } else {
            return CommonResult.success(null, "今年该课目未设置过最终成绩");
        }
    }

}
