package com.artfess.examine.controller;


import com.alibaba.fastjson.JSON;
import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.exception.BaseException;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.valid.AddGroup;
import com.artfess.base.valid.UpdateGroup;
import com.artfess.examine.manager.ExamUserEvaluationManager;
import com.artfess.examine.manager.ExamYearAssessmentSubManager;
import com.artfess.examine.model.ExamSubjectInfo;
import com.artfess.examine.model.ExamYearAssessmentSub;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.util.Arrays;
import java.util.List;

/**
 * 年度考核课目 前端控制器
 *
 * @author min.wu
 * @company 阿特菲斯信息技术有限公司
 * @since 2022-11-25
 */
@Slf4j
@RestController
@Api(tags = "基础数据-考核课目")
@RequestMapping("/exam/year/assessmentSub/")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class ExamYearAssessmentSubController extends BaseController<ExamYearAssessmentSubManager, ExamYearAssessmentSub> {

    @Resource
    private ExamUserEvaluationManager userEvaluationManager;

    @Override
    @PostMapping("/")
    @ApiOperation("添加实体的接口")
    public CommonResult<String> create(@ApiParam(name = "model", value = "实体信息") @RequestBody @Validated({AddGroup.class}) ExamYearAssessmentSub t) {
        log.info("考核课目保存请求参数:{}", JSON.toJSONString(t));
        boolean result = baseService.createInfo(t);
        if (!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
        return new CommonResult<>();
    }


    @Override
    @PutMapping("/")
    @ApiOperation("更新实体")
    public CommonResult<String> updateById(@ApiParam(name = "model", value = "实体信息") @RequestBody @Validated({UpdateGroup.class}) ExamYearAssessmentSub t) {
        log.info("修改考核课目请求参数:{}", JSON.toJSONString(t));
        boolean result = baseService.updateInfo(t);
        if (!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "更新实体失败");
        }
        return new CommonResult<>();
    }

    @Override
    @GetMapping("/{id}")
    @ApiOperation("根据id查询实体")
    public ExamYearAssessmentSub getById(@ApiParam(name = "id", value = "实体id") @PathVariable String id) {
        return baseService.findById(id);
    }

    @Override
    @DeleteMapping("/")
    @ApiOperation("根据id集合批量删除")
    public CommonResult<String> deleteByIds(@ApiParam(name = "ids", value = "实体集合") @RequestParam String... ids) {
        QueryWrapper<ExamYearAssessmentSub> queryWrapper = new QueryWrapper<>();
        queryWrapper.in("id_", Arrays.asList(ids));
        List<ExamYearAssessmentSub> examYearAssessmentSubs = this.baseService.getBaseMapper().selectList(queryWrapper);
        examYearAssessmentSubs.forEach(sub -> {
            boolean result =userEvaluationManager.delPositionSubject(sub.getSubjectIds(), sub.getPositionId(), sub.getYear());
            if(!result) {
                throw new BaseException("删除实体失败");
            }
        });

        return new CommonResult<>();
    }

    @PostMapping("/deleteSub")
    @ApiOperation("根据id删除考核课目")
    public CommonResult<String> deleteSub(@ApiParam(name = "sub", value = "实体集合") @RequestBody ExamYearAssessmentSub sub) {
        boolean result = userEvaluationManager.delPositionSubject(sub.getSubjectIds(), sub.getPositionId(), sub.getYear());
        if (!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "删除实体失败");
        }
        return new CommonResult<>();
    }

    @GetMapping("/findByPositionId/{positionId}")
    @ApiOperation("根据岗位id获取当年考核课目")
    public List<ExamSubjectInfo> findByPositionId(@ApiParam(name = "positionId", value = "岗位id") @PathVariable String positionId) {
        return baseService.findByPositionId(positionId);
    }

    @PostMapping(value = "/findByPage", produces = {"application/json; charset=utf-8"})
    @ApiOperation("年度考核试卷选择课目分页接口")
    public PageList<ExamSubjectInfo> findByPage(@ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<ExamSubjectInfo> queryFilter) {
        return baseService.findByPage(queryFilter);
    }


}
