package com.artfess.data.controller;


import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.annotation.SyncAnnotation;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.exception.RequiredException;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.valid.UpdateGroup;
import com.artfess.data.manager.BizUserCheckManager;
import com.artfess.data.model.BizUserCheck;
import com.artfess.poi.util.ExcelUtils;
import com.artfess.poi.util.FileDownloadUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.ClassPathResource;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Arrays;
import java.util.List;

/**
 * 人员考勤数据 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author min.wu
 * @since 2024-09-02
 */
@Slf4j
@RestController
@Api(tags = "训练数据-人员考勤数据")
@RequestMapping("/biz/userCheck/")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class BizUserCheckController extends BaseController<BizUserCheckManager, BizUserCheck> {


    @Override
    @PutMapping("/")
    @ApiOperation("更新实体(修改考勤状态)")
    public CommonResult<String> updateById(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({UpdateGroup.class}) BizUserCheck t) {
        BizUserCheck userCheck = this.getById(t.getId());
        userCheck.setStatus("1");
        userCheck.setTrainFile(t.getTrainFile());
        boolean result = baseService.updateById(userCheck);
        if(!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "更新实体失败");
        }
        return new CommonResult<>();
    }

    @Override
    @PostMapping(value="/query", produces={"application/json; charset=utf-8" })
    @ApiOperation("分页查询结果")
    public PageList<BizUserCheck> query(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<BizUserCheck> queryFilter) {
        return baseService.query(queryFilter);
    }

    @ApiOperation(value = "导入")
    @PostMapping("/importExcel")
    public CommonResult<String> importExcel(@RequestParam("file") MultipartFile file, String planId) {
        try {
            ExcelUtils<BizUserCheck> excelUtil = new ExcelUtils<>(BizUserCheck.class);
            List<BizUserCheck> list = excelUtil.importExcel(null, file.getInputStream());
            baseService.importList(list, planId);
            return new CommonResult<>();
        } catch (Exception e) {
            throw new IllegalArgumentException("导入失败," + e.getMessage());
        }
    }

    @ApiOperation(value = "导出")
    @PostMapping("/export")
    public void export(HttpServletResponse response, HttpServletRequest request,
                       @ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<BizUserCheck> queryFilter) throws Exception {
        PageList<BizUserCheck> page = this.query(queryFilter);
        ExcelUtils<BizUserCheck> util = new ExcelUtils<BizUserCheck>(BizUserCheck.class);
        util.exportExcel(response, request, page.getRows(), "人员考勤数据");
    }

    /**
     * 下载导入模板
     *
     * @param response
     * @return
     */
    @ApiOperation(value = "下载导入模板")
    @GetMapping("/downModel")
    public void downTemplate(HttpServletResponse response) {
        try {
            ClassPathResource classPathResource = new ClassPathResource("model/训练活动数据/人员考勤数据.xlsx");
            FileDownloadUtil.fileDownload(response, classPathResource.getInputStream(), "人员考勤数据模板.xlsx");
        } catch (Exception e) {
            response.setCharacterEncoding("utf-8");
            throw new RequiredException("你所下载的资源不存在");
        }
    }

    @SyncAnnotation(httpMethod = "post")
    @PostMapping("/updateStatus")
    @ApiOperation("根据id集合修改考勤状态")
    public CommonResult<String> updateStatus(@ApiParam(name="ids", value="实体集合") @RequestParam String...ids) {
        List<BizUserCheck> userCheckList = baseService.listByIds(Arrays.asList(ids));
        userCheckList.forEach(check -> {
            check.setStatus("1");
        });
        boolean result = this.baseService.updateBatchById(userCheckList);
        if(!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "删除实体失败");
        }
        return new CommonResult<>();
    }
}
