package com.artfess.data.controller;


import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.exception.RequiredException;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.DictionaryUtils;
import com.artfess.base.valid.AddGroup;
import com.artfess.base.valid.UpdateGroup;
import com.artfess.data.manager.BizTrainingDataManager;
import com.artfess.data.model.BizTrainingData;
import com.artfess.poi.util.ExcelUtils;
import com.artfess.poi.util.FileDownloadUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.ClassPathResource;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.List;

/**
 * 训练消耗数据 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author min.wu
 * @since 2024-09-02
 */
@Slf4j
@RestController
@Api(tags = "训练数据-训练消耗明细")
@RequestMapping("/biz/training/data/")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class BizTrainingDataController extends BaseController<BizTrainingDataManager, BizTrainingData> {


    @Override
    @PostMapping("/")
    @ApiOperation("添加实体的接口")
    public CommonResult<String> create(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({AddGroup.class}) BizTrainingData t) {
        boolean result = baseService.saveInfo(t);
        if(!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
        return new CommonResult<>();
    }

    @Override
    @PutMapping("/")
    @ApiOperation("更新实体")
    public CommonResult<String> updateById(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({UpdateGroup.class}) BizTrainingData t) {
        boolean result = baseService.updateInfo(t);
        if(!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "更新实体失败");
        }
        return new CommonResult<>();
    }

    @ApiOperation(value = "导入")
    @PostMapping("/importExcel")
    public CommonResult<String> importExcel(@RequestParam("file") MultipartFile file) {
        try {
            ExcelUtils<BizTrainingData> excelUtil = new ExcelUtils<>(BizTrainingData.class);
            List<BizTrainingData> list = excelUtil.importExcel(null, file.getInputStream());
            baseService.saveList(list);
            return new CommonResult<>();
        } catch (Exception e) {
            throw new IllegalArgumentException("导入失败," + e.getMessage());
        }
    }

    @ApiOperation(value = "导出")
    @PostMapping("/export")
    public void export(HttpServletResponse response, HttpServletRequest request,
                       @ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<BizTrainingData> queryFilter) throws Exception {
        PageList<BizTrainingData> page = baseService.findByPage(queryFilter);
        page.getRows().forEach(data -> {
            if (null != DictionaryUtils.findByDictValue("crklx", data.getOperateType())) {
                String type = DictionaryUtils.findByDictValue("crklx", data.getOperateType()).getName();
                data.setOperateType(type);
            }
        });
        ExcelUtils<BizTrainingData> util = new ExcelUtils<BizTrainingData>(BizTrainingData.class);
        util.exportExcel(response, request, page.getRows(), "训练消耗指标使用明细");
    }

    /**
     * 下载导入模板
     *
     * @param response
     * @return
     */
    @ApiOperation(value = "下载导入模板")
    @GetMapping("/downModel")
    public void downTemplate(HttpServletResponse response) {
        try {
            ClassPathResource classPathResource = new ClassPathResource("model/训练保障数据/训练消耗指标使用明细数据.xlsx");
            FileDownloadUtil.fileDownload(response, classPathResource.getInputStream(), "训练消耗指标使用明细数据模板.xlsx");
        } catch (Exception e) {
            response.setCharacterEncoding("utf-8");
            throw new RequiredException("你所下载的资源不存在");
        }
    }

}
