package com.artfess.cqlt.controller;


import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.exception.RequiredException;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.Direction;
import com.artfess.base.query.FieldSort;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.StringUtil;
import com.artfess.base.valid.AddGroup;
import com.artfess.base.valid.UpdateGroup;
import com.artfess.cqlt.manager.QfOperationMacrDManager;
import com.artfess.cqlt.manager.QfOperationMacrMManager;
import com.artfess.cqlt.model.QfOperationKpiM;
import com.artfess.cqlt.model.QfOperationMacrD;
import com.artfess.cqlt.model.QfOperationMacrM;
import com.artfess.cqlt.vo.FaTargetRespVo;
import com.artfess.cqlt.vo.ReportReqVo;
import com.artfess.i18n.util.I18nUtil;
import com.artfess.poi.util.ExcelUtils;
import com.artfess.poi.util.FileDownloadUtil;
import com.artfess.uc.api.impl.util.ContextUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.core.io.ClassPathResource;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.Arrays;
import java.util.List;

/**
 * 运营--宏观数据填报主表 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author 黎沐华
 * @since 2023-02-21
 */
@Slf4j
@RestController
@Api(tags = "运营--宏观数据填报")
@RequestMapping("/qf/operation/macr/v1/")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class QfOperationMacrMController extends BaseController<QfOperationMacrMManager, QfOperationMacrM> {

    @Autowired
    private QfOperationMacrDManager detailManager;

    @Override
    @PostMapping(value="/query", produces={"application/json; charset=utf-8" })
    @ApiOperation("分页查询结果")
    public PageList<QfOperationMacrM> query(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<QfOperationMacrM> queryFilter) {
        List<FieldSort> sorter = queryFilter.getSorter();
        sorter.add(new FieldSort("fill_month_", Direction.ASC));
        queryFilter.setSorter(sorter);
        return baseService.query(queryFilter);
    }

    @Override
    @PostMapping("/")
    @ApiOperation("添加实体的接口")
    public CommonResult<String> create(@ApiParam(name = "model", value = "实体信息") @RequestBody @Validated({AddGroup.class}) QfOperationMacrM t) {
        t.setFillUser(ContextUtil.getCurrentUserId());
        t.setFillDate(LocalDate.now());
        boolean result = baseService.insertInfo(t);
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("option.fail", LocaleContextHolder.getLocale()), null);
        }

        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), t.getId());
    }

    @Override
    @PutMapping("/")
    @ApiOperation("更新实体")
    public CommonResult<String> updateById(@ApiParam(name = "model", value = "实体信息") @RequestBody @Validated({UpdateGroup.class}) QfOperationMacrM t) {
        boolean result = baseService.updateInfo(t);
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("option.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }

    @PutMapping("/updateStatus")
    @ApiOperation("生效")
    public CommonResult<String> updateStatus(@ApiParam(name = "model", value = "实体信息") @RequestBody QfOperationMacrM t) {
        boolean result = baseService.updateStatus(t);
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("option.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }


    @Override
    @DeleteMapping("/{id}")
    @ApiOperation("根据id删除")
    public CommonResult<String> deleteById(@ApiParam(name = "id", value = "实体id") @PathVariable String id) {
        boolean result = baseService.removeById(id);
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("delete.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }

    @Override
    @DeleteMapping("/")
    @ApiOperation("根据id集合批量删除")
    public CommonResult<String> deleteByIds(@ApiParam(name = "ids", value = "实体集合") @RequestParam String... ids) {
        boolean result = baseService.removeByIds(Arrays.asList(ids));
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("delete.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }

    /**
     * @param file
     * @param mainId
     * @return
     */
    @ApiOperation(value = "宏观数据导入")
    @PostMapping("/importExcel")
    public CommonResult<String> importExcel(@RequestParam("file") MultipartFile file, @RequestParam(value = "mainId", required = false) String mainId) {
        try {
            ExcelUtils<QfOperationMacrD> excelUtil = new ExcelUtils<>(QfOperationMacrD.class);
            List<QfOperationMacrD> list = excelUtil.importExcel(null, file.getInputStream());
            boolean result = baseService.importExcel(list, mainId);
            if (!result) {
                throw new IllegalArgumentException(I18nUtil.getMessage("import.fail", LocaleContextHolder.getLocale()));
            }
            return new CommonResult<>();
        } catch (Exception e) {
            throw new IllegalArgumentException(I18nUtil.getMessage("import.fail", LocaleContextHolder.getLocale()) + "," + e.getMessage());
        }
    }

    /**
     * 下载导入模板
     *
     * @param response
     * @return
     */
    @ApiOperation(value = "下载导入模板")
    @GetMapping("/downModel")
    public void downTemplate(HttpServletResponse response) {
        try {
            ClassPathResource classPathResource = new ClassPathResource("model/macrDetail.xlsx");
            FileDownloadUtil.fileDownload(response, classPathResource.getInputStream(), "宏观数据导入模板.xlsx");
        } catch (Exception e) {
            response.setCharacterEncoding("utf-8");
            throw new RequiredException(I18nUtil.getMessage("resources.noexist", LocaleContextHolder.getLocale()));
        }
    }

    @ApiOperation(value = "宏观数据导出")
    @PostMapping("/export")
    public void export(HttpServletRequest request, HttpServletResponse response, @ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<QfOperationMacrD> queryFilter) throws Exception {
        PageList<QfOperationMacrD> page = detailManager.query(queryFilter);
        ExcelUtils<QfOperationMacrD> util = new ExcelUtils<QfOperationMacrD>(QfOperationMacrD.class);
        page.getRows().forEach(detail -> {
            String spaceStr = StringUtil.addSpace(detail.getSubjectLevel());
            detail.setSubjectName(spaceStr + detail.getSubjectName());
            if(StringUtil.isNotEmpty(detail.getSubjectUnit()) && "%".equals(detail.getSubjectUnit())) {
                detail.setFillData(detail.getFillData().multiply(new BigDecimal(100)));
            }
        });
        util.exportExcel(response, request, page.getRows(), "宏观数据信息");
    }

    @PostMapping("/oneLevelData")
    @ApiOperation(value = "宏观数据一级界面统计接口", response = FaTargetRespVo.class)
    public CommonResult<String> oneLevelData(@ApiParam(name = "model", value = "请求参数") @RequestBody ReportReqVo t) {
        List<FaTargetRespVo> data = baseService.data(t);
        return CommonResult.success(data, null);
    }

    @PostMapping("/fromUnderData")
    @ApiOperation("宏观数据包含当前指标的年度、季度、月度报表)")
    public CommonResult<String> fromUnderData(@ApiParam(name = "model", value = "请求参数") @RequestBody ReportReqVo t) {
        List<FaTargetRespVo> data = baseService.fromUnderData(t);
        return CommonResult.success(data, null);
    }

}
