package com.artfess.portal.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.google.common.hash.Hashing;
import com.artfess.base.conf.SaaSConfig;
import com.artfess.base.constants.SystemConstants;
import com.artfess.base.context.BaseContext;
import com.artfess.base.controller.BaseController;
import com.artfess.base.handler.MultiTenantHandler;
import com.artfess.base.handler.MultiTenantIgnoreResult;
import com.artfess.base.jwt.JwtTokenHandler;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.*;
import com.artfess.base.util.AppUtil;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.StringUtil;
import com.artfess.base.util.UniqueIdUtil;
import com.artfess.base.util.time.TimeUtil;
import com.artfess.portal.model.ShorturlManage;
import com.artfess.portal.persistence.manager.ShorturlManageManager;
import com.artfess.uc.api.impl.util.ContextUtil;
import com.artfess.uc.api.model.IUser;
import com.artfess.uc.api.service.IUserService;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.nio.charset.StandardCharsets;
import java.time.LocalDateTime;

/**
 * 前端控制器
 *
 * @author chens
 * @company 阿特菲斯信息技术有限公司
 * @since 2020-12-01
 */
@RestController
@RequestMapping("/portal/shorturlManage/v1/")
public class ShorturlManageController extends BaseController<ShorturlManageManager, ShorturlManage> {

    @Resource
    JwtTokenHandler jwtTokenHandler;
    @Resource
    SaaSConfig saaSConfig;
    @Resource
    BaseContext baseContext;

    @GetMapping("/getLongUrlByShortUrl")
    @ApiOperation(value = "根据短连接获取短连接配置相关信息", httpMethod = "GET", notes = "根据短连接获取短连接配置相关信息")
    public CommonResult<ShorturlManage> getLongUrlByShortUrl(
            @ApiParam(name = "shortUrl", value = "用户账号") @RequestParam(required = true) String shortUrl)
            throws Exception {
        ShorturlManage shortUrlManage = null;
        if (saaSConfig.isEnable()) {
            //由于这里是未登录状态，也无法获取租户ID，只能通过短链接的租户ID来识别租户。
            try (MultiTenantIgnoreResult multiTenantIgnoreResult = MultiTenantHandler.setThreadLocalIgnore()) {
                QueryWrapper<ShorturlManage> wrapper = new QueryWrapper<>();
                wrapper.eq("SHORT_URL_", shortUrl);
                shortUrlManage = baseService.getOne(wrapper);
            }
        } else {
            shortUrlManage = baseService.getByShortUrl(shortUrl);
        }

        String errorMsg = "";
        if (BeanUtils.isEmpty(shortUrlManage)) {
            errorMsg = "2";
        } else if ("2".equals(shortUrlManage.getEnabled())) {
            errorMsg = "3";
        } else if (BeanUtils.isNotEmpty(shortUrlManage.getLoseTime()) && TimeUtil.getTimeMillis(shortUrlManage.getLoseTime()) < TimeUtil.getTimeMillis(LocalDateTime.now())) {
            errorMsg = "4";
        }
        if (StringUtil.isNotEmpty(errorMsg)) {
            return new CommonResult<>(false, errorMsg);
        }
        if (saaSConfig.isEnable()) {
            //设置当前的租户ID
            baseContext.setTempTenantId(shortUrlManage.getTenantId());
            IUserService userService = AppUtil.getBean(IUserService.class);
            IUser userByAccount = userService.getUserByAccount(SystemConstants.GUEST_ACCOUNT);
            shortUrlManage.setGuestToken(jwtTokenHandler.generateToken(userByAccount));
        } else {
            shortUrlManage.setGuestToken(jwtTokenHandler.generateToken(SystemConstants.GUEST_ACCOUNT));
        }

        return new CommonResult<>(true, "获取成功", shortUrlManage);
    }

    /**
     * 根据长连接获取短连接，长链接已存在记录，则直接返回已有配置。不存在则生成新的短链接并保存
     *
     * @param vo
     * @return
     * @throws Exception
     */
    @PostMapping("/getShortUrlByLong")
    @ApiOperation(value = "根据长连接获取短连接，长链接已存在记录，则直接返回已有配置。不存在则生成新的短链接并保存", httpMethod = "POST", notes = "更新短链接对象")
    public CommonResult<ShorturlManage> getShortUrlByLong(@ApiParam(name = "getShortUrlByLong", value = "短链接对象", required = true) @RequestBody ShorturlManage vo) throws Exception {
        CommonResult<ShorturlManage> rtn = null;
        try {
            String shourtUrl = "";
            if (saaSConfig.isEnable()) {
                //不同租户下，可能会生成多个相同的短链接
                shourtUrl = "surl/" + Hashing.murmur3_32().hashString(vo.getUrl() + ContextUtil.getCurrentUser().getTenantId(), StandardCharsets.UTF_8).toString();
            } else {
                shourtUrl = "surl/" + Hashing.murmur3_32().hashString(vo.getUrl(), StandardCharsets.UTF_8).toString();
            }

            QueryFilter<ShorturlManage> queryFilter = QueryFilter.build();
            queryFilter.withPage(new PageBean(1, -1));
            queryFilter.addFilter("short_url_", shourtUrl, QueryOP.RIGHT_LIKE);
            PageList<ShorturlManage> query = baseService.query(queryFilter);
            //如果生成的短链接已存在，并且长url和当前url一致。则直接返回已有配置
            if (BeanUtils.isNotEmpty(query) && BeanUtils.isNotEmpty(query.getRows())) {
                for (ShorturlManage shorturlManage : query.getRows()) {
                    if (vo.getUrl().equals(shorturlManage.getUrl())) {
                        return new CommonResult<ShorturlManage>(true, "获取短链接成功", shorturlManage);
                    }
                }
                //短链接在库里已存在，但是长连接不一样。则为其拼接后缀，保证短链接的唯一性
                shourtUrl += query.getRows().size();
            }
            vo.setShortUrl(shourtUrl);
            vo.setId(UniqueIdUtil.getSuid());
            baseService.save(vo);
            rtn = new CommonResult<ShorturlManage>(true, "根据长连接获取！", vo);
        } catch (Exception e) {
            e.printStackTrace();
            rtn = new CommonResult<ShorturlManage>(false, "根据长连接获取!", null);
        }
        return rtn;
    }

    /**
     * 更新链接配置对象
     *
     * @param vo
     * @return
     * @throws Exception
     */
    @PostMapping("/updateShorturlManage")
    @ApiOperation(value = "更新链接配置对象 ", httpMethod = "POST", notes = "更新链接配置对象")
    public CommonResult<ShorturlManage> updateShorturlManage(
            @ApiParam(name = "shorturlManage", value = "短链接对象", required = true) @RequestBody ShorturlManage vo)
            throws Exception {
        CommonResult<ShorturlManage> rtn = null;
        try {
            baseService.update(vo);
            rtn = new CommonResult<ShorturlManage>(true, "更新成功！", vo);
        } catch (Exception e) {
            e.printStackTrace();
            rtn = new CommonResult<ShorturlManage>(false, "更新失败!", null);
        }
        return rtn;
    }
}
