package com.artfess.manage.duty.manager.impl;


import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.RandomUtil;
import cn.hutool.json.JSONArray;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import cn.hutool.poi.excel.ExcelUtil;
import cn.hutool.poi.excel.ExcelWriter;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.QueryOP;
import com.artfess.base.util.StringUtil;
import com.artfess.manage.duty.dao.*;
import com.artfess.manage.duty.manager.CmgtDutyWorkarrangeMemberManager;
import com.artfess.manage.duty.manager.dto.CmgtDutyTeamMemberDto;
import com.artfess.manage.duty.manager.dto.CmgtDutyWorkarrangeDto;
import com.artfess.manage.duty.manager.mapper.CmgtDutyTeamMemberDtoMapper;
import com.artfess.manage.duty.manager.mapper.CmgtDutyWorkarrangeDtoMapper;
import com.artfess.manage.duty.model.*;
import com.artfess.manage.duty.manager.CmgtDutyWorkarrangeManager;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.query.PageBean;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.manage.duty.vo.ExcelArrangeVo;
import com.artfess.manage.duty.vo.WorkArrangeInfoVo;
import com.artfess.manage.duty.vo.WorkArrangeMemberVo;
import com.artfess.manage.duty.vo.WorkarrangeVo;
import com.artfess.manage.safty.model.CmgtSaftyTrainingProject;
import com.artfess.manage.utils.DateUtil;
import com.artfess.manage.utils.ExcelUtils;
import com.artfess.manage.utils.GeoUtils;
import com.artfess.uc.dao.OrgDao;
import com.artfess.uc.exception.BadRequestException;
import com.artfess.uc.model.Org;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 排班信息 服务实现类
 *
 * @author wujl
 * @company 阿特菲斯信息技术有限公司
 * @since 2022-07-28
 */
@Service
public class CmgtDutyWorkarrangeManagerImpl extends BaseManagerImpl<CmgtDutyWorkarrangeDao, CmgtDutyWorkarrange> implements CmgtDutyWorkarrangeManager {

    @Resource
    private CmgtDutyWorkarrangeDao cmgtDutyWorkarrangeDao;

    @Resource
    CmgtDutyWorkarrangeMemberManager memberManager;

    @Resource
    CmgtDutyWorkarrangeDtoMapper cmgtDutyWorkarrangeDtoMapper;

    @Resource
    private CmgtDutyWorkclassDao cmgtDutyWorkclassDao;

    @Resource
    private CmgtDutyTeamDao cmgtDutyTeamDao;

    @Resource
    private CmgtDutyTeamMemberDao cmgtDutyTeamMemberDao;

    @Resource
    private CmgtDutyTeamMemberDtoMapper cmgtDutyTeamMemberDtoMapper;

    @Resource
    private CmgtDutyJobDao cmgtDutyJobDao;

    @Resource
    private CmgtDutyTeamMemberGpsinfoDao cmgtDutyTeamMemberGpsinfoDao;


    @Resource
    private OrgDao orgDao;

    @Override
    public PageList<CmgtDutyWorkarrange> pageQuery(QueryFilter<CmgtDutyWorkarrange> queryFilter) {
        PageBean pageBean = queryFilter.getPageBean();
        IPage<CmgtDutyWorkarrange> result = baseMapper.queryPage(convert2IPage(pageBean), convert2Wrapper(queryFilter, currentModelClass()));
        return new PageList<CmgtDutyWorkarrange>(result);
    }


    /**
     * 根据月份获取部门每月排班信息
     *
     * @param date
     * @param orgId
     * @return
     */
    @Override
    public List<WorkArrangeInfoVo> findAllByMonth(String date, String orgId) {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        if (StringUtil.isEmpty(date)) {
            date = format.format(new Date());
        }
        String startDate = DateUtil.getMinMonthDate(date);
        String endDate = DateUtil.getMaxMonthDate(date);
        Map<String, Object> params = new HashMap<>();
        params.put("startDate", startDate);
        params.put("endDate", endDate);
        params.put("orgId", orgId);
        return cmgtDutyWorkarrangeDao.findAllByMonth(params);
    }


    /**
     * 按队员查询排班情况
     *
     * @param date
     * @param teamMemberId
     * @return
     */
    public List<CmgtDutyWorkarrangeDto> findTeamMemberArrangeing(String date, String teamMemberId) {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        if (StringUtil.isEmpty(date)) {
            date = format.format(new Date());
        }
        String startDate = DateUtil.getMinMonthDate(date);
        String endDate = DateUtil.getMaxMonthDate(date);
        return findTeamMemberArrangeing(startDate, endDate, teamMemberId);
    }

    /**
     * @param startDate
     * @param endDate
     * @param teamMemberId
     * @return
     */
    public List<CmgtDutyWorkarrangeDto> findTeamMemberArrangeing(String startDate, String endDate, String teamMemberId) {
        QueryWrapper<CmgtDutyWorkarrange> queryWrapper = new QueryWrapper();
        queryWrapper.between("arrange_date_", startDate, endDate);
        queryWrapper.like("teammember_id_", teamMemberId);
        return cmgtDutyWorkarrangeDao.selectList(queryWrapper).stream().map(r -> {
            CmgtDutyWorkarrangeDto t = cmgtDutyWorkarrangeDtoMapper.toDto(r);
            t.setCmgtDutyWorkclass(cmgtDutyWorkclassDao.selectById(r.getWorkclassId()));
            t.setCmgtDutyTeam(cmgtDutyTeamDao.selectById(r.getTeamId()));
            return t;
        }).collect(Collectors.toList());
    }

    /**
     * 查询排班
     *
     * @param arrangeDate
     * @param workclassId
     * @param orgId
     * @return
     */
    public List<CmgtDutyWorkarrange> findBy(LocalDate arrangeDate, String workclassId, String orgId) {
        QueryFilter<CmgtDutyWorkarrange> filter = QueryFilter.build();
        filter.addFilter("arrange_date_", arrangeDate, QueryOP.EQUAL);
        filter.addFilter("workclass_id_", workclassId, QueryOP.EQUAL);
        filter.addFilter("org_id_", orgId, QueryOP.EQUAL);
        return this.queryNoPage(filter);
    }

    @Transactional(readOnly = false)
    public void saveInfo(CmgtDutyWorkarrangeDto t, boolean isAdd) {
        CmgtDutyWorkarrange cmgtDutyWorkarrange = cmgtDutyWorkarrangeDtoMapper.toEntity(t);
        if (isAdd) {
            this.createInfo(cmgtDutyWorkarrange);
        } else {
            this.updateInfo(cmgtDutyWorkarrange);
        }
        if (t.getSelectDays() != null && t.getSelectDays().length > 0) {
            for (String day : t.getSelectDays()) {
                if (day.equals(cmgtDutyWorkarrange.getArrangeDate().format(DateTimeFormatter.ofPattern("yyyy-MM-dd")))) {
                    continue;
                }
                CmgtDutyWorkarrange workarrange = cmgtDutyWorkarrangeDtoMapper.toEntity(t);
                workarrange.setId(null);
                workarrange.setCreateBy(null);
                workarrange.setCreateTime(null);
                workarrange.setCreateOrgId(null);
                workarrange.setArrangeDate(LocalDate.parse(day, DateTimeFormatter.ofPattern("yyyy-MM-dd")));
                //覆盖删除
                if (t.getIsCover() != null && t.getIsCover()) {
                    List<CmgtDutyWorkarrange> workarrangeList = findBy(workarrange.getArrangeDate(), workarrange.getWorkclassId(), workarrange.getOrgId());
                    for (CmgtDutyWorkarrange w : workarrangeList) {
                        deleteInfo(w);
                    }
                }
                this.createInfo(workarrange);
            }
        }
    }

    @Override
    @Transactional(readOnly = false)
    public boolean createInfo(CmgtDutyWorkarrange t) {
        if (t.getTeammemberId() == null) {
            throw new BadRequestException("排班未指定队员");
        }
        if (findBy(t.getArrangeDate(), t.getWorkclassId(), t.getOrgId()).size() > 0) {
            throw new BadRequestException(t.getArrangeDate() + ",该班已存在，不能重复排班");
        }
        t.setTeammemberNames(StringUtil.join(Arrays.asList(t.getTeammemberId().split(",")).stream().map(mid -> {
            return cmgtDutyTeamMemberDao.selectById(mid).getName();
        }).collect(Collectors.toList()), ","));

        boolean res = this.save(t);
        this.saveMember(t.getTeammemberId(), t);
        return res;
    }

    @Override
    @Transactional(readOnly = false)
    public String updateInfo(CmgtDutyWorkarrange t) {
        if (t.getTeammemberId() == null) {
            throw new BadRequestException("排班未指定队员");
        }
        t.setTeammemberNames(StringUtil.join(Arrays.asList(t.getTeammemberId().split(",")).stream().map(mid -> {
            return cmgtDutyTeamMemberDao.selectById(mid).getName();
        }).collect(Collectors.toList()), ","));

        this.updateById(t);
        this.delMember(t);//删除
        this.saveMember(t.getTeammemberId(), t);//重新保存
        return t.getId();
    }

    @Override
    @Transactional(readOnly = false)
    public void deleteInfo(CmgtDutyWorkarrange t) {
        this.baseMapper.deleteById(t.getId());
        this.delMember(t);//删除队员信息
    }

    /**
     * 一键排班
     *
     * @param params
     * @return
     */
    @Override
    public boolean fastWorkarrange(Map<String, Object> params) throws ParseException {
        if (params == null) {
            return false;
        }
        if (params.get("orgId") == null || StringUtil.isEmpty(params.get("orgId").toString())) {
            return false;
        }
        if (params.get("date") == null || StringUtil.isEmpty(params.get("date").toString())) {
            return false;
        }
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM");
        String date = params.get("date").toString();//获取排班日期
        String orgId = params.get("orgId").toString();//获取排班部门ID
//        String startDate = DateUtil.getMinMonthDate(date);//排班当月第一天
//        String endDate = DateUtil.getMaxMonthDate(date);//排班当月第最后一天
        //获取上一个月yyyy-MM信息
        String lastMonth = DateUtil.getPrevMonthDate(date, 1);
        System.out.println("---------getMaxMonthDate-------" + lastMonth);
        Map<String, Object> para = new HashMap<>();
        para.put("startDate", DateUtil.getMinMonthDate(lastMonth + "-01"));
        para.put("endDate", DateUtil.getMaxMonthDate(lastMonth + "-01"));
        para.put("orgId", orgId);
        System.out.println("------------para----" + para);
        //查询上一个月排班信息
        List<WorkArrangeInfoVo> lastArrangeList = cmgtDutyWorkarrangeDao.findAllByMonth(para);

        System.out.println(lastArrangeList.size() + "------------lastArrangeList----" + para);
        //获取当前排班月的总天数
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(format.parse(date));
        int day = calendar.getActualMaximum(Calendar.DAY_OF_MONTH);//获取当月总共天数
        for (int i = 1; i <= day; i++) {
            //String arrangeDate = date+"-"+i;
            int y = i;
            lastArrangeList.stream().forEach(a -> {
                String d = Integer.toString(y);
                if (y < 10) {
                    d = "0" + d;
                }
                if (a.getArrangeDate().substring(8, 10).equals(d)) {
                    CmgtDutyWorkarrange arr = new CmgtDutyWorkarrange();
                    LocalDate localDate = LocalDate.parse(date + "-" + d);
                    arr.setArrangeDate(localDate);//时间
                    arr.setDataType(a.getDataType());//数据类型，1：小组，0人员
                    arr.setTeamId(a.getTid());//小组ID
                    arr.setTeammemberId(a.getMids());//队员ID
                    arr.setWorkclassId(a.getClassId());//班种ID
                    arr.setIsDele("0");
                    arr.setStatus("1");
                    arr.setOrgId(orgId);

                    cmgtDutyWorkarrangeDao.insert(arr);
                    this.saveMember(a.getMids(), arr);
                }
            });
        }
        return true;
    }

    public void saveMember(String mids, CmgtDutyWorkarrange arr) {
        String[] memberIds = mids.split(",");
        List<CmgtDutyWorkarrangeMember> list = new ArrayList<>();
        for (String mid : memberIds) {
            CmgtDutyWorkarrangeMember member = new CmgtDutyWorkarrangeMember();
            member.setMemberId(mid);
            member.setWorkarrangeId(arr.getId());
            list.add(member);
        }
        memberManager.saveBatch(list);
    }

    void delMember(CmgtDutyWorkarrange arr) {
        QueryFilter<CmgtDutyWorkarrangeMember> filter = QueryFilter.build();
        filter.addFilter("WORK_ARRANGE_", arr.getId(), QueryOP.EQUAL);
        List<CmgtDutyWorkarrangeMember> list = memberManager.queryNoPage(filter);
        List<String> ids = list.stream().map(l -> l.getId()).collect(Collectors.toList());
        memberManager.removeByIds(ids);
    }

    /**
     * 按小组导出
     *
     * @param request
     * @param response
     * @param date
     */
    @Override
    public void exportExcel(HttpServletRequest request, HttpServletResponse response, String date) {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        if (StringUtil.isEmpty(date)) {
            date = format.format(new Date());
        }
        String startDate = DateUtil.getMinMonthDate(date);//当月第一天
        String endDate = DateUtil.getMaxMonthDate(date);//当月第最后一天
        Calendar calendar = Calendar.getInstance();
        try {
            calendar.setTime(format.parse(date));
        } catch (ParseException e) {
            e.printStackTrace();
        }
        int day = calendar.getActualMaximum(Calendar.DAY_OF_MONTH);//获取当月总共天数
        List reslist = new ArrayList<>();
        for (int i = 1; i <= day; i++) {
            Set teams = new HashSet();
            Map<String, Object> params = new HashMap<>();
            String arrangeDate = startDate.substring(0, 8) + i;
            params.put("arrangeDate", arrangeDate);
            //一天排班信息
            List<WorkarrangeVo> vo = cmgtDutyWorkarrangeDao.findArrangeInfoByDay(params);
            //包装一天排班信息
            vo.forEach(v -> {
                StringBuilder dy = new StringBuilder();//队员
                StringBuilder dz = new StringBuilder();//队长
                Set<String> fzr = new HashSet<>(); //内勤组负责人
                Set<String> fzr2 = new HashSet<>(); //督察保障组负责人
                Set<String> headman = new HashSet<>(); //组长
                v.getDetails().forEach(m -> {
                    if (m.getTeamname().equals("内勤组") && m.getPost().equals("1")) { //负责人
                        fzr.add(m.getMembername() + "(" + m.getPhone() + ") ");
                    }
                    if (m.getTeamname().equals("督察保障组") && m.getPost().equals("1")) { //负责人
                        fzr2.add(m.getMembername() + "(" + m.getPhone() + ") ");
                    }
                    if (m.getPost().equals("2")) {//队长
                        dz.append(m.getMembername() + "(队长)" + m.getPhone() + " ");
                    }
                    if (m.getPost().equals("3")) {//中队长
                        dz.append(m.getMembername() + "(中队长)" + m.getPhone() + " ");
                    }
                    if (m.getPost().equals("4")) {//队员
                        dy.append(m.getMembername() + m.getPhone() + " ");
                    }
                    headman.add(m.getHeadman());
                });

                ExcelArrangeVo evo = new ExcelArrangeVo(arrangeDate, headman.toString(), fzr.toString(), fzr2.toString(),
                        v.getClassName() + "(" + v.getStartDate() + "-" + v.getEndDate() + ")", dz.toString(), dy.toString());
                reslist.add(evo);
            });
        }
        ExcelWriter writer = ExcelUtil.getWriter();
        writer.addHeaderAlias("date", "日期");
        writer.addHeaderAlias("headman", "组长");
        writer.addHeaderAlias("fzr", "内勤组");
        writer.addHeaderAlias("fzr2", "督察保障组");
        writer.addHeaderAlias("classname", "班次");
        writer.addHeaderAlias("dz", "队长");
        writer.addHeaderAlias("dy", "队员");
        writer.merge(6, startDate + "至" + endDate + "排班表");
        writer.write(reslist, false);
        //自适应列宽（必须放在writer.write(list, true)写入后）
        ExcelUtils.setSizeColumn(writer.getSheet(), 6);
        //浏览器导出
        ExcelUtils.downloadExcel(response, startDate + "至" + endDate + "排班表", writer);
    }

    /**
     * 按人员导出
     *
     * @param request
     * @param response
     * @param date
     * @throws ParseException
     */
    @Override
    public void exportMemberArrangeExcel(HttpServletRequest request, HttpServletResponse response, String date) throws ParseException {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        if (StringUtil.isEmpty(date)) {
            date = format.format(new Date());
        }
        String startDate = DateUtil.getMinMonthDate(date);//当月第一天
        String endDate = DateUtil.getMaxMonthDate(date);//当月第最后一天
        Calendar calendar = Calendar.getInstance();
        try {
            calendar.setTime(format.parse(date));
        } catch (ParseException e) {
            e.printStackTrace();
        }
        int day = calendar.getActualMaximum(Calendar.DAY_OF_MONTH);//获取当月总共天数
        String arrangeDate = startDate.substring(0, 8);//截取年月：yyyy-MM-
        ExcelWriter writer = ExcelUtil.getWriter();
        List<Object> days = new ArrayList<>();//每天名称
        List<Object> weeks = new ArrayList<>();//每天的星期名称
        days.add("序号");
        days.add("姓名 日期");
        weeks.add("");
        weeks.add("");
        writer.merge(day + 1, arrangeDate + "交巡组排班表");
        writer.passCurrentRow();
        for (int i = 1; i <= day; i++) {
            days.add(i);
            weeks.add(dayOfWeek(format.parse(arrangeDate + i)));
        }
        writer.writeRow(days);
        writer.writeRow(weeks);
        //查询数据
        Map<String, Object> params = new HashMap<>();
        params.put("startDate", startDate);
        params.put("endDate", endDate);
        List<WorkArrangeMemberVo> mlist = cmgtDutyWorkarrangeDao.findMemberArrange(params);
        int rowNum = 1;
        for (WorkArrangeMemberVo m : mlist) {
            String[] col = new String[day + 2];
            col[0] = rowNum + "";
            col[1] = m.getName();
            m.getDetails().forEach(d -> {
                col[d.getDate() + 1] = "全勤";
            });
            writer.writeRow(Arrays.asList(col));
            ++rowNum;
        }
        // writer.set(arrangeDate+"交巡组排班表");
        //合并序号、日期姓名单元格
        writer.merge(1, 2, 0, 0, "序号", false);
        writer.merge(1, 2, 1, 1, "姓名 日期", false);
        // ExcelUtils.setSizeColumn(writer.getSheet(),6);
        //浏览器导出
        ExcelUtils.downloadExcel(response, startDate + "至" + endDate + "排班表", writer);
    }

    public String dayOfWeek(Date date) {
        int day = cn.hutool.core.date.DateUtil.dayOfWeek(date);
        switch (day) {
            case 1:
                return "日";
            case 2:
                return "一";
            case 3:
                return "二";
            case 4:
                return "三";
            case 5:
                return "四";
            case 6:
                return "五";
            case 7:
                return "六";
            default:
                return "";
        }
    }

    public List<CmgtDutyWorkarrangeDto> findCmgtDutyWorkarrangeDtosBy(String arrangeDate) {
        QueryWrapper<CmgtDutyWorkarrange> queryWrapper = new QueryWrapper();
        queryWrapper.eq("arrange_date_", arrangeDate);

        return cmgtDutyWorkarrangeDtoMapper.toDto(cmgtDutyWorkarrangeDao.selectList(queryWrapper)).stream().map(t -> {

            if (t.getTeamId() != null) {
                t.setCmgtDutyTeam(cmgtDutyTeamDao.selectById(t.getTeamId()));
            } else {
                t.setCmgtDutyTeam(new CmgtDutyTeam());
            }
            if (t.getOrgId() != null) {
                t.setOrg(orgDao.selectById(t.getOrgId()));
            } else {
                t.setOrg(new Org());
            }
            if (t.getWorkclassId() != null) {
                t.setCmgtDutyWorkclass(cmgtDutyWorkclassDao.selectById(t.getWorkclassId()));
            } else {
                t.setCmgtDutyWorkclass(new CmgtDutyWorkclass());
            }
            t.setTeamMembers(Arrays.asList(t.getTeammemberId().split(",")).stream().map(mid -> {
                CmgtDutyTeamMemberDto memberDto = cmgtDutyTeamMemberDtoMapper.toDto(cmgtDutyTeamMemberDao.selectById(mid));
                memberDto.setOrg(orgDao.selectById(memberDto.getOrgId()));
                if (memberDto.getJob() != null) {
                    memberDto.setCmgtDutyJob(cmgtDutyJobDao.selectById(memberDto.getJob()));
                }
                return memberDto;
            }).collect(Collectors.toList()));
            return t;
        }).collect(Collectors.toList());
    }

    /**
     * 查询指定排班日期和时间的排班信息
     *
     * @param arrangeDate
     * @param dutyTime
     * @return
     */
    public List<CmgtDutyWorkarrangeDto> findCmgtDutyWorkarrangeDtosBy(String arrangeDate, LocalTime dutyTime) {
        List<CmgtDutyWorkarrangeDto> r = this.findCmgtDutyWorkarrangeDtosBy(arrangeDate);
        return r.stream().filter(e -> {
            //大于开始时间且小于截止时间
            if (dutyTime.compareTo(e.getCmgtDutyWorkclass().getScheduleStart()) == 1
                    && dutyTime.compareTo(e.getCmgtDutyWorkclass().getDuringTime()) == -1) {
                return true;
            }
            //跨日班次,则只需大于开始时间
            if (e.getCmgtDutyWorkclass().getScheduleStart().compareTo(e.getCmgtDutyWorkclass().getDuringTime()) == 1
                    && dutyTime.compareTo(e.getCmgtDutyWorkclass().getScheduleStart()) == 1) {
                return true;
            }
            return false;
        }).collect(Collectors.toList());
    }


    /**
     * 查询当前执勤人员
     *
     * @return
     */
    public JSONArray findCurrentStaffOnDuty(List<String> dutyJobIds) {
        DateTimeFormatter dtf = DateTimeFormatter.ofPattern("HH:mm");
        JSONArray r = JSONUtil.createArray();
        List<String> mIds = new ArrayList<String>();
        findCmgtDutyWorkarrangeDtosBy(DateUtil.getNowTime(), LocalTime.now()).forEach(t -> {
            for (CmgtDutyTeamMemberDto member : t.getTeamMembers()) {
                //过滤岗位条件
                if (dutyJobIds != null && !dutyJobIds.contains(member.getJob())) {
                    continue;
                }
                if (mIds.contains(member.getId())) {
                    continue;
                }

                r.add(JSONUtil.createObj().putOpt("dutyDate", t.getArrangeDate())
                        .putOpt("id", member.getId())
                        .putOpt("job", member.getJob())
                        .putOpt("jobName", (member.getCmgtDutyJob() != null ? member.getCmgtDutyJob().getName() : ""))
                        .putOpt("dutyStart", (t.getCmgtDutyWorkclass().getScheduleStart() != null ? t.getCmgtDutyWorkclass().getScheduleStart().format(dtf) : ""))
                        .putOpt("dutyEnd", (t.getCmgtDutyWorkclass().getDuringTime() != null ? t.getCmgtDutyWorkclass().getDuringTime().format(dtf) : ""))
                        .putOpt("name", member.getName())
                        .putOpt("sex", member.getSex())
                        .putOpt("post", member.getPost())
                        .putOpt("phone", member.getPhone())
                        .putOpt("org", member.getOrg().getName())
                        .putOpt("team", t.getCmgtDutyTeam().getName())
                        .putOpt("status", member.getStatus())
                        .putOpt("statusName", TeamMemberStatusEnum.findByCode(member.getStatus()))
                        .putOpt("location", member.getLocation())
                        .putOpt("receiveTime", member.getReceiveTime())
                );
                mIds.add(member.getId());
            }
        });
        return r;
    }


    public JSONArray findHistoryDutyStaff(List<String> dutyJobIds, List<LocalDateTime> rqs, JSONObject params) {

        JSONArray r = JSONUtil.createArray();
        List<String> mIds = new ArrayList<String>();
        DateTimeFormatter dtf = DateTimeFormatter.ofPattern("HH:mm");
        LocalDateTime arrangeDate = rqs.get(1);
        findCmgtDutyWorkarrangeDtosBy(arrangeDate.format(DateTimeFormatter.ofPattern("yyyy-MM-dd")), arrangeDate.toLocalTime()).forEach(t -> {
            for (CmgtDutyTeamMemberDto member : t.getTeamMembers()) {
                //过滤岗位条件
                if (dutyJobIds != null && !dutyJobIds.contains(member.getJob())) {
                    continue;
                }
                if (mIds.contains(member.getId())) {
                    continue;
                }
                if (!params.isNull("member") && !params.get("member").equals(member.getId())) {
                    continue;
                }

                List<CmgtDutyTeamMemberGpsinfo> gps = cmgtDutyTeamMemberGpsinfoDao.selectList(
                        new QueryWrapper<CmgtDutyTeamMemberGpsinfo>().eq("team_member_id_", member.getId()).between("receive_time_", rqs.get(0), rqs.get(1)));
                if (gps.size() > 0) {
                    r.add(JSONUtil.createObj().putOpt("dutyDate", t.getArrangeDate())
                            .putOpt("id", member.getId())
                            .putOpt("job", member.getJob())
                            .putOpt("jobName", (member.getCmgtDutyJob() != null ? member.getCmgtDutyJob().getName() : ""))
                            .putOpt("dutyStart", (t.getCmgtDutyWorkclass().getScheduleStart() != null ? t.getCmgtDutyWorkclass().getScheduleStart().format(dtf) : ""))
                            .putOpt("dutyEnd", (t.getCmgtDutyWorkclass().getDuringTime() != null ? t.getCmgtDutyWorkclass().getDuringTime().format(dtf) : ""))
                            .putOpt("name", member.getName())
                            .putOpt("sex", member.getSex())
                            .putOpt("post", member.getPost())
                            .putOpt("phone", member.getPhone())
                            .putOpt("org", member.getOrg().getName())
                            .putOpt("team", t.getCmgtDutyTeam().getName())
                            .putOpt("status", member.getStatus())
                            .putOpt("statusName", TeamMemberStatusEnum.findByCode(member.getStatus()))
                            .putOpt("location", gps.get(0).getLongitude() + "," + gps.get(0).getLatitude())
                            .putOpt("receiveTime", member.getReceiveTime())
                            .putOpt("track", gps.stream().map(g -> {
                                return new Float[]{Float.valueOf(g.getLongitude()), Float.valueOf(g.getLatitude())};
                            }).collect(Collectors.toList()))
                    );
                    mIds.add(member.getId());
                }
            }
        });
        return r;
    }


    static String[] MEMBER_STATUS = {"NORMAL", "ONLINE", "CROSSING", "ABSENT"};

    @Transactional
    public void createSumilatedTeamMemberLoationData() {
        List<CmgtDutyWorkarrangeDto> l = findCmgtDutyWorkarrangeDtosBy(DateUtil.getNowTime(), LocalTime.now());

        CmgtDutyWorkarrangeDto t = l.get(RandomUtil.randomInt(0, l.size() - 1));

        List<CmgtDutyTeamMember> members = cmgtDutyTeamMemberDtoMapper.toEntity(t.getTeamMembers());
        for (CmgtDutyTeamMemberDto memberDto : t.getTeamMembers()) {
            CmgtDutyTeamMember member = cmgtDutyTeamMemberDtoMapper.toEntity(memberDto);

            CmgtDutyTeamMemberGpsinfo gpsinfo = new CmgtDutyTeamMemberGpsinfo();
            gpsinfo.setTeamMemberId(member.getId());
            gpsinfo.setReceiveTime(LocalDateTime.now());
            gpsinfo.setMemo("随机生成模拟数据");
            if (member.getLocation() == null) {
                gpsinfo.setLongitude(RandomUtil.randomBigDecimal(BigDecimal.valueOf(106.537288), BigDecimal.valueOf(106.55633)) + "");
                gpsinfo.setLatitude(RandomUtil.randomBigDecimal(BigDecimal.valueOf(29.606626), BigDecimal.valueOf(29.615619)) + "");
            } else {
                BigDecimal lg = new BigDecimal(member.getLocation().split(",")[0]);
                BigDecimal la = new BigDecimal(member.getLocation().split(",")[1]);
                lg = lg.add(RandomUtil.randomBigDecimal(BigDecimal.valueOf(-0.0009), BigDecimal.valueOf(0.0009)));
                la = la.add(RandomUtil.randomBigDecimal(BigDecimal.valueOf(-0.0009), BigDecimal.valueOf(0.0009)));

                if (lg.doubleValue() > 106.55633 || lg.doubleValue() < 106.537288 || la.doubleValue() > 29.615619 || la.doubleValue() < 29.606626) {
                    gpsinfo.setLongitude(RandomUtil.randomBigDecimal(BigDecimal.valueOf(106.537288), BigDecimal.valueOf(106.55633)) + "");
                    gpsinfo.setLatitude(RandomUtil.randomBigDecimal(BigDecimal.valueOf(29.606626), BigDecimal.valueOf(29.615619)) + "");
                } else {
                    gpsinfo.setLongitude(lg + "");
                    gpsinfo.setLatitude(la + "");
                }
            }
            cmgtDutyTeamMemberGpsinfoDao.insert(gpsinfo);
            member.setLocation(gpsinfo.getLongitude() + "," + gpsinfo.getLatitude());
            member.setReceiveTime(gpsinfo.getReceiveTime());
            if (memberDto.getCmgtDutyJob() != null
                    && !GeoUtils.pointInPolygon(member.getLocation(), memberDto.getCmgtDutyJob().getArea())) {
                member.setStatus(TeamMemberStatusEnum.CROSSING.name());
            } else {
                member.setStatus(MEMBER_STATUS[RandomUtil.randomInt(0, 1)]);
            }
            //System.out.println("================createSumilatedTeamMemberLoationData===============1111111111===============" + GeoUtils.pointInPolygon(member.getLocation(), memberDto.getCmgtDutyJob().getArea()));
            cmgtDutyTeamMemberDao.updateById(member);
        }
    }
}
