package com.artfess.form.controller;

import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.Direction;
import com.artfess.base.query.FieldSort;
import com.artfess.base.query.PageBean;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.JsonUtil;
import com.artfess.base.util.StringUtil;
import com.artfess.bo.instance.BoDataHandler;
import com.artfess.bo.model.BoData;
import com.artfess.bo.model.BoDef;
import com.artfess.bo.model.BoEnt;
import com.artfess.bo.persistence.manager.BoAttributeManager;
import com.artfess.bo.persistence.manager.BoDefManager;
import com.artfess.bo.persistence.manager.BoEntManager;
import com.artfess.form.model.FormMeta;
import com.artfess.form.persistence.manager.FormMetaManager;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * bo定义控制器
 *
 * @author heyifan
 * @company 阿特菲斯信息技术有限公司
 * @email heyf@jee-soft.cn
 * @date 2018年4月18日
 */
@RestController
@RequestMapping("/bo/def/v1/")
@Api(tags = "业务对象定义")
@ApiGroup(group = {ApiGroupConsts.GROUP_FORM})
public class BoDefController extends BaseController<BoDefManager, BoDef> {
    @Resource
    BoEntManager boEntManager;
    @Resource
    BoDataHandler boDataHandler;
    @Resource
    FormMetaManager bpmFormDefManager;
    @Resource
    BoAttributeManager boAttributeManager;

    @RequestMapping(value = "list", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "获取bo定义列表（带分页信息）", httpMethod = "POST", notes = "获取bo定义列表")
    public PageList<BoDef> listJson(@ApiParam(name = "queryFilter", value = "通用查询对象") @RequestBody QueryFilter<BoDef> queryFilter) throws Exception {
        if (BeanUtils.isEmpty(queryFilter.getSorter())) {
            queryFilter.getSorter().add(new FieldSort("createTime", Direction.DESC));
        }
        return super.query(queryFilter);
    }

    @RequestMapping(value = "detail", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "获取bo定义详情", httpMethod = "GET", notes = "获取bo定义详情")
    public BoDef detail(@ApiParam(name = "alias", value = "bo定义别名", required = true) @RequestParam String alias) throws Exception {
        return baseService.getPureByAlias(alias);
    }

    @RequestMapping(value = "getJson", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "获取bo定义详情", httpMethod = "GET", notes = "获取bo定义详情")
    public ObjectNode getJson(@ApiParam(name = "id", value = "id", required = true) @RequestParam String id) throws Exception {
        return baseService.getBoDefDetails(id);
    }

    @RequestMapping(value = "save", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "添加bo定义", httpMethod = "POST", notes = "添加bo定义")
    public CommonResult<String> save(@ApiParam(name = "json", value = "bo定义的json数据", required = true) @RequestBody String json) throws Exception {
        baseService.save(json);
        return new CommonResult<String>("添加成功");
    }

    @RequestMapping(value = "remove", method = RequestMethod.DELETE, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "删除bo定义", httpMethod = "DELETE", notes = "删除bo定义")
    public CommonResult<Integer> remove(@ApiParam(name = "alias", value = "bo定义别名", required = true) @RequestParam String alias) throws Exception {
        int removeItems = baseService.removeByAlias(alias);
        if (removeItems > 0) {
            return new CommonResult<Integer>(true, "删除成功", removeItems);
        } else {
            return new CommonResult<Integer>(false, "删除失败", removeItems);
        }
    }

    @RequestMapping(value = "removes", method = RequestMethod.DELETE, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "批量删除bo定义", httpMethod = "DELETE", notes = "批量删除bo定义")
    public CommonResult<String> batchRemove(@ApiParam(name = "ids", value = "bo主键集合", required = true) @RequestParam String... ids) throws Exception {
        List<String> errList = removeBoDefDetails(ids);
        if (errList.size() == 0) {
            return new CommonResult<String>(true, "删除成功");
        }
        return new CommonResult<String>(false, "删除失败" + errList.size() + "个," + StringUtil.join(errList, ","));
    }

    @RequestMapping(value = "getBoJson", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "根据bo别名返回bo对应的JSON数据结构", httpMethod = "GET", notes = "根据bo定义返回bo对应的JSON数据结构")
    public JsonNode getBoJson(@ApiParam(name = "alias", value = "bo定义别名", required = true) @RequestParam String alias) throws Exception {
        BoData boData = boDataHandler.getByBoDefCode(alias);
        return JsonUtil.toJsonNode(boData.toString());
    }

    @SuppressWarnings("rawtypes")
    @RequestMapping(value = "deploy", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "发布", httpMethod = "GET", notes = "发布")
    public CommonResult deploy(@ApiParam(name = "id", value = "id", required = true) @RequestParam String id) throws Exception {
        BoDef boDef = baseService.getByDefId(id);
        boDef.setDeployed(true);
        baseService.update(boDef);
        return new CommonResult(boDef.getDescription() + "发布成功");
    }

    @SuppressWarnings("rawtypes")
    @RequestMapping(value = "setStatus", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "修改状态", httpMethod = "GET", notes = "修改状态")
    public CommonResult setStatus(@ApiParam(name = "id", value = "id", required = true) @RequestParam String id,
                                  @ApiParam(name = "status", value = "状态") @RequestParam String status) throws Exception {
        if (status == null || "".equals(status)) {
            status = "normal";
        }
        BoDef boDef = baseService.getByDefId(id);
        boDef.setStatus(status);
        baseService.update(boDef);
        return new CommonResult(boDef.getDescription() + "更改状态成功");
    }

    @RequestMapping(value = "getBOTree", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "获取BO树形数据", httpMethod = "POST", notes = "获取BO树形数据")
    public ObjectNode getBOTree(@ApiParam(name = "ids", value = "ids", required = true) @RequestBody String ids) throws Exception {
        return baseService.getBoTreeData(ids);
    }

    @RequestMapping(value = "getAllBos", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "获取所有bo实体列表", httpMethod = "GET", notes = "获取所有bo实体列表")
    public List<BoEnt> getAllBos() throws Exception {
        QueryFilter<BoEnt> queryFilter = QueryFilter.<BoEnt>build().withPage(new PageBean(1, PageBean.WITHOUT_PAGE));
        queryFilter.addFilter("is_create_table_", 1, QueryOP.EQUAL);
        PageList<BoEnt> pageList = boEntManager.query(queryFilter);
        return pageList.getRows();
    }

    @RequestMapping(value = "getBoEntById", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "获取所有bo实体列表", httpMethod = "GET", notes = "获取所有bo实体列表")
    public BoEnt getBoEntById(@ApiParam(name = "id", value = "id", required = true) @RequestParam String id) throws Exception {
        return boEntManager.getById(id);
    }

    @RequestMapping(value = "getSupportDb", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "获取所有bo实体列表", httpMethod = "GET", notes = "获取所有bo实体列表")
    public boolean getSupportDb(@ApiParam(name = "alias", value = "bo别名", required = true) @RequestParam String alias) throws Exception {
        return baseService.getPureByAlias(alias).isSupportDb();
    }

    @RequestMapping(value = "getObject", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "获取bodef", httpMethod = "GET", notes = "获取bodef")
    public BoDef getObject(@ApiParam(name = "id", value = "bodef主键", required = false) @RequestParam(required = false) String id, @ApiParam(name = "key", value = "bodef key", required = false) @RequestParam(required = false) String key) throws Exception {
        if (StringUtil.isNotEmpty(id)) {
            BoDef boDef = baseService.get(id);
            return boDef;
        }
        if (StringUtil.isNotEmpty(key)) {
            BoDef boDef = baseService.getByAlias(key);
            return boDef;
        }
        return null;
    }

    @RequestMapping(value = "updateCategory", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "修改bodef分类", httpMethod = "GET", notes = "修改bodef分类")
    public CommonResult<String> updateCategory(@ApiParam(name = "ids", value = "ids", required = true) @RequestParam(required = true) String[] ids, @ApiParam(name = "categoryId", value = "categoryId", required = true) @RequestParam(required = true) String categoryId, @ApiParam(value = "categoryName", name = "categoryName", required = true) @RequestParam(required = true) String categoryName) throws IOException {
        for (int i = 0; i < ids.length; i++) {
            BoDef boDef = baseService.get(ids[i]);
            boDef.setCategoryId(categoryId);
            boDef.setCategoryName(categoryName);
            baseService.updateCategory(boDef);
        }
        return new CommonResult<>(true, "设置分类成功");
    }

    @RequestMapping(value = "removeAttr", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "修改boAttribute分类", httpMethod = "POST", notes = "修改bodef分类")
    public CommonResult<String> removeAttr(@ApiParam(name = "alias", value = "alias", required = true) @RequestParam(required = true) String alias,
                                           @ApiParam(name = "json", value = "json", required = true) @RequestBody(required = true) String json) throws Exception {
        boAttributeManager.updateAttrStatus(json, alias);
        return new CommonResult<>(true, "字段删除成功");
    }

    @RequestMapping(value = "createTableForm", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "创建表单", httpMethod = "POST", notes = "修改bodef分类")
    public CommonResult<String> createTableForm(@ApiParam(name = "json", value = "json", required = true) @RequestBody(required = true) String json) throws Exception {
        CommonResult<String> vresult = validateBoAttr(json);
        if (!vresult.getState()) {
            return vresult;
        }
        String defId = baseService.saveFormData(json);
        return new CommonResult<>(true, defId);
    }

    @SuppressWarnings("rawtypes")
    @RequestMapping(value = "getBindData", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "获取绑定数据", httpMethod = "GET", notes = "修改bodef分类")
    public String getBindData(@ApiParam(name = "id", value = "id", required = true) @RequestParam(required = true) String id, @ApiParam(name = "alias", value = "alias", required = true) @RequestParam(required = true) String alias) throws IOException {
        Map result = baseService.getBindData(id, alias);
        String json = JsonUtil.toJson(result);
        return json;
    }

    @RequestMapping(value = "deleteAttr", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "删除attr数据", httpMethod = "POST", notes = "删除attr数据")
    public CommonResult<String> deleteAttr(@ApiParam(name = "id", value = "id", required = true) @RequestParam(required = true) String id) throws IOException {
        boAttributeManager.remove(id);
        return new CommonResult<>(true, "字段删除成功");
    }

    @RequestMapping(value = "recovery", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "还原删除数据", httpMethod = "POST", notes = "还原删除数据")
    public CommonResult<String> recovery(@ApiParam(name = "alias", value = "alias", required = true) @RequestParam(required = true) String alias,
                                         @ApiParam(name = "json", value = "json", required = true) @RequestBody String json) throws Exception {
        boAttributeManager.recovery(json, alias);
        return new CommonResult<>(true, "字段还原成功");
    }

    @RequestMapping(value = "getHideAttr", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "获取隐藏字段", httpMethod = "GET", notes = "获取隐藏字段")
    public List getHideAttr(@ApiParam(name = "tableName", value = "表名") @RequestParam String tableName) throws Exception {
        return baseService.getHideAttr(tableName);
    }

    @RequestMapping(value = "getSubEntInfo", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "获取BO建模中是否存在子表、孙表", httpMethod = "GET", notes = "获取BO建模中是否存在子表、孙表")
    public ObjectNode getSubEntInfo(@ApiParam(name = "ids", value = "bo定义id，多个用“,”号分割", required = true) @RequestParam String ids,
                                    @ApiParam(name = "isMobile", value = "是否移动端", required = false) @RequestParam Optional<Boolean> isMobile) throws Exception {
        ObjectNode infoNode = JsonUtil.getMapper().createObjectNode();
        boolean hasSub = false;
        boolean hasSun = false;
        ArrayNode entArray = JsonUtil.getMapper().createArrayNode();
        if (StringUtil.isNotEmpty(ids)) {
            String[] idArray = ids.split(",");
            for (String id : idArray) {
                BoDef def = baseService.getByDefId(id);
                BoEnt mainEnt = def.getBoEnt();
                entArray.add(getEntNode(mainEnt.getId(), mainEnt.getName(), mainEnt.getDesc(), "main", isMobile.orElse(false)));
                List<BoEnt> subEnts = mainEnt.getChildEntList();
                if (BeanUtils.isNotEmpty(subEnts)) {
                    hasSub = true;
                    for (BoEnt boEnt : subEnts) {
                        entArray.add(getEntNode(boEnt.getId(), boEnt.getName(), boEnt.getDesc(), "sub", isMobile.orElse(false)));
                        List<BoEnt> sunEnts = boEnt.getChildEntList();
                        if (BeanUtils.isNotEmpty(sunEnts)) {
                            hasSun = true;
                            for (BoEnt sunEnt : sunEnts) {
                                entArray.add(getEntNode(sunEnt.getId(), sunEnt.getName(), sunEnt.getDesc(), "sun", isMobile.orElse(false)));
                            }
                        }
                    }
                }
            }
        }
        infoNode.put("hasSub", hasSub);
        infoNode.put("hasSun", hasSun);
        infoNode.set("entList", entArray);
        return infoNode;
    }

    private ObjectNode getEntNode(String id, String alias, String name, String type, boolean isMobile) {
        ObjectNode entNode = JsonUtil.getMapper().createObjectNode();
        entNode.put("id", id);
        entNode.put("alias", alias);
        entNode.put("name", name);
        entNode.put("type", type);
        entNode.put("template", "main".equals(type) ? "grid" : "table");
        entNode.put("column", isMobile ? 1 : 2);
        return entNode;
    }

    @GetMapping("reloadByEntId")
    public @ResponseBody
    ObjectNode reloadByEntId(@RequestParam(value = "entId", required = true) String entId,
                             @RequestParam(value = "alias", required = true) String alias) throws Exception {
        return boEntManager.reloadByEntId(entId, alias);
    }

    private List<String> removeBoDefDetails(String... ids) {
        List<String> delIdList = new ArrayList<>();
        List<String> errList = new ArrayList<>();
        for (String id : ids) {
            //判断该bo是否绑定表单
            List<FormMeta> list = bpmFormDefManager.getByBODefId(id);
            if (list.size() == 0) {
                //没有绑定，放入待删除数组中
                delIdList.add(id);
            } else {
                List<String> formNameList = new ArrayList<>();
                for (FormMeta bpmFormDef : list) {
                    formNameList.add(bpmFormDef.getName());
                }
                BoDef bo = baseService.get(id);
                String msg = "【" + bo.getDescription() + "】已绑定表单：" + StringUtil.join(formNameList, ",");
                errList.add(msg);
            }
        }
        String[] delIds = delIdList.toArray(new String[delIdList.size()]);

        if (errList.size() == 0) {
            baseService.removeByIds(delIds);
            return errList;
        }
        return errList;
    }

    @RequestMapping(value = "reload", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "刷新元数据", httpMethod = "GET", notes = "刷新元数据")
    public CommonResult<String> reload(@ApiParam(name = "id", value = "id", required = true) @RequestParam(required = true) String id,
                                       @RequestParam(value = "alias", required = true) String alias) throws IOException {
        try {
            boEntManager.reloadByEntId(id, alias);
            return new CommonResult<String>(true, "刷新元数据成功");
        } catch (Exception e) {
            e.printStackTrace();
            return new CommonResult<String>(false, "刷新元数据失败");
        }
    }

    @RequestMapping(value = "deleteExternalEnt", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "删除外部表的子实体以及孙实体", httpMethod = "POST", notes = "删除外部表的子实体以及孙实体")
    public CommonResult<String> deleteExternalEnt(@ApiParam(name = "要被删除ent数据", value = "ent", required = true) @RequestBody Map<String, String> ent) throws Exception {
        try {
            String entjson = ent.get("entjson");
            baseService.deleteExternalEnt(entjson);
        } catch (Exception e) {
            return new CommonResult<>(false, "删除失败");
        }

        return new CommonResult<>(true, "删除成功!");
    }

    /**
     * 由于前端某些情况下无法校验到重复的字段，而且直接调用接口可能也会出现重复字段，一旦出现会导致子实体被清空，所以在保存前，后端再做一次校验
     *
     * @param json
     * @return
     */
    private CommonResult<String> validateBoAttr(String json) {
        try {
            JsonNode jsonObj = JsonUtil.toJsonNode(json);
            if (jsonObj.has("ents")) {
                ArrayNode entArray = (ArrayNode) jsonObj.get("ents");
                for (JsonNode entNode : entArray) {
                    if (entNode.has("attributeList")) {
                        String type = "main".equals(entNode.get("relation").asText()) ? "主" : "子";
                        ArrayNode attrArray = (ArrayNode) entNode.get("attributeList");
                        Set<String> attrSet = new HashSet<String>();
                        for (JsonNode attrNode : attrArray) {
                            String name = attrNode.get("name").asText();
                            if (attrSet.contains(name)) {
                                String entName = entNode.get("desc").asText();
                                return new CommonResult<String>(false, type + "实体【" + entName + "（" + entNode.get("name").asText() + "）】中的字段名：" + name + " 重复，请检查修改后再保存。");
                            } else {
                                attrSet.add(name);
                            }
                        }
                    }
                    if (entNode.has("children")) {
                        ArrayNode childrens = (ArrayNode) entNode.get("children");
                        if (BeanUtils.isNotEmpty(childrens)) {
                            for (JsonNode sunEntNode : childrens) {
                                if (sunEntNode.has("attributeList")) {
                                    ArrayNode attrArray = (ArrayNode) sunEntNode.get("attributeList");
                                    Set<String> attrSet = new HashSet<String>();
                                    for (JsonNode attrNode : attrArray) {
                                        String name = attrNode.get("name").asText();
                                        if (attrSet.contains(name)) {
                                            String entName = sunEntNode.get("desc").asText();
                                            return new CommonResult<String>(false, "孙实体【" + entName + "（" + sunEntNode.get("name").asText() + "）】中的字段名：" + name + " 重复，请检查修改后再保存。");
                                        } else {
                                            attrSet.add(name);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        } catch (IOException e) {
            return new CommonResult<String>(false, "保存数据校验不通过：" + e.getMessage());
        }
        return new CommonResult<String>(true, "校验通过");
    }

    @RequestMapping(value = "getPkTypeByDefAlias", method = RequestMethod.GET)
    @ApiOperation(value = "根据bo别名获取主键保存方式", httpMethod = "GET", notes = "根据bo别名获取主键保存方式")
    public String getPkTypeByDefAlias(@ApiParam(name = "alias", value = "alias") @RequestParam String alias) {
        BoDef def = baseService.getByAlias(alias);
        if (BeanUtils.isEmpty(def) || BeanUtils.isEmpty(def.getBoEnt())) {
            return null;
        }
        return def.getBoEnt().getPkType();
    }
}
