package com.artfess.device.monitor.controller;


import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.enums.ProductTypeEnum;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryField;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.device.base.manager.DeviceInfoManager;
import com.artfess.device.base.manager.DeviceProductInfoManager;
import com.artfess.device.base.manager.DeviceStatusLogManager;
import com.artfess.device.base.model.DeviceInfo;
import com.artfess.device.base.model.DeviceProductType;
import com.artfess.device.base.model.DeviceStatusLog;
import com.artfess.device.base.vo.CountVo;
import com.artfess.device.base.vo.DeviceStatusTimeVo;
import com.artfess.device.base.vo.FailureDeviceCountVo;
import com.artfess.device.monitor.manager.*;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

/**
 * 实时监控-实时数据
 *
 * @company 阿特菲斯信息技术有限公司
 * @author min.wu
 * @since 2022-08-08
 */
@Slf4j
@RestController
@Api(tags = "设施数据-实时数据")
@RequestMapping("/device/data/")
@ApiGroup(group = {ApiGroupConsts.DEVICE_BIZ})
public class DeviceDataController {

    @Autowired
    private DeviceDataBarricadeManager deviceDataBarricadeManager;

    @Autowired
    private DeviceDataBarricadeStatusManager deviceDataBarricadeStatusManager;

    @Autowired
    private DeviceDataHazardManager deviceDataHazardManager;

    @Autowired
    private DeviceDataSignboaerdManager deviceDataSignboaerdManager;

    @Autowired
    private DeviceDataWaterManager deviceDataWaterManager;

    @Autowired
    private DeviceProductInfoManager deviceProductInfoManager;

    @Autowired
    private DeviceInfoManager deviceInfoManager;

    @Autowired
    private DeviceStatusLogManager deviceStatusLogManager;


    @PostMapping(value="/query", produces={"application/json; charset=utf-8" })
    @ApiOperation("分页获取设备实时数据")
    public CommonResult query(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter queryFilter, String id) {
        DeviceProductType productType = deviceProductInfoManager.findByDeviceId(id);
        ProductTypeEnum type = ProductTypeEnum.getByType(productType.getCode());
        if(null == type) {
            return new CommonResult();
        }
        List<QueryField> querys = queryFilter.getQuerys();
        QueryField queryField = new QueryField();
        queryField.setProperty("device_code_");
        queryField.setValue(productType.getDeviceCode());
        queryField.setOperation(QueryOP.EQUAL);
        querys.add(queryField);
        CommonResult commonResult = new CommonResult();
        switch (type) {
            case barricade:
                commonResult = deviceDataBarricadeStatusManager.findByDeviceId(type.getCode(), queryFilter);
                break;
            case water:
                commonResult = deviceDataWaterManager.findByDeviceId(type.getCode(), queryFilter);
                break;
            case signboaerd:
                commonResult = deviceDataSignboaerdManager.findByDeviceId(type.getCode(), queryFilter);
                break;
            case hazard:
                commonResult = deviceDataHazardManager.findByDeviceId(type.getCode(), queryFilter);
                break;
            default:
                break;
        }
        return commonResult;
    }


    @GetMapping(value="/statistics/{productType}", produces={"application/json; charset=utf-8" })
    @ApiOperation("根据设备类型统计分析")
    public CommonResult statistics(@ApiParam(name="productType", value="设备类型,升降路障:barricade,危险源监测:hazard,低洼积水监测:water,标识牌监测:signboaerd") @PathVariable String productType) {
        List<CountVo> data = deviceInfoManager.statistics(productType);
        return CommonResult.success(data, "统计成功");
    }

    @PostMapping(value="/failureStatistics", produces={"application/json; charset=utf-8" })
    @ApiOperation("故障次数统计")
    public CommonResult failureStatistics(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<DeviceInfo> queryFilter) {
        PageList<FailureDeviceCountVo> pageList = deviceInfoManager.failureStatistics(queryFilter);

        return CommonResult.success(pageList, null);
    }


    @PostMapping(value="/timeStatistics", produces={"application/json; charset=utf-8" })
    @ApiOperation("分页获取设备时长统计")
    public CommonResult timeStatistics(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<DeviceStatusLog> queryFilter) {
        PageList<DeviceStatusTimeVo> pageList = deviceStatusLogManager.timeStatistics(queryFilter);

        return CommonResult.success(pageList, null);
    }

}
