package com.artfess.device.base.controller;

import com.alibaba.fastjson.JSON;
import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.annotation.PowerLogInfo;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.enums.LogType;
import com.artfess.base.enums.OperationType;
import com.artfess.base.enums.ProductTypeEnum;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryField;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.valid.AddGroup;
import com.artfess.base.valid.UpdateGroup;
import com.artfess.device.base.manager.DeviceInfoManager;
import com.artfess.device.base.model.DeviceInfo;
import com.artfess.device.base.vo.DeviceCountVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

/**
 * 设备信息 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author min.wu
 * @since 2022-07-15
 */
@Slf4j
@RestController
@Api(tags = "设施设备-设备管理")
@RequestMapping("/device/info/")
@ApiGroup(group = {ApiGroupConsts.DEVICE_BIZ})
public class DeviceInfoController extends BaseController<DeviceInfoManager, DeviceInfo> {


    @PostMapping("/findAll")
    @ApiOperation("获取设备列表")
    @PowerLogInfo(logType = LogType.BIZ, operaionType = OperationType.QUERY, description = "获取设备列表")
    public CommonResult<String> findAll(@ApiParam(name = "model", value = "获取设备列表") @RequestBody DeviceInfo entity) {
        log.info("获取设备列表请求参数:{}", JSON.toJSONString(entity));
        List<DeviceInfo> list = baseService.findAll(entity);
        return CommonResult.success(list, null);
    }

    @Override
    @PostMapping("/")
    @ApiOperation("添加实体的接口")
    public CommonResult<String> create(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({AddGroup.class}) DeviceInfo t) {
        String id = baseService.createInfo(t);
        if(!StringUtils.isNotBlank(id)) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
        return new CommonResult<>();
    }

    @Override
    @PutMapping("/")
    @ApiOperation("更新实体")
    public CommonResult<String> updateById(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({UpdateGroup.class})  DeviceInfo t) {
        String id = baseService.updateInfo(t);
        if(!StringUtils.isNotBlank(id)) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "更新实体失败");
        }
        return new CommonResult<>();
    }

    @PostMapping("/getDeviceInfo")
    @ApiOperation(value = "获取设备统计情况", notes = "0 全部设备; -1 报废; 1 正常; 2 故障; 3 报警; 9 其他; 10 已禁用; 11 已启用")
    @PowerLogInfo(logType = LogType.BIZ, operaionType = OperationType.QUERY, description = "获取设备统计情况")
    public CommonResult<String> getDeviceInfo(@ApiParam(name = "productId", value = "产品ID") @RequestParam String productId) {
        DeviceCountVo data = baseService.getDeviceInfo(productId);
        return CommonResult.success(data, null);
    }

    @PostMapping("/modifyEnabled")
    @ApiOperation("启禁用设备")
    @PowerLogInfo(logType = LogType.BIZ, operaionType = OperationType.UPDATE, description = "启禁用设备")
    public CommonResult<String> modifyEnabled(@ApiParam(name = "model", value = "设备") @RequestBody DeviceInfo entity) {
        log.info("启禁用设备请求参数:{}", JSON.toJSONString(entity));
        boolean b = baseService.modifyEnabled(entity);
        log.info("启禁用设备响应结果:{}", b);
        if (!b) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
        return new CommonResult<>();
    }

    @Override
    @GetMapping("/{id}")
    @ApiOperation("根据id查询实体")
    public DeviceInfo getById(@ApiParam(name="id", value="实体id") @PathVariable String id) {
        return baseService.findById(id);
    }

    @GetMapping("/detail/{id}")
    @ApiOperation("根据id获取设备详情")
    public DeviceInfo detail(@ApiParam(name="id", value="实体id") @PathVariable String id) {
        return baseService.detail(id);
    }



    @PostMapping(value="/findByProductType", produces={"application/json; charset=utf-8" })
    @ApiOperation("获取路障设备分页信息")
    public PageList<DeviceInfo> findByProductType(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<DeviceInfo> queryFilter) {
        List<QueryField> querys = queryFilter.getQuerys();
        QueryField queryField = new QueryField();
        queryField.setProperty("t.code_");
        queryField.setValue(ProductTypeEnum.barricade.getCode());
        queryField.setOperation(QueryOP.EQUAL);
        querys.add(queryField);
        return baseService.findByProductType(queryFilter);
    }

    @Override
    @PostMapping(value="/query", produces={"application/json; charset=utf-8" })
    @ApiOperation("分页查询结果")
    public PageList<DeviceInfo> query(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<DeviceInfo> queryFilter) {
        return baseService.findByPage(queryFilter);
    }

    @PostMapping("failureEquipment")
    @ApiOperation(value = "故障设备", notes = "包括设备和视频点位")
    public PageList<DeviceInfo> failureEquipment (@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<DeviceInfo> queryFilter) {
        return baseService.failureEquipment(queryFilter);
    }

}
