package com.artfess.poi.util;

import cn.hutool.core.util.NumberUtil;
import com.alibaba.fastjson.JSON;
import com.artfess.base.annotation.Excel;
import com.artfess.base.util.DateUtils;
import com.artfess.poi.DefaultExcelStyle;
import com.artfess.poi.style.font.BoldWeight;
import org.apache.poi.hssf.usermodel.DVConstraint;
import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFCellStyle;
import org.apache.poi.hssf.usermodel.HSSFDataValidation;
import org.apache.poi.hssf.usermodel.HSSFFont;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.hssf.util.HSSFColor;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.DateUtil;
import org.apache.poi.ss.usermodel.ExcelStyleDateFormatter;
import org.apache.poi.ss.usermodel.HorizontalAlignment;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.VerticalAlignment;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.apache.poi.ss.util.CellRangeAddressList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.ObjectUtils;
import org.springframework.util.ResourceUtils;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.ByteArrayOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.net.URLEncoder;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static java.time.format.DateTimeFormatter.ISO_DATE;

/**
 * Excel相关处理
 *
 * @author min.wu
 */
public class ExcelUtils<T> {

    public final static String EXCEL2003 = ".xls";

    private static final Logger log = LoggerFactory.getLogger(ExcelUtils.class);

    public Class<T> clazz;

    public ExcelUtils(Class<T> clazz) {
        this.clazz = clazz;
    }

    /**
     *
     * @param sheetName
     * @param input
     * @param theRow 从第几行开始读取数据
     * @return
     * @throws Exception
     */
    public List<T> importExcel(String sheetName, InputStream input, Integer theRow) throws Exception {
        if(null == theRow) {
            theRow = 1;
        }
        List<T> list = new ArrayList<T>();

        Workbook workbook = WorkbookFactory.create(input);
        Sheet sheet;
        if (!StringUtils.isEmpty(sheetName)) {
            sheet = workbook.getSheet(sheetName); // 如果指定sheet名,则取指定sheet中的内容.
        } else {
            sheet = workbook.getSheetAt(0); // 如果传入的sheet名不存在则默认指向第1个sheet.
        }
        int rows = sheet.getPhysicalNumberOfRows();

        if (rows > 0) {
            // 有数据时才处理
            Field[] allFields = clazz.getDeclaredFields(); // 得到类的所有field.
            Map<Integer, Field> fieldsMap = new HashMap<Integer, Field>(); // 定义一个map用于存放列的序号和field.
            for (Field field : allFields) {
                // 将有注解的field存放到map中.
                if (field.isAnnotationPresent(Excel.class)) {
                    Excel attr = field.getAnnotation(Excel.class);
                    int col = getExcelCol(attr.column());// 获得列号
                    field.setAccessible(true);// 设置类的私有字段属性可访问.
                    fieldsMap.put(col, field);
                }
            }
            for (int i = theRow; i < rows; i++) {
                // 从第2行开始取数据,默认第一行是表头.
                Row row = sheet.getRow(i);
                if (null == row) {
                    continue;
                }
                int cellNum = sheet.getRow(theRow - 1).getPhysicalNumberOfCells();
                T entity = null;
                for (int j = 0; j < cellNum; j++) {
                    Cell cell = row.getCell(j);
                    if (cell == null) {
                        continue;
                    }
                    entity = (entity == null ? clazz.newInstance() : entity);// 如果不存在实例则新建.
                    Field field = fieldsMap.get(j);// 从map中得到对应列的field.
                    if (null == field) {
                        continue;
                    }
                    Class<?> fieldType = field.getType();

                    Object cellFormatValue = getCellFormatValue(cell, fieldType);
                    if(!ObjectUtils.isEmpty(cellFormatValue)) {
                        field.set(entity, cellFormatValue);
                    }
                }
                if (entity != null) {
                    try {
                        list.add(entity);
                    } catch (Exception e) {
                        log.error("导入失败，异常信息：{},数据信息：{}", e.getMessage(), JSON.toJSON(entity));
                    }

                }
            }
        }

        return list;
    }


    /**
     * 获取单元格的值，数字类型返回字符串
     *
     * @param cell
     * @param fieldType
     * @return
     */
    public static Object getCellFormatValue(Cell cell, Class<?> fieldType) throws ParseException {
        Object cellValue = null;
        if (!ObjectUtils.isEmpty(cell)) {
            if (cell.getCellType() == CellType.FORMULA) {
                if (DateUtil.isCellDateFormatted(cell)) {
                    // 转换为日期格式yyyy-MM-dd
                    ExcelStyleDateFormatter formatter = new ExcelStyleDateFormatter("yyyy-MM-dd");
                    cellValue = formatter.format(cell.getDateCellValue());
                } else {
                    return getCellValueOfCellType(cell, fieldType);
                }
            } else {
                return getCellValueOfCellType(cell, fieldType);
            }
        } else {
            cellValue = "";
        }
        return cellValue;
    }

    private static Object getCellValueOfCellType(Cell cell, Class<?> fieldType) throws ParseException {
        Object cellValue = "";
        if (String.class == fieldType) {
            cellValue = String.valueOf(cell.getStringCellValue()).trim();
        } else if ((Integer.TYPE == fieldType) || (Integer.class == fieldType)) {
            cellValue = cell.getNumericCellValue();
        } else if ((Long.TYPE == fieldType) || (Long.class == fieldType)) {
            cellValue = cell.getNumericCellValue();
        } else if ((Float.TYPE == fieldType) || (Float.class == fieldType)) {
            cellValue = cell.getNumericCellValue();
        } else if ((Short.TYPE == fieldType) || (Short.class == fieldType)) {
            cellValue = cell.getNumericCellValue();
        } else if ((Double.TYPE == fieldType) || (Double.class == fieldType)) {
            cellValue = cell.getNumericCellValue();
        } else if (Character.TYPE == fieldType) {
            cellValue = Character.valueOf(cell.getStringCellValue().charAt(0));
        } else if (LocalDate.class == fieldType) {
            if(null != cell.getDateCellValue()) {
                try {
                    if ((DateUtil.isCellDateFormatted(cell))) {
                        Date d = cell.getDateCellValue();
                        LocalDate fillDate = LocalDate.parse(DateUtils.formatDate(d, "yyyy-MM-dd"), ISO_DATE);
                        cellValue = fillDate;
                    }
                }catch (Exception e){
                    String d = cell.getStringCellValue();
                    LocalDate fillDate = LocalDate.parse(DateUtils.formatDate(DateUtils.parseDate(d, "yyyy-MM-dd"), "yyyy-MM-dd"), ISO_DATE);
                    cellValue = fillDate;
                }
            }

        }else if (LocalDateTime.class == fieldType) {
            if ((DateUtil.isCellDateFormatted(cell))) {
                Date d = cell.getDateCellValue();
                LocalDate fillDate = LocalDate.parse(DateUtils.formatDate(d, "yyyy-MM-dd HH:mm:ss"), ISO_DATE);
                cellValue = fillDate;
            }

        }
        else if (BigDecimal.class == fieldType) {
            //保留4位小数
            cellValue = NumberUtil.roundDown(cell.getNumericCellValue(), 4);
        }
        return cellValue;
    }


    /**
     * 对list数据源将其里面的数据导入到excel表单
     *
     * @param sheetName 工作表的名称
     */
    public void exportExcel(HttpServletResponse response, HttpServletRequest request, List<T> list, String sheetName) {

        try {
            HSSFWorkbook workbook = createHssfWorkbook(response, request, list, sheetName);
            String filename = encodingFilename(sheetName);
            //TODO 2021.11.26 LZ 此处不再进行输出流操作
//            OutputStream out = new FileOutputStream(getfile() + filename);
//            workbook.write(out);
//            out.close();
            this.downloadExcel(workbook, filename, response);
        } catch (Exception e) {
            log.error("关闭flush失败{}", e);
        }
    }

    private HSSFWorkbook createHssfWorkbook(HttpServletResponse response, HttpServletRequest request, List<T> list, String sheetName) {
        // 得到所有定义字段
        Field[] allFields = clazz.getDeclaredFields();
        List<Field> fields = new ArrayList<Field>();
        // 得到所有field并存放到一个list中.
        for (Field field : allFields) {
            if (field.isAnnotationPresent(Excel.class)) {
                fields.add(field);
            }
        }

        // 产生工作薄对象
        HSSFWorkbook workbook = new HSSFWorkbook();

        // excel2003中每个sheet中最多有65536行
        int sheetSize = 65536;
        // 取出一共有多少个sheet.
        double sheetNo = Math.ceil(list.size() / sheetSize);
        for (int index = 0; index <= sheetNo; index++) {
            // 产生工作表对象
            HSSFSheet sheet = workbook.createSheet();
            // 冻结单元格
            sheet.createFreezePane(0, 1, 0, 1);
            if (sheetNo == 0) {
                workbook.setSheetName(index, sheetName);
            } else {
                // 设置工作表的名称.
                workbook.setSheetName(index, sheetName + index);
            }
            // 产生一行
            HSSFRow row = sheet.createRow(0);
            // 产生单元格
            HSSFCell cell = row.createCell(0);
            // 写入各个字段的列头名称
            for (int i = 0; i < fields.size(); i++) {
                Field field = fields.get(i);
                Excel attr = field.getAnnotation(Excel.class);
                // 获得列号
                int col = getExcelCol(attr.column());
                // 创建列
                cell = row.createCell(col);
                // 设置列中写入内容为String类型
                cell.setCellType(CellType.STRING);
                HSSFCellStyle cellStyle = workbook.createCellStyle();
                //居中
                cellStyle.setAlignment(HorizontalAlignment.CENTER);
                //垂直
                cellStyle.setVerticalAlignment(VerticalAlignment.CENTER);
                if (attr.name().indexOf("注：") >= 0) {
                    HSSFFont font = workbook.createFont();
                    font.setColor(HSSFFont.COLOR_RED);
                    cellStyle.setFont(font);
                    cellStyle.setFillForegroundColor(HSSFColor.HSSFColorPredefined.YELLOW.getIndex());
                    sheet.setColumnWidth(i, 6000);
                } else {
                    HSSFFont font = workbook.createFont();

                    font.setBold(BoldWeight.BOLD.getWeight());
                    // 选择需要用到的字体格式
                    cellStyle.setFont(font);
                    cellStyle.setFillForegroundColor(HSSFColor.HSSFColorPredefined.YELLOW.getIndex());
                    // 设置列宽
                    sheet.setColumnWidth(i, 3766);
                }
                DefaultExcelStyle defaultExcelStyle = new DefaultExcelStyle();

                cellStyle.setFillPattern(defaultExcelStyle.getFillPattern());
                cellStyle.setWrapText(true);
                cell.setCellStyle(cellStyle);
                // 写入列名
                cell.setCellValue(attr.name());

                // 如果设置了提示信息则鼠标放上去提示.
                if (!attr.prompt().trim().equals("")) {
                    // 这里默认设了2-101列提示.
                    setHSSFPrompt(sheet, "", attr.prompt(), 1, 100, col, col);
                }
                // 如果设置了combo属性则本列只能选择不能输入
                if (attr.combo().length > 0) {
                    // 这里默认设了2-101列只能选择不能输入.
                    setHSSFValidation(sheet, attr.combo(), 1, 100, col, col);
                }
            }

            int startNo = index * sheetSize;
            int endNo = Math.min(startNo + sheetSize, list.size());
            // 写入各条记录,每条记录对应excel表中的一行
            HSSFCellStyle cs = workbook.createCellStyle();
            // 居中
            cs.setAlignment(HorizontalAlignment.CENTER);
            //垂直
            cs.setVerticalAlignment(VerticalAlignment.CENTER);
            for (int i = startNo; i < endNo; i++) {
                row = sheet.createRow(i + 1 - startNo);
                // 得到导出对象.
                T vo = (T) list.get(i);
                for (int j = 0; j < fields.size(); j++) {
                    // 获得field.
                    Field field = fields.get(j);
                    // 设置实体类私有属性可访问
                    field.setAccessible(true);
                    Excel attr = field.getAnnotation(Excel.class);
                    try {
                        // 根据Excel中设置情况决定是否导出,有些情况需要保持为空,希望用户填写这一列.
                        if (attr.isExport()) {
                            // 创建cell
                            cell = row.createCell(getExcelCol(attr.column()));
                            cell.setCellStyle(cs);
                            try {
                                if (String.valueOf(field.get(vo)).length() > 10) {
                                    throw new Exception("长度超过10位就不用转数字了");
                                }
                                // 如果可以转成数字则导出为数字类型
                                BigDecimal bc = new BigDecimal(String.valueOf(field.get(vo)));
                                cell.setCellType(CellType.NUMERIC);
                                cell.setCellType(CellType.STRING);
                                cell.setCellValue(bc.doubleValue());
                            } catch (Exception e) {
                                cell.setCellType(CellType.STRING);
                                if (vo == null) {
                                    // 如果数据存在就填入,不存在填入空格.
                                    cell.setCellValue("");
                                } else {
                                    // 如果数据存在就填入,不存在填入空格.
                                    cell.setCellValue(field.get(vo) == null ? "" : String.valueOf(field.get(vo)));
                                }
                            }
                        }
                    } catch (Exception e) {
                        log.error("导出Excel失败{}", e);
                    }
                }
            }
        }
        return workbook;
    }

    public void downloadExcel(HSSFWorkbook workBook, String fileName, HttpServletResponse response) throws IOException {
        String filedisplay = URLEncoder.encode(fileName, "utf-8");
        response.setContentType("APPLICATION/OCTET-STREAM");
        response.setHeader("Access-Control-Expose-Headers", "Content-Disposition");
        response.addHeader("Content-Disposition", "attachment;filename=" + filedisplay);
        response.addHeader("filename", filedisplay);
        OutputStream os = null;
        try {
            os = response.getOutputStream();
            workBook.write(os);
            os.flush();
            os.close();
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (os != null) {
                os.close();
            }

        }
    }

    /**
     * 将EXCEL中A,B,C,D,E列映射成0,1,2,3
     *
     * @param col
     */
    public static int getExcelCol(String col) {
        col = col.toUpperCase();
        // 从-1开始计算,字母重1开始运算。这种总数下来算数正好相同。
        int count = -1;
        char[] cs = col.toCharArray();
        for (int i = 0; i < cs.length; i++) {
            count += (cs[i] - 64) * Math.pow(26, cs.length - 1 - i);
        }
        return count;
    }

    /**
     * 设置单元格上提示
     *
     * @param sheet         要设置的sheet.
     * @param promptTitle   标题
     * @param promptContent 内容
     * @param firstRow      开始行
     * @param endRow        结束行
     * @param firstCol      开始列
     * @param endCol        结束列
     * @return 设置好的sheet.
     */
    public static HSSFSheet setHSSFPrompt(HSSFSheet sheet, String promptTitle, String promptContent, int firstRow,
                                          int endRow, int firstCol, int endCol) {
        // 构造constraint对象
        DVConstraint constraint = DVConstraint.createCustomFormulaConstraint("DD1");
        // 四个参数分别是：起始行、终止行、起始列、终止列
        CellRangeAddressList regions = new CellRangeAddressList(firstRow, endRow, firstCol, endCol);
        // 数据有效性对象
        HSSFDataValidation data_validation_view = new HSSFDataValidation(regions, constraint);
        data_validation_view.createPromptBox(promptTitle, promptContent);
        sheet.addValidationData(data_validation_view);
        return sheet;
    }

    /**
     * 设置某些列的值只能输入预制的数据,显示下拉框.
     *
     * @param sheet    要设置的sheet.
     * @param textlist 下拉框显示的内容
     * @param firstRow 开始行
     * @param endRow   结束行
     * @param firstCol 开始列
     * @param endCol   结束列
     * @return 设置好的sheet.
     */
    public static HSSFSheet setHSSFValidation(HSSFSheet sheet, String[] textlist, int firstRow, int endRow,
                                              int firstCol, int endCol) {
        // 加载下拉列表内容
        DVConstraint constraint = DVConstraint.createExplicitListConstraint(textlist);
        // 设置数据有效性加载在哪个单元格上,四个参数分别是：起始行、终止行、起始列、终止列
        CellRangeAddressList regions = new CellRangeAddressList(firstRow, endRow, firstCol, endCol);
        // 数据有效性对象
        HSSFDataValidation data_validation_list = new HSSFDataValidation(regions, constraint);
        sheet.addValidationData(data_validation_list);
        return sheet;
    }

    /**
     * 编码文件名
     */
    public String encodingFilename(String filename) {
        Date now = new Date();
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMddHHmmss");
        filename = filename + "_" + dateFormat.format(now) + EXCEL2003;
        return filename;
    }

    public String getfile() throws FileNotFoundException {
        return ResourceUtils.getURL("classpath:").getPath();
    }

    public byte[] getByte(HttpServletResponse response, HttpServletRequest request, List<T> list, String fileName) {
        HSSFWorkbook workbook = createHssfWorkbook(response, request, list, fileName);

        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        try {
            workbook.write(bos);
        } catch (IOException e) {
        } finally {
            try {
                bos.close();
                workbook.close();
            } catch (IOException e) {
            }
        }
        byte[] bytes = bos.toByteArray();
        return bytes;
    }
}
