package com.artfess.portal.controller;

import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.model.CommonResult;
import com.artfess.base.util.FileUtil;
import com.artfess.base.util.HttpUtil;
import com.artfess.base.util.JsonUtil;
import com.artfess.base.util.StringUtil;
import com.artfess.base.util.ZipUtil;
import com.artfess.base.util.time.DateFormatUtil;
import com.artfess.sysConfig.persistence.manager.SysTagManager;
import com.artfess.sysConfig.persistence.model.SysTag;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.time.LocalDateTime;
import java.util.List;

/**
 * <pre>
 * 描述：portal_sys_tag 控制器类
 * 构建组：x7
 * 作者:heyf
 * 邮箱:heyf@jee-soft.cn
 * 日期:2020-07-27 11:17:28
 * 版权：广州宏天软件股份有限公司
 * </pre>
 */
@RestController
@RequestMapping(value = "/portal/sysTag/v1")
@Api(tags = "sysTagController")
@ApiGroup(group = {ApiGroupConsts.GROUP_SYSTEM})
public class SysTagController extends BaseController<SysTagManager, SysTag> {
    @Resource
    SysTagManager sysTagManager;

    /**
     * 新增portal_sys_tag
     *
     * @param sysTag
     * @return
     * @throws Exception
     * @throws
     */
    @PostMapping(value = "save")
    @ApiOperation(value = "新增,更新系统标签数据", httpMethod = "POST", notes = "新增,更新系统标签数据")
    public CommonResult<String> save(@ApiParam(name = "sysTag", value = "系统标签业务对象", required = true) @RequestBody SysTag sysTag) throws Exception {
        String msg = "添加系统标签成功";
        if (StringUtil.isEmpty(sysTag.getId())) {
            sysTagManager.createWithCheck(sysTag);
        } else {
            sysTagManager.update(sysTag);
            msg = "更新系统标签成功";
        }
        return new CommonResult<String>(msg);
    }

    /**
     * 批量删除portal_sys_tag记录
     *
     * @param ids
     * @return
     * @throws Exception
     * @throws
     */
    @DeleteMapping(value = "/remove")
    @ApiOperation(value = "批量删除系统标签记录", httpMethod = "DELETE", notes = "批量删除系统标签记录")
    public CommonResult<String> removes(@ApiParam(name = "ids", value = "业务主键数组,多个业务主键之间用逗号分隔", required = true) @RequestParam String... ids) throws Exception {
        sysTagManager.removeByIds(ids);
        return new CommonResult<String>(true, "批量删除成功");
    }

    @DeleteMapping(value = "/removeByTypeId")
    @ApiOperation(value = "批量删除系统标签记录", httpMethod = "DELETE", notes = "根据分类ID删除系统标签记录")
    public CommonResult<String> removeByTypeId(@ApiParam(name = "id", value = "分类ID", required = true) @RequestParam String id) throws Exception {
        sysTagManager.removeByTypeId(id);
        return new CommonResult<String>(true, "删除成功");
    }

    @GetMapping(value = "/tags")
    @ApiOperation(value = "根据分类ID获取系统标签记录", httpMethod = "GET", notes = "根据分类ID获取系统标签记录")
    public List<SysTag> getTagsByTypeId(@ApiParam(name = "id", value = "分类ID", required = true) @RequestParam String typeId) throws Exception {
        return sysTagManager.getTagsByTypeId(typeId);
    }

    @RequestMapping(value = "/export")
    @ApiOperation(value = "导出系统标签", httpMethod = "GET", notes = "根据分类ID导出系统标签")
    public void export(@ApiParam(name = "typeId", value = "分类ID", required = true) @RequestParam String typeId,
                       HttpServletRequest request, HttpServletResponse response) throws Exception {
        List<SysTag> tagsByTypeId = baseService.getTagsByTypeId(typeId);
        String json = JsonUtil.toJson(tagsByTypeId);
        String fileName = "sysTags_" + DateFormatUtil.format(LocalDateTime.now(), "yyyy_MMdd_HHmm");
        HttpUtil.downLoadFile(request, response, json, "sysTags.json", fileName);

    }

    @RequestMapping(value = "/import")
    @ApiOperation(value = "根据分类ID导入系统标签", httpMethod = "GET", notes = "根据分类ID导入系统标签")
    public CommonResult<String> importSysTag(@ApiParam(name = "typeId", value = "分类ID", required = true) String typeId, MultipartHttpServletRequest request) throws Exception {
        MultipartFile file = request.getFile("file");
        String unZipFilePath = "";
        try {
            String rootRealPath = (FileUtil.getIoTmpdir() + "/attachFiles/unZip/").replace("/", File.separator);
            FileUtil.createFolder(rootRealPath, true);
            String name = file.getOriginalFilename();
            String fileDir = StringUtil.substringBeforeLast(name, ".");
            ZipUtil.unZipFile(file, rootRealPath); // 解压文件
            unZipFilePath = rootRealPath + File.separator + fileDir; // 解压后文件的真正路径
            baseService.importFile(unZipFilePath, typeId);
            return new CommonResult<>(true, "导入成功");
        } catch (Exception e) {
            return new CommonResult<>(false, "导入失败：" + e.getMessage());
        } finally {
            if (StringUtil.isNotEmpty(unZipFilePath)) {
                File zipFile = new File(unZipFilePath);
                if (zipFile.exists()) {
                    zipFile.delete();
                }
            }
        }
    }
}
