package com.artfess.workflow.runtime.manager.impl;

import com.artfess.base.context.BaseContext;
import com.artfess.base.groovy.GroovyScriptEngine;
import com.artfess.base.handler.MultiTenantHandler;
import com.artfess.base.handler.MultiTenantIgnoreResult;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.model.CommonResult;
import com.artfess.base.util.Base64;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.JsonUtil;
import com.artfess.base.util.StringUtil;
import com.artfess.base.util.time.DateUtil;
import com.artfess.base.util.time.TimeUtil;
import com.artfess.bpm.api.helper.identity.BpmIdentityExtractService;
import com.artfess.bpm.api.model.identity.BpmIdentity;
import com.artfess.bpm.api.service.BoDataService;
import com.artfess.bpm.model.identity.DefaultBpmIdentity;
import com.artfess.bpm.persistence.manager.BpmDefinitionManager;
import com.artfess.bpm.persistence.model.DefaultBpmDefinition;
import com.artfess.bpm.util.BoDataUtil;
import com.artfess.uc.api.impl.util.ContextUtil;
import com.artfess.uc.api.model.IUser;
import com.artfess.workflow.runtime.dao.BpmAutoStartConfDao;
import com.artfess.workflow.runtime.manager.BpmAutoStartConfManager;
import com.artfess.workflow.runtime.manager.IProcessManager;
import com.artfess.workflow.runtime.model.BpmAutoStartConf;
import com.artfess.workflow.runtime.params.StartFlowParamObject;
import com.artfess.workflow.runtime.params.StartResult;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.IOException;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;

/**
 * 
 * <pre>
 *  
 * 描述：流程自动发起配置表 处理实现类
 * 构建组：x7
 * 作者:heyf
 * 邮箱:heyf@jee-soft.cn
 * 日期:2020-04-07 10:51:28
 * 版权：广州宏天软件股份有限公司
 * </pre>
 */
@Service("boAutoStartConfManager")
public class BpmAutoStartConfManagerImpl extends BaseManagerImpl<BpmAutoStartConfDao, BpmAutoStartConf>
implements BpmAutoStartConfManager {
	@Resource
	BpmAutoStartConfDao boAutoStartConfDao;
	@Resource
	BpmIdentityExtractService bpmIdentityExtractService;
	@Resource
	BoDataService boDataService;
	@Resource
	BpmDefinitionManager bpmDefinitionManager;
	@Resource
	IProcessManager iProcessService;
	@Resource
	GroovyScriptEngine groovyScriptEngine;
	@Resource
	BaseContext baseContext;

	@Override
	public BpmAutoStartConf getByDefKey(String defKey) {
		return this.getOne(Wrappers.<BpmAutoStartConf> lambdaQuery().eq(BpmAutoStartConf::getDefKey, defKey));
	}

	@Override
	public ObjectNode defAutoStart() throws Exception {
		List<BpmAutoStartConf> all = null;
		try(MultiTenantIgnoreResult setThreadLocalIgnore = MultiTenantHandler.setThreadLocalIgnore()){
			all = super.getAll();
		}

		if (BeanUtils.isNotEmpty(all)) {
			for (BpmAutoStartConf conf : all) {
				String trigger = conf.getTrigger();	
				String startUser = conf.getStartUser();
				// 1,未配置启动人，2，未配置触发时机，3如果未到启动时间或者已过。 则跳过
				if (StringUtil.isEmpty(startUser) || StringUtil.isEmpty(trigger) || !checkTrigger(trigger) ) {
					continue;
				}
				// 将流程的租户id设置为当前线程的租户id
				baseContext.setTempTenantId(conf.getTenantId());

				List<DefaultBpmIdentity> list = JsonUtil.toBean(startUser,
						new TypeReference<List<DefaultBpmIdentity>>() {
				});
				if (BeanUtils.isEmpty(list)) {
					continue;
				}
				// 重构代码 将原来的for循环改为下面的流处理方式 jdk8特性
				List<BpmIdentity> bpmIdentities = list.parallelStream().filter(predicate->{
					return StringUtil.isNotEmpty(predicate.getId()) || StringUtil.isNotEmpty(predicate.getCode());
				}).collect(Collectors.<BpmIdentity>toList());

				List<IUser> extractUser = bpmIdentityExtractService.extractUser(bpmIdentities);
				if (BeanUtils.isEmpty(extractUser)) {
					continue;
				}
				DefaultBpmDefinition mainDef = bpmDefinitionManager.getMainByDefKey(conf.getDefKey(), false);
				List<ObjectNode> boDatas = boDataService.getDataByDefId(mainDef.getId());
				ObjectNode jsondata = BoDataUtil.hanlerData(mainDef.getId(), boDatas, true);
				ExecutorService executorService = Executors.newCachedThreadPool();
				for (IUser iUser : extractUser) {
					executorService.execute(() -> {
						try {
							startByConf(iUser, conf, jsondata, mainDef);
						} catch (Exception e) {
							e.printStackTrace();
						}
					});
				}
				baseContext.clearTempTenantId();
			}
		}
		return (ObjectNode) JsonUtil.toJsonNode(new CommonResult<>("执行成功"));
	}

	/**
	 * 根据配置自动启动流程
	 * 
	 * @param iUser
	 * @param conf
	 * @param jsondata
	 * @return
	 * @throws Exception
	 */
	private StartResult startByConf(IUser iUser, BpmAutoStartConf conf, ObjectNode jsondata,
			DefaultBpmDefinition mainDef) throws Exception {
		ContextUtil.setCurrentUser(iUser);
		baseContext.setTempTenantId(conf.getTenantId());
		StartFlowParamObject sObject = new StartFlowParamObject();
		sObject.setAccount(iUser.getAccount());
		sObject.setDefId(mainDef.getDefId());
		sObject.setFlowKey(mainDef.getDefKey());
		sObject.setExpression(BpmAutoStartConf.START_OPINION);
		String formData = conf.getFormData();
		if (StringUtil.isNotEmpty(formData)) {
			ObjectNode dataConf = (ObjectNode) JsonUtil.toJsonNode(formData);
			Map<String, Object> varMap = new HashMap<>();
			varMap.put("startUser", iUser.getUserId());
			varMap.put("flowKey_", mainDef.getDefKey());
			for (Iterator<Entry<String, JsonNode>> iterator = jsondata.fields(); iterator.hasNext();) {
				Entry<String, JsonNode> ent = iterator.next();
				String entKey = ent.getKey();
				if (ent.getValue() instanceof ObjectNode) {
					ObjectNode entity = (ObjectNode) ent.getValue();
					for (Iterator<Entry<String, JsonNode>> iterator2 = entity.fields(); iterator2.hasNext();) {
						Entry<String, JsonNode> filed = iterator2.next();
						String filePath = entKey + "." + filed.getKey();
						if (BeanUtils.isNotEmpty(dataConf.get(filePath))) {
							Object executeObject = groovyScriptEngine.executeObject(dataConf.get(filePath).asText(),
									varMap);
							JsonUtil.putObjectToJson(entity, filed.getKey(), executeObject);
						}
					}
				}

			}
		}
		sObject.setData(Base64.getBase64(JsonUtil.toJson(jsondata)));
		CompletableFuture<StartResult> start = iProcessService.start(sObject);
		return start.get();
	}

	/**
	 * 检查流程是否到了配置的启动时间
	 * 
	 * @param trigger
	 * @return
	 * @throws IOException
	 */
	private boolean checkTrigger(String trigger) throws IOException {
		ObjectNode triggerObj = (ObjectNode) JsonUtil.toJsonNode(trigger);
		if (!triggerObj.hasNonNull("rdoTimeType")) {
			return false;
		}
		String triggerType = triggerObj.get("rdoTimeType").asText();
		String curDateStr = DateUtil.getCurrentTime("HH:mm");
		String triggerDate = "";
		switch (triggerType) {
		// 只触发一次
		case "1":
			curDateStr = DateUtil.getCurrentTime("yyyy-MM-dd HH:mm");
			triggerDate = JsonUtil.getString(triggerObj, "sampleDate");
			if (StringUtil.isNotEmpty(triggerDate)) {
				triggerDate = TimeUtil.getDateString(TimeUtil.convertString(triggerDate), "yyyy-MM-dd HH:mm");
			}
			// 如果到了触发时间，则返回可以执行的结果
			if (curDateStr.equals(triggerDate)) {
				return true;
			}
			break;
			// 每天定时触发
		case "2":
			triggerDate = JsonUtil.getString(triggerObj, "txtDay");
			// 如果到了触发时间，则返回可以执行的结果
			if (curDateStr.equals(triggerDate)) {
				return true;
			}
			break;
			// 每周定时触发
		case "4":
			triggerDate = JsonUtil.getString(triggerObj, "txtWeek");
			String chkWeek = JsonUtil.getString(triggerObj, "chkWeek");
			// 如果到了触发时间，则返回可以执行的结果
			if (curDateStr.equals(triggerDate)
					&& chkWeek.indexOf(String.valueOf(LocalDateTime.now().getDayOfWeek().getValue())) > -1) {
				return true;
			}
			break;
		case "5":
			triggerDate = triggerObj.get("txtMon").asText();
			String chkMons = JsonUtil.getString(triggerObj, "chkMons");
			// 如果到了触发时间，则返回可以执行的结果
			if (curDateStr.equals(triggerDate)
					&& chkMons.indexOf(String.valueOf(LocalDateTime.now().getDayOfMonth())) > -1) {
				return true;
			}
			break;
		default:
			break;
		}
		return false;
	}

}
