package com.artfess.examine.controller;


import com.alibaba.fastjson.JSON;
import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.annotation.PowerLogInfo;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.context.BaseContext;
import com.artfess.base.controller.BaseController;
import com.artfess.base.enums.CommonRedisKey;
import com.artfess.base.enums.LogType;
import com.artfess.base.enums.OperationType;
import com.artfess.base.enums.QuestionStateEnum;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.exception.BaseException;
import com.artfess.base.exception.RequiredException;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.Direction;
import com.artfess.base.query.FieldSort;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.util.AuthenticationUtil;
import com.artfess.examine.manager.ExamUserRecordManager;
import com.artfess.examine.model.ExamPaperBase;
import com.artfess.examine.model.ExamUserRecord;
import com.artfess.examine.vo.*;
import com.artfess.poi.util.ExcelUtils;
import com.artfess.poi.util.FileDownloadUtil;
import com.artfess.redis.util.RedisLockUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ClassPathResource;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 考生考试记录（人员考试成绩） 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author min.wu
 * @since 2022-10-19
 */
@Slf4j
@RestController
@Api(tags = "考试管理-考试评卷")
@RequestMapping("/exam/user/record")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class ExamUserRecordController extends BaseController<ExamUserRecordManager, ExamUserRecord> {

    @Autowired
    private RedisLockUtils redisLockUtils;

    @Autowired
    private ExamUserRecordManager userRecordManager;

    @Resource
    BaseContext baseContext;


    @PostMapping(value="/myPaper", produces={"application/json; charset=utf-8" })
    @ApiOperation("我的试卷")
    public PageList<ExamUserRecord> myPaper(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<ExamUserRecord> queryFilter) {
        queryFilter.addFilter("user_id_", AuthenticationUtil.getCurrentUserId(), QueryOP.EQUAL);
        return baseService.myPaper(queryFilter);
    }

    /**
     * 保存得分
     * @param reqVo
     * @return
     */
    @ApiOperation(value = "阅卷-保存分数")
    @PostMapping("/saveScore")
    public CommonResult saveScore(@RequestBody SubmitAnswerReqVo reqVo){
        log.info("保存得分》入参：{}", JSON.toJSONString(reqVo));
        baseService.saveScore(reqVo);
        return new CommonResult();
    }


    @GetMapping("/getTaskInfo/{id}")
    @ApiOperation("我的试卷封面")
    public ExamPaperBase getTaskInfo(@ApiParam(name = "id", value = "考试记录id") @PathVariable String id) {
        ExamPaperBase examPaperTask = userRecordManager.findByMyRecord(id);
        return examPaperTask;
    }

    @GetMapping("/getUserRecord/{id}")
    @ApiOperation("答卷记录")
    public MyExamInfoVo getUserRecord(@ApiParam(name = "id", value = "考试记录id") @PathVariable String id) {
        MyExamInfoVo myExamInfoVo = userRecordManager.getUserRecord(id);
        return myExamInfoVo;
    }

    @GetMapping("/myPaperInfo/{id}")
    @ApiOperation("我的答题记录（包括成绩信息）")
    public MyExamInfoVo myPaperInfo(@ApiParam(name = "id", value = "考试记录id") @PathVariable String id) {
        MyExamInfoVo myExamInfoVo = userRecordManager.myPaperInfo(id);
        return myExamInfoVo;
    }

    /**
     * 开始考试
     *
     * @param reqVo
     * @return
     */
    @PostMapping("/startExam")
    @ApiOperation("开始考试-进入我的试卷开始答题")
    public MyExamInfoVo startExam(@RequestBody ExamReqVo reqVo) {
        String currentUserId = AuthenticationUtil.getCurrentUserId();
        reqVo.setUserId(currentUserId);
        log.info("开始考试》入参：{}", JSON.toJSONString(reqVo));
        MyExamInfoVo myExamInfoVo = userRecordManager.startExam(reqVo);
        log.info("开始考试》响应：{}", JSON.toJSONString(myExamInfoVo));
        return myExamInfoVo;
    }

    /**
     * 提交答卷
     *
     * @param vo
     */
    @PostMapping("/submitAnswer")
    @ApiOperation(value = "提交答卷")
    public CommonResult submitAnswer(@RequestBody SubmitAnswerReqVo vo) {
        Assert.hasText(vo.getRecordId(), "考试记录id不能为空");

        log.info("提交答卷》参数：{}", JSON.toJSONString(vo));

        CommonRedisKey commonLockType = CommonRedisKey.SUBMIT_ANSWER_KEY;
        boolean lock = redisLockUtils.tryLock(commonLockType.name() + ":" + vo.getRecordId(), 10, TimeUnit.SECONDS);
        if (!lock) {
            throw new RuntimeException("系统繁忙");
        }
        try {
            String currentUserId = AuthenticationUtil.getCurrentUserId();
            vo.setUserId(currentUserId);
            userRecordManager.submitAnswer(vo);
            return new CommonResult();
        } catch (BaseException e) {
            log.error("提交答卷:{}", e.getMessage());
            return new CommonResult();
        } finally {
            redisLockUtils.unLock(commonLockType.name() + ":" + vo.getRecordId());
        }
    }

    //todo 任务状态 已结束和补考中
    @GetMapping("/switchCount/{id}")
    @ApiOperation("增加页面切换次数")
    public CommonResult switchCount(@ApiParam(name = "id", value = "考试记录id") @PathVariable String id) {
        userRecordManager.switchCount(id);
        return new CommonResult();
    }

    @GetMapping("/updateStatus/{id}")
    @ApiOperation("设为作弊")
    public CommonResult updateStatus(@ApiParam(name = "id", value = "考试记录id") @PathVariable String id) {
        userRecordManager.updateStatus(id);
        return new CommonResult();
    }

    @GetMapping("/applyExamPaper/{id}")
    @ApiOperation("申请补考（试卷已结束 显示该按钮）")
    public CommonResult applyExamPaper(@ApiParam(name = "id", value = "考试记录id") @PathVariable String id) {
        userRecordManager.applyExamPaper(id);
        return new CommonResult();
    }

    @PostMapping(value="/applyPaperRecord", produces={"application/json; charset=utf-8" })
    @ApiOperation("补考申请")
    public PageList<ExamUserRecord> applyPaperRecord(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<ExamUserRecord> queryFilter) {
        queryFilter.addFilter("s.status_", "6", QueryOP.EQUAL);
        List<FieldSort> sorter = queryFilter.getSorter();
        sorter.add(new FieldSort("s.apply_time_", Direction.DESC));
        queryFilter.setSorter(sorter);
        List<String> currentAndChildOrgIds = baseContext.getCurrentAndChildOrgIds();
        queryFilter.addFilter("b.create_org_id_", currentAndChildOrgIds, QueryOP.IN);
        return baseService.myPaper(queryFilter);
    }

    @PostMapping("/modifyStatus")
    @ApiOperation("补考申请审批")
    @PowerLogInfo(logType = LogType.BIZ, operaionType = OperationType.UPDATE, description = "补考申请审批")
    public CommonResult<String> modifyEnabled(@ApiParam(name = "model", value = "公告") @RequestBody ApplyPaperReqVo reqVo) {
        log.info("补考申请审批请求参数:{}", JSON.toJSONString(reqVo));
        boolean b = baseService.modifyStatus(reqVo);
        log.info("补考申请审批响应结果:{}", b);
        if (!b) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
        return new CommonResult<>();
    }

    @PostMapping(value="/paperQuery", produces={"application/json; charset=utf-8" })
    @ApiOperation("考试查询")
    public PageList<ExamUserRecord> paperQuery(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<ExamUserRecord> queryFilter) {
        List<FieldSort> sorter = queryFilter.getSorter();
        sorter.add(new FieldSort("s.status_", Direction.DESC));
        queryFilter.setSorter(sorter);
        List<String> currentAndChildOrgIds = baseContext.getCurrentAndChildOrgIds();
        queryFilter.addFilter("b.create_org_id_", currentAndChildOrgIds, QueryOP.IN);
        return baseService.myPaper(queryFilter);
    }

    @PostMapping(value="/paperEvaluationQuery", produces={"application/json; charset=utf-8" })
    @ApiOperation("考试评卷")
    public PageList<ExamUserRecord> paperEvaluationQuery(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<ExamUserRecord> queryFilter) {
        queryFilter.addFilter("s.status_", "3", QueryOP.EQUAL);
        List<String> currentAndChildOrgIds = baseContext.getCurrentAndChildOrgIds();
        queryFilter.addFilter("b.create_org_id_", currentAndChildOrgIds, QueryOP.IN);
        return baseService.myPaper(queryFilter);
    }


    @PostMapping(value="/myResultsQuery", produces={"application/json; charset=utf-8" })
    @ApiOperation("我的成绩")
    public PageList<ExamUserRecord> myResultsQuery(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<ExamUserRecord> queryFilter) {
        queryFilter.addFilter("user_id_", AuthenticationUtil.getCurrentUserId(), QueryOP.EQUAL);
        queryFilter.addFilter("s.status_", QuestionStateEnum.finish.getType(), QueryOP.EQUAL);
        List<FieldSort> sorter = queryFilter.getSorter();
        sorter.add(new FieldSort("s.end_time_", Direction.DESC));
        queryFilter.setSorter(sorter);
        return baseService.myPaper(queryFilter);
    }

    @PostMapping(value="/resultsQuery", produces={"application/json; charset=utf-8" })
    @ApiOperation("成绩查询")
    public PageList<ExamUserRecord> resultsQuery(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<ExamUserRecord> queryFilter) {
//        queryFilter.addFilter("s.status_", QuestionStateEnum.finish.getType(), QueryOP.EQUAL);
        List<FieldSort> sorter = queryFilter.getSorter();
        sorter.add(new FieldSort("s.end_time_", Direction.DESC));
        queryFilter.setSorter(sorter);
        List<String> currentAndChildOrgIds = baseContext.getCurrentAndChildOrgIds();
        queryFilter.addFilter("b.create_org_id_", currentAndChildOrgIds, QueryOP.IN);
        return baseService.myPaper(queryFilter);
    }

    @ApiOperation(value = "成绩导入")
    @PostMapping("/importExcel")
    public CommonResult<String> importExcel(@RequestParam("file") MultipartFile file) {
        try {
            return new CommonResult<>();
        } catch (Exception e) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
    }

    @ApiOperation(value = "成绩导出")
    @PostMapping("/export")
    public void export(HttpServletResponse response, HttpServletRequest request,
                       @ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<ExamUserRecord> queryFilter) throws Exception {
        queryFilter.addFilter("s.status_", QuestionStateEnum.finish.getType(), QueryOP.EQUAL);
        List<FieldSort> sorter = queryFilter.getSorter();
        sorter.add(new FieldSort("s.end_time_", Direction.DESC));
        queryFilter.setSorter(sorter);
        PageList<ExamUserRecord> page = baseService.myPaper(queryFilter);
        ExcelUtils<ExamUserRecord> util = new ExcelUtils<ExamUserRecord>(ExamUserRecord.class);
        util.exportExcel(response, request, page.getRows(), "成绩信息");
    }

    /**
     * 下载导入模板
     *
     * @param response
     * @return
     */
    @ApiOperation(value = "下载导入模板")
    @GetMapping("/downModel")
    public void downTemplate(HttpServletResponse response) {
        try {
            ClassPathResource classPathResource = new ClassPathResource("model/导入模板.xlsx");
            FileDownloadUtil.fileDownload(response, classPathResource.getInputStream(), "导入模板.xlsx");
        } catch (Exception e) {
            response.setCharacterEncoding("utf-8");
            throw new RequiredException("你所下载的资源不存在");
        }
    }

    @ApiOperation(value = "获取用户年度科目成绩")
    @GetMapping("/getSubjectScore/{userId}/{year}")
    public CommonResult<List<YearSubjectScoreVO>> getYearSubjectScore(@ApiParam(name = "userId", value = "用户ID") @PathVariable String userId,
                                                                      @ApiParam(name = "year", value = "查询的年份") @PathVariable Integer year) {
        List<YearSubjectScoreVO> list = baseService.getYearSubjectScore(userId, year);
        return new CommonResult<>(true, "获取成功！", list);
    }
}
