package com.artfess.examine.controller;


import com.alibaba.fastjson.JSON;
import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.annotation.PowerLogInfo;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.enums.LogType;
import com.artfess.base.enums.OperationType;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.exception.BaseException;
import com.artfess.base.exception.RequiredException;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.Direction;
import com.artfess.base.query.FieldSort;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.valid.AddGroup;
import com.artfess.base.valid.UpdateGroup;
import com.artfess.examine.manager.ExamQuestionsInfoManager;
import com.artfess.examine.manager.ExamSubjectInfoManager;
import com.artfess.examine.model.ExamQuestionsInfo;
import com.artfess.examine.model.ExamSubjectInfo;
import com.artfess.examine.model.ExamSubjectPos;
import com.artfess.examine.model.ExamSubjectType;
import com.artfess.examine.vo.SubjectReqVo;
import com.artfess.examine.vo.UserInfoVo;
import com.artfess.poi.util.ExcelUtils;
import com.artfess.poi.util.FileDownloadUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ClassPathResource;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Arrays;
import java.util.List;

/**
 * 训练考试科目 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author min.wu
 * @since 2022-10-19
 */
@Slf4j
@RestController
@Api(tags = "基础数据-科目管理")
@RequestMapping("/exam/subject/info/")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class ExamSubjectInfoController extends BaseController<ExamSubjectInfoManager, ExamSubjectInfo> {

    @Autowired
    private ExamQuestionsInfoManager questionsInfoManager;

    @PostMapping("/getTree")
    @ApiOperation("获取科目下拉树")
    @PowerLogInfo(logType = LogType.BIZ, operaionType = OperationType.QUERY, description = "获取科目下拉树")
    public CommonResult<String> getTree(@ApiParam(name = "model", value = "获取科目下拉树") @RequestBody ExamSubjectInfo entity) {
        log.info("获取科目下拉树请求参数:{}", JSON.toJSONString(entity));
        List<ExamSubjectType> list = baseService.getTree(entity);
        return CommonResult.success(list, null);
    }


    @PostMapping(value="/findByPage", produces={"application/json; charset=utf-8" })
    @ApiOperation("根据专业装备获取对应科目信息")
    public PageList<ExamSubjectInfo> findByPage(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<ExamSubjectInfo> queryFilter) {
        queryFilter.addFilter("s.IS_DELE_", '0', QueryOP.EQUAL);
        List<FieldSort> sorter = queryFilter.getSorter();
        sorter.add(new FieldSort("s.create_time_", Direction.DESC));
        queryFilter.setSorter(sorter);
        return baseService.findByPage(queryFilter);
    }

    @PostMapping(value="/notSubjectPage", produces={"application/json; charset=utf-8" })
    @ApiOperation("获取未被年度考核选择的科目信息")
    public PageList<ExamSubjectInfo> notSubjectPage(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<ExamSubjectInfo> queryFilter) {
        queryFilter.addFilter("IS_DELE_", '0', QueryOP.EQUAL);
        return baseService.notSubjectPage(queryFilter);
    }

    @Override
    @PostMapping("/")
    @ApiOperation("添加实体的接口")
    public CommonResult<String> create(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({AddGroup.class}) ExamSubjectInfo t) {
        boolean result = baseService.createInfo(t);
        if(!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
        return new CommonResult<>();
    }

    @Override
    @PutMapping("/")
    @ApiOperation("更新实体")
    public CommonResult<String> updateById(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({UpdateGroup.class}) ExamSubjectInfo t) {
        boolean result = baseService.updateInfo(t);
        if(!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "更新实体失败");
        }
        return new CommonResult<>();
    }

    @GetMapping("/posList/{id}")
    @ApiOperation("试卷-根据课目id获取已绑定的组织岗位（）")
    public List<ExamSubjectPos> findByBindPos(@ApiParam(name="id", value="实体id") @PathVariable String id) {
        return baseService.findByBindPos(id);
    }

    @GetMapping("/findById/{id}")
    @ApiOperation("根据id获取科目信息")
    public ExamSubjectInfo findById(@ApiParam(name="id", value="实体id") @PathVariable String id) {
        return baseService.findById(id);
    }

    @PostMapping("/findBySubjectIds")
    @ApiOperation("试卷-根据选中课目id获取已绑定的组织岗位（考试选择课目后调用）")
    public List<ExamSubjectPos> findBySubjectIds(@ApiParam(name="id", value="实体id") @RequestBody SubjectReqVo t) {
        return baseService.findBySubjectIds(t);
    }

    @PostMapping("/getSubjectList")
    @ApiOperation("根据科目id获取科目信息")
    public List<ExamSubjectInfo> getSubjectList(@ApiParam(name="SubjectReqVo", value="实体") @RequestBody SubjectReqVo t) {
        return baseService.getSubjectList(t);
    }

    @PostMapping("/getSubjectUserList")
    @ApiOperation(value = "根据选中科目id获取科目下岗位关联人员")
    public List<UserInfoVo> getSubjectUserList(@ApiParam(name = "SubjectReqVo", value = "实体") @RequestBody SubjectReqVo t){
        return baseService.getSubjectUserList(t);
    }


    @ApiOperation(value = "导入")
    @PostMapping("/importExcel")
    public CommonResult<String> importExcel(@RequestParam("file") MultipartFile file) {
        try {
            ExcelUtils<ExamSubjectInfo> excelUtil = new ExcelUtils<>(ExamSubjectInfo.class);
            List<ExamSubjectInfo> list = excelUtil.importExcel(null, file.getInputStream());
            boolean result = baseService.importExcel(list);
            if(!result) {
                throw new IllegalArgumentException("导入失败");
            }
            return new CommonResult<>();
        } catch (Exception e) {
            throw new IllegalArgumentException("导入失败,"+ e.getMessage());
        }
    }

    @ApiOperation(value = "导出")
    @PostMapping("/export")
    public void export(HttpServletResponse response, HttpServletRequest request,
                       @ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<ExamSubjectInfo> queryFilter) throws Exception {
        PageList<ExamSubjectInfo> page = baseService.findByPage(queryFilter);
        ExcelUtils<ExamSubjectInfo> util = new ExcelUtils<ExamSubjectInfo>(ExamSubjectInfo.class);
        util.exportExcel(response, request, page.getRows(), "课目信息");
    }

    /**
     * 下载导入模板
     *
     * @param response
     * @return
     */
    @ApiOperation(value = "下载导入模板")
    @GetMapping("/downModel")
    public void downTemplate(HttpServletResponse response) {
        try {
            ClassPathResource classPathResource = new ClassPathResource("model/subject.xlsx");
            FileDownloadUtil.fileDownload(response, classPathResource.getInputStream(), "课目模板.xlsx");
        } catch (Exception e) {
            response.setCharacterEncoding("utf-8");
            throw new RequiredException("你所下载的资源不存在");
        }
    }

    @Override
    @DeleteMapping("/")
    @ApiOperation("根据id集合批量删除")
    public CommonResult<String> deleteByIds(@ApiParam(name="ids", value="实体集合") @RequestParam String...ids) {
        List<String> idList = Arrays.asList(ids);
        List<ExamQuestionsInfo> list = questionsInfoManager.getQuestionList(idList);
        if(!CollectionUtils.isEmpty(list)) {
            throw new BaseException("该课目下还有题目，请先删除题目信息");
        }

        boolean result = baseService.removeByIds(idList);
        if(!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "删除实体失败");
        }
        return new CommonResult<>();
    }


    @PostMapping(value="/coachSubjectList", produces={"application/json; charset=utf-8" })
    @ApiOperation("获取教练训练课目列表")
    public PageList<UserInfoVo> coachSubjectList(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<ExamSubjectInfo> queryFilter) {
        return baseService.coachSubjectList(queryFilter);
    }

}
