package com.artfess.examine.controller;


import com.alibaba.fastjson.JSON;
import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.enums.CommonRedisKey;
import com.artfess.base.enums.PaperTypeEnum;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.exception.BaseException;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.util.AuthenticationUtil;
import com.artfess.base.valid.AddGroup;
import com.artfess.base.valid.UpdateGroup;
import com.artfess.examine.manager.ExamImitateRecordManager;
import com.artfess.examine.manager.ExamPaperBaseManager;
import com.artfess.examine.model.ExamImitateRecord;
import com.artfess.examine.model.ExamPaperBase;
import com.artfess.examine.vo.ExamReqVo;
import com.artfess.examine.vo.MyExamInfoVo;
import com.artfess.examine.vo.QuestionsInfoVo;
import com.artfess.examine.vo.SubmitAnswerReqVo;
import com.artfess.redis.util.RedisLockUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.Assert;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 考生模拟考试记录（人员考试成绩） 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author min.wu
 * @since 2022-10-19
 */
@Slf4j
@RestController
@Api(tags = "基础数据-模拟考试")
@RequestMapping("/exam/imitate/record/")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class ExamImitateRecordController extends BaseController<ExamImitateRecordManager, ExamImitateRecord> {

    @Autowired
    private ExamPaperBaseManager paperBaseManager;

    @Autowired
    private RedisLockUtils redisLockUtils;

    @PutMapping("/update")
    @ApiOperation("修改模拟试卷")
    public CommonResult<String> updateById(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({UpdateGroup.class}) ExamPaperBase t) {
        t.setType(PaperTypeEnum.simulation.getType());
        String id = paperBaseManager.updateInfo(t);
        if(!StringUtils.isNotBlank(id)) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "更新实体失败");
        }
        return new CommonResult<>();
    }

    @PostMapping("/save")
    @ApiOperation("添加模拟试卷")
    public CommonResult<String> create(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({AddGroup.class}) ExamPaperBase t) {
        t.setType(PaperTypeEnum.simulation.getType());
        String id = paperBaseManager.createInfo(t);
        if(!StringUtils.isNotBlank(id)) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
        return new CommonResult<>();
    }

    @GetMapping("/findById/{id}")
    @ApiOperation("根据id获取模拟试卷信息")
    public ExamPaperBase findById(@ApiParam(name="id", value="模拟试卷id") @PathVariable String id) {
        return paperBaseManager.findById(id);
    }

    @GetMapping("viewPaper/{id}")
    @ApiOperation("预览试卷")
    public ExamPaperBase viewPaper(@ApiParam(name="id", value="模拟试卷id") @PathVariable String id) {
        return paperBaseManager.viewPaper(id);
    }


    @GetMapping("/getUserRecord/{id}")
    @ApiOperation("答卷记录")
    public MyExamInfoVo getUserRecord(@ApiParam(name = "id", value = "考试记录id") @PathVariable String id) {
        MyExamInfoVo myExamInfoVo = baseService.getUserRecord(id);
        return myExamInfoVo;
    }

    /**
     * 开始考试
     *
     * @param reqVo
     * @return
     */
    @PostMapping("/startExam")
    @ApiOperation("开始考试-进入我的试卷开始答题")
    public MyExamInfoVo startExam(@RequestBody ExamReqVo reqVo) {
        String currentUserId = AuthenticationUtil.getCurrentUserId();
        reqVo.setUserId(currentUserId);
        log.info("开始考试》入参：{}", JSON.toJSONString(reqVo));
        MyExamInfoVo myExamInfoVo = baseService.startExam(reqVo);
        log.info("开始考试》响应：{}", JSON.toJSONString(myExamInfoVo));
        return myExamInfoVo;
    }

    /**
     * 提交答卷
     *
     * @param vo
     */
    @PostMapping("/submitAnswer")
    @ApiOperation(value = "提交答卷")
    public CommonResult submitAnswer(@RequestBody SubmitAnswerReqVo vo) {
        Assert.hasText(vo.getRecordId(), "考试记录id不能为空");

        log.info("提交答卷》参数：{}", JSON.toJSONString(vo));

        CommonRedisKey commonLockType = CommonRedisKey.SUBMIT_ANSWER_KEY;
        boolean lock = redisLockUtils.tryLock(commonLockType.name() + ":" + vo.getRecordId(), 10, TimeUnit.SECONDS);
        if (!lock) {
            throw new RuntimeException("系统繁忙");
        }
        try {
            String currentUserId = AuthenticationUtil.getCurrentUserId();
            vo.setUserId(currentUserId);
            baseService.submitAnswer(vo);
            return new CommonResult();
        } catch (BaseException e) {
            log.error("提交答卷:{}", e.getMessage());
            return new CommonResult();
        } finally {
            redisLockUtils.unLock(commonLockType.name() + ":" + vo.getRecordId());
        }
    }


    @PostMapping(value="/trainingPaperQuery", produces={"application/json; charset=utf-8" })
    @ApiOperation("训练成绩")
    public PageList<ExamImitateRecord> trainingPaperQuery(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<ExamImitateRecord> queryFilter) {
        queryFilter.addFilter("user_id_", AuthenticationUtil.getCurrentUserId(), QueryOP.EQUAL);
        return baseService.trainingPaperQuery(queryFilter);
    }

    @GetMapping("/errorQuestionsList/{id}")
    @ApiOperation("错题本")
    public List<QuestionsInfoVo> errorQuestionsList(@ApiParam(name = "id", value = "试卷id") @PathVariable String id) {
        List<QuestionsInfoVo> list = baseService.errorQuestionsList(id);
        return list;
    }

}
