package com.artfess.ljzc.loan.manager.impl;

import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.StringUtil;
import com.artfess.ljzc.loan.dao.AssetLoanInfoDao;
import com.artfess.ljzc.loan.manager.AssetLoanInfoManager;
import com.artfess.ljzc.loan.manager.AssetLoanInterestManager;
import com.artfess.ljzc.loan.model.AssetLoanInfo;
import com.artfess.ljzc.loan.model.AssetLoanInterest;
import com.artfess.ljzc.loan.vo.MonthVo;
import com.artfess.uc.util.ContextUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.google.api.client.util.Sets;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 债权资产信息 服务实现类
 *
 * @author min.wu
 * @company 阿特菲斯信息技术有限公司
 * @since 2023-11-27
 */
@Service
public class AssetLoanInfoManagerImpl extends BaseManagerImpl<AssetLoanInfoDao, AssetLoanInfo> implements AssetLoanInfoManager {

    @Autowired
    AssetLoanInterestManager loanInterestManager;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public String insertLoanInfo(AssetLoanInfo loanInfo) throws Exception {
        if (BeanUtils.isNotEmpty(loanInfo)) {
            if (StringUtil.isEmpty(loanInfo.getAssetAudit())) {
                loanInfo.setAssetAudit("0");
            }
            if (StringUtil.isEmpty(loanInfo.getBelongsOrgId())) {
                loanInfo.setBelongsOrgId(ContextUtil.getCurrentOrgId());
                loanInfo.setBelongsOrgName(ContextUtil.getCurrentOrgName());
                loanInfo.setBelongsOrgFullId(ContextUtil.getCurrentOrgFullId());
            }
            this.baseMapper.insert(loanInfo);
            List<AssetLoanInterest> list = loanInfo.getLoanInterest();
            if (!CollectionUtils.isEmpty(list)) {
                list.forEach(s -> {
                    s.setLoanId(loanInfo.getId());
                });
                this.loanInterestManager.saveBatch(list);
            }
            return loanInfo.getId();
        }
        return null;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean updateLoanInfo(AssetLoanInfo loanInfo) throws Exception {
        if (BeanUtils.isNotEmpty(loanInfo)) {
            this.saveOrUpdate(loanInfo);
            List<AssetLoanInterest> list = loanInfo.getLoanInterest();
            QueryWrapper<AssetLoanInterest> delWrapper = new QueryWrapper<>();
            delWrapper.eq("loan_id_", loanInfo.getId());
            this.loanInterestManager.getBaseMapper().delete(delWrapper);
            if (!CollectionUtils.isEmpty(list)) {
                list.forEach(s -> {
                    s.setLoanId(loanInfo.getId());
                });
                this.loanInterestManager.saveOrUpdateBatch(list);
            }
            return true;
        }
        return null;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean deleteLoanInfo(String id) {
        Assert.hasText(id, "请选择要删除的资产！");
        boolean flag = this.removeById(id);
        if (flag) {
            QueryWrapper<AssetLoanInterest> delWrapper = new QueryWrapper<>();
            delWrapper.eq("loan_id_", id);
            this.loanInterestManager.getBaseMapper().delete(delWrapper);
        }
        return flag;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean deleteBachLoanInfo(List<String> ids) {
        Assert.noNullElements(ids, "请选择要删除的资产！");
        boolean flag = this.removeByIds(ids);
        if (flag) {
            QueryWrapper<AssetLoanInterest> delWrapper = new QueryWrapper<>();
            delWrapper.in("loan_id_", ids);
            this.loanInterestManager.getBaseMapper().delete(delWrapper);
        }
        return flag;
    }

    @Override
    public AssetLoanInfo findLoanInfo(String id) {
        Assert.hasText(id, "请选择要查询的资产！");
        AssetLoanInfo assetLoanInfo = this.getById(id);
        if (BeanUtils.isNotEmpty(assetLoanInfo)) {
            QueryWrapper<AssetLoanInterest> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq("loan_id_", id);
            List<AssetLoanInterest> list = this.loanInterestManager.list(queryWrapper);
            assetLoanInfo.setLoanInterest(list);
            return assetLoanInfo;
        }
        return null;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean auditLoan(String status, String msg, List<String> idList) {
        Assert.hasText(status, "状态不能为空！");
        Assert.notNull(idList, "请选择要审核的资产");
        Integer result = null;
        UpdateWrapper<AssetLoanInfo> updateWrapper = new UpdateWrapper<>();
        updateWrapper.set("asset_audit_", status);
        updateWrapper.set("audit_msg_", msg);
        updateWrapper.in("id_", idList);
        result = this.baseMapper.update(null, updateWrapper);
        return null != result && result >= 1;
    }

    @Override
    public List<MonthVo> loanStatistics(String fullId) {
        Integer year = LocalDate.now().getYear();
        List<MonthVo> list = this.baseMapper.loanStatistics(fullId, year);

        Set<Integer> monthList = list.stream()
                .map(MonthVo::getMonth)
                .collect(Collectors.toSet());
        Set<Integer> noMonthList = Sets.newHashSet();
        for (int i = 1; i <= 12; i++) {
            if(!monthList.contains(i)) {
                noMonthList.add(i);
            }
        }
        noMonthList.forEach(month -> {
            MonthVo monthVo = new MonthVo();
            monthVo.setMonth(month);
            monthVo.setAddUpMoney(BigDecimal.ZERO);
            list.add(monthVo);
        });
        list.sort(Comparator.comparing(MonthVo::getMonth));
        return list;
    }
}
