package com.artfess.ljzc.business.manager.impl;

import com.alibaba.fastjson.JSONObject;
import com.artfess.base.enums.DelStatusEnum;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.query.PageBean;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.ljzc.business.dao.BizAssetBusinessInfoDao;
import com.artfess.ljzc.business.manager.BizAssetBusinessInfoManager;
import com.artfess.ljzc.business.manager.BizAssetBusinessUseManager;
import com.artfess.ljzc.business.manager.BizAssetBusinessWarrentsManager;
import com.artfess.ljzc.business.model.BizAssetBusinessInfo;
import com.artfess.ljzc.business.model.BizAssetBusinessUse;
import com.artfess.ljzc.business.model.BizAssetBusinessWarrents;
import com.artfess.ljzc.fixed.dao.BizAssetFixedInfoDao;
import com.artfess.ljzc.land.dao.BizAssetLandInfoDao;
import com.artfess.ljzc.loan.dao.AssetLoanInfoDao;
import com.artfess.ljzc.stock.dao.AssetStockInfoDao;
import com.artfess.ljzc.welfare.dao.AssetPubilcInfoDao;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.List;

/**
 * 经营性资产的基础信息 服务实现类
 *
 * @author min.wu
 * @company 阿特菲斯信息技术有限公司
 * @since 2023-11-27
 */
@Service
public class BizAssetBusinessInfoManagerImpl extends BaseManagerImpl<BizAssetBusinessInfoDao, BizAssetBusinessInfo> implements BizAssetBusinessInfoManager {

    @Autowired
    private BizAssetBusinessUseManager businessUseManager;

    @Autowired
    private BizAssetBusinessWarrentsManager businessWarrentsManager;

    @Resource
    private BizAssetLandInfoDao assetLandInfoDao;

    @Resource
    private BizAssetFixedInfoDao fixedInfoDao;

    @Resource
    private AssetPubilcInfoDao pubilcInfoDao;

    @Resource
    private AssetStockInfoDao stockInfoDao;

    @Resource
    private AssetLoanInfoDao loanInfoDao;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean audit(List<String> ids, String status, String assetMsg) {
        Assert.notEmpty(ids, "请选择要审核的资产");
        Assert.hasText(status, "请选择审核状态");
        List<BizAssetBusinessInfo> bizAssetBusinessInfos = listByIds(ids);
        Assert.notEmpty(bizAssetBusinessInfos, "资产信息不存在");
        bizAssetBusinessInfos.forEach(info -> {
            info.setAssetAudit(status);
            info.setAssetMsg(assetMsg);
        });
        boolean b = updateBatchById(bizAssetBusinessInfos);
        return b;
    }

    @Override
    public PageList<BizAssetBusinessInfo> findByPage(QueryFilter<BizAssetBusinessInfo> queryFilter) {
        queryFilter.addFilter("is_dele_", DelStatusEnum.N.getType(), QueryOP.EQUAL);
        PageBean pageBean = queryFilter.getPageBean();
        Class<BizAssetBusinessInfo> currentModelClass = currentModelClass();
        IPage<BizAssetBusinessInfo> result = baseMapper.findByPage(convert2IPage(pageBean), convert2Wrapper(queryFilter, currentModelClass));
        return new PageList<BizAssetBusinessInfo>(result);
    }

    @Override
    public JSONObject bookValue(String fullId) {
        JSONObject data = new JSONObject();
        //经营性资产账面价值
        BigDecimal businessBookValue = baseMapper.bookValue(fullId);
        data.put("businessBookValue", businessBookValue);
        //土地资产账面价值
        BigDecimal landBookValue = assetLandInfoDao.bookValue(fullId);
        data.put("landBookValue", landBookValue);

        //固定资产账面价值
        BigDecimal fixedBookValue = fixedInfoDao.bookValue(fullId);
        data.put("fixedBookValue", fixedBookValue.divide(new BigDecimal(10000), 2, BigDecimal.ROUND_HALF_UP));

        //公益性资产账面价值
        BigDecimal publicBookValue = pubilcInfoDao.bookValue(fullId);
        data.put("publicBookValue", publicBookValue);

        //股权资产账面价值
        BigDecimal stockBookValue = stockInfoDao.bookValue(fullId);
        data.put("stockBookValue", stockBookValue);

        //债权资产账面价值
        BigDecimal loanBookValue = loanInfoDao.bookValue(fullId);
        data.put("loanBookValue", loanBookValue);

        BigDecimal totalBookValue = businessBookValue.add(landBookValue)
                .add(fixedBookValue)
                .add(publicBookValue)
                .add(stockBookValue)
                .add(loanBookValue);
        data.put("totalBookValue", totalBookValue);
        return data;
    }

    @Override
    public JSONObject businessStatistics(String fullId) {
        List<JSONObject> list = this.baseMapper.businessStatistics(fullId);
        BigDecimal totalStructureArea = BigDecimal.ZERO;
        BigDecimal totalUnconfirmArea = BigDecimal.ZERO;
        for (JSONObject jsonObject : list) {
            if (jsonObject.containsKey("0")) {
                jsonObject.put("name", "物业资产");
            }
            if (jsonObject.containsKey("1")) {
                jsonObject.put("name", "商品房");
            }
            totalStructureArea = totalStructureArea.add(jsonObject.getBigDecimal("structureArea"));
            totalUnconfirmArea = totalUnconfirmArea.add(jsonObject.getBigDecimal("unconfirmArea"));
        }
        JSONObject object = new JSONObject();
        object.put("name", "建筑面积");
        object.put("structureArea", totalStructureArea);
        object.put("unconfirmArea", totalUnconfirmArea);
        list.add(object);
        List<JSONObject> assetStatusList = this.baseMapper.assetStatusList(fullId);
        JSONObject data = new JSONObject();
        data.put("areaInfo", list);
        data.put("bookValueInfo", assetStatusList);
        return data;
    }

    @Override
    public JSONObject gatherStatistics(String fullId) {
        return this.baseMapper.gatherStatistics(fullId);
    }

    @Override
    public List<JSONObject> sourceStatistics(String fullId) {
        return this.baseMapper.sourceStatistics(fullId);
    }

    @Override
    public List<JSONObject> statusStatistics(String fullId) {
        return this.baseMapper.statusStatistics(fullId);
    }

    @Override
    public List<JSONObject> operationStatistics(String fullId) {
        return this.baseMapper.operationStatistics(fullId);
    }

    @Override
    public List<JSONObject> typeStatistics(String fullId) {
        return this.baseMapper.typeStatistics(fullId);
    }

    @Override
    public List<JSONObject> belongStatistics(String fullId) {
        return this.baseMapper.belongStatistics(fullId);
    }

    @Override
    public List<JSONObject> natureStatistics(String fullId) {
        return this.baseMapper.natureStatistics(fullId);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean createInfo(BizAssetBusinessInfo t) {
        boolean save = this.save(t);
        if (save) {
            processInfo(t);
        }
        return save;
    }

    private void processInfo(BizAssetBusinessInfo t) {
        BizAssetBusinessWarrents assetBusinessWarrents = t.getAssetBusinessWarrents();
        assetBusinessWarrents.setAssetId(t.getId());
        assetBusinessWarrents.setProjectId(t.getProjectId());
        assetBusinessWarrents.setBuildingId(t.getBuildingId());
        businessWarrentsManager.saveOrUpdate(assetBusinessWarrents);

        BizAssetBusinessUse assetBusinessUse = t.getAssetBusinessUse();
        assetBusinessUse.setAssetId(t.getId());
        assetBusinessUse.setProjectId(t.getProjectId());
        assetBusinessUse.setBuildingId(t.getBuildingId());
        businessUseManager.saveOrUpdate(assetBusinessUse);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean updateInfo(BizAssetBusinessInfo t) {
        boolean b = this.updateById(t);
        if (b) {
            processInfo(t);
        }
        return b;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean removeInfo(List<String> ids) {
        boolean b = this.removeByIds(ids);
        if (b) {
            QueryWrapper<BizAssetBusinessWarrents> warrentsQueryWrapper = new QueryWrapper<>();
            warrentsQueryWrapper.in("asset_id_", ids);
            businessWarrentsManager.remove(warrentsQueryWrapper);

            QueryWrapper<BizAssetBusinessUse> useQueryWrapper = new QueryWrapper<>();
            useQueryWrapper.in("asset_id_", ids);
            businessUseManager.remove(useQueryWrapper);
        }
        return b;
    }

    @Override
    public BizAssetBusinessInfo findById(String id) {
        Assert.hasText(id, "请选择要查看的资产");
        BizAssetBusinessInfo businessInfo = this.get(id);
        QueryWrapper<BizAssetBusinessWarrents> warrentsQueryWrapper = new QueryWrapper<>();
        warrentsQueryWrapper.eq("asset_id_", id);
        BizAssetBusinessWarrents warrents = businessWarrentsManager.getOne(warrentsQueryWrapper);
        businessInfo.setAssetBusinessWarrents(warrents);
        QueryWrapper<BizAssetBusinessUse> useQueryWrapper = new QueryWrapper<>();
        useQueryWrapper.eq("asset_id_", id);
        BizAssetBusinessUse bizAssetBusinessUse = businessUseManager.getOne(useQueryWrapper);
        businessInfo.setAssetBusinessUse(bizAssetBusinessUse);
        return businessInfo;
    }
}
