package com.artfess.cgpt.contract.controller;


import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.exception.BaseException;
import com.artfess.base.feign.UCFeignService;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.FieldRelation;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.util.BeanUtils;
import com.artfess.cgpt.contract.manager.ContractCallBackManager;
import com.artfess.cgpt.contract.vo.ContractVo;
import com.artfess.cgpt.purchasing.model.BizPurchaseOrder;
import com.artfess.cgpt.utils.BizUtils;
import com.artfess.uc.api.impl.util.ContextUtil;
import io.swagger.annotations.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import com.artfess.base.controller.BaseController;
import com.artfess.cgpt.contract.model.BizContract;
import com.artfess.cgpt.contract.manager.BizContractManager;

import javax.servlet.http.HttpServletResponse;
import java.util.ArrayList;
import java.util.List;

/**
 * 合同表 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author 管理员
 * @since 2024-03-27
 */

@RestController
@RequestMapping("/bizContract/v1/")
@Api(tags = "采购合同表")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class BizContractController extends BaseController<BizContractManager, BizContract> {

    @Autowired
    private UCFeignService ucFeignService;
    @Autowired
    private ContractCallBackManager contractCallBackManager;

    @PostMapping("/pageAll")
    @ApiOperation(value = "M-分页查询所有数据")
    public CommonResult<PageList<BizContract>> queryAllByPage(@ApiParam(name="queryfilter",value = "通用查询器")@RequestBody QueryFilter<BizContract> queryFilter){
        PageList<BizContract> result = baseService.queryAllByPage(queryFilter);
        return new CommonResult<>(true,"查询成功",result);
    }

    @PostMapping("/pageVendorGroup")
    @ApiOperation(value = "M-分页查询供应商分组")
    public CommonResult<PageList<BizContract>> pageVendorGroup(@ApiParam(name="queryfilter",value = "通用查询器")@RequestBody QueryFilter<BizContract> queryFilter){
        PageList<BizContract> result = baseService.pageVendorGroup(queryFilter);
        return new CommonResult<>(true,"查询成功",result);
    }

    @PostMapping("/pageByOrg")
    @ApiOperation(value = "M-分页查询当前企业相关数据")
    public CommonResult<PageList<BizContract>> queryByOrg(@ApiParam(name="queryfilter",value = "通用查询器")@RequestBody QueryFilter<BizContract> queryFilter){
        String orgId = ContextUtil.getCurrentOrgId();
        if(BeanUtils.isEmpty(orgId)){
            return new CommonResult<>(true,"查询成功",new PageList<>(new ArrayList<>()));
        }

        queryFilter.addFilter("MAT_COMPANY_ID_",orgId,QueryOP.EQUAL, FieldRelation.OR,"orgQuery");
        queryFilter.addFilter("VENDOR_ID_",orgId,QueryOP.EQUAL, FieldRelation.OR,"orgQuery");

        PageList<BizContract> result = baseService.queryAllByPage(queryFilter);
        return new CommonResult<>(true,"查询成功",result);
    }

    @PostMapping("/saveOrUpdateContr")
    @ApiOperation(value = "M-保存或更新合同")
    public CommonResult saveOrUpdateEntity(@RequestBody BizContract entity) {
        String id = baseService.saveOrUpdateEntity(entity);
        return new CommonResult<>(true, "操作成功",id);
    }

    @PostMapping("/getById")
    @ApiOperation(value = "M-根据id查询")
    public CommonResult<BizContract> getDataById(@RequestParam("id") String id){
        if(BeanUtils.isEmpty(id)){
            return new CommonResult<>(false,"请选择数据");
        }
        BizContract result = baseService.getDetailById(id);
        return new CommonResult<>(true,"查询成功",result);
    }

    @PostMapping("/removeByIds")
    @ApiOperation(value = "M-根据id批量删除")
    public CommonResult remove(@RequestBody List<String> ids){
        if(BeanUtils.isEmpty(ids) || ids.size()<=0){
            return new CommonResult<>(false,"请选择数据");
        }
        baseService.removeData(ids);
        return new CommonResult<>(true,"删除成功");
    }

    @PostMapping("/updateStatus")
    @ApiOperation(value = "M-根据id修改状态")
    public CommonResult updateStatus(@RequestParam("ids") String ids,@RequestParam("status") String status){
        if(BeanUtils.isEmpty(ids)){
            return new CommonResult<>(false,"请选择数据");
        }
        if(BeanUtils.isEmpty(status)){
            return new CommonResult<>(false,"请传入状态");
        }
        baseService.updateStatusByIds(ids,status);
        return new CommonResult<>(true,"操作成功",ids);
    }

    /**
     * 浏览合同
     * @param threeContrId 第三方合同id
     * @return 网页url
     */
    @PostMapping("/browseContract")
    @ApiOperation(value = "M-浏览合同")
    public CommonResult browseContract(@ApiParam(name = "threeContrId",value = "第三方合同id") @RequestParam("threeContrId") String threeContrId){
        if(BeanUtils.isEmpty(threeContrId)){
            return new CommonResult<>(false,"请选择数据");
        }
        String result = baseService.browseContract(threeContrId);
        return new CommonResult(result);
    }

    /**
     * 批量下载合同文件
     * @param threeContrIds 第三方合同id集合
     * @return 网页url
     */
    @PostMapping("/downBatchContract")
    @ApiOperation(value = "M-批量下载合同")
    public CommonResult downBatchContract(@ApiParam(name = "threeContrIds",value = "第三方合同id集合") @RequestBody List<Long> threeContrIds, HttpServletResponse response){
        if(BeanUtils.isEmpty(threeContrIds) || threeContrIds.size()<=0){
            return new CommonResult<>(false,"请选择数据");
        }

        baseService.downBatchContract(threeContrIds,response);
        return new CommonResult("下载成功");
    }

    /**
     * 生成电子合同草稿
     * @param id 本地合同表id
     * @return 合同id
     */
    @PostMapping("/createContract")
    @ApiOperation(value = "M-生成电子合同草稿")
    public CommonResult createContract(@ApiParam(name = "id",value = "本地合同表id") @RequestParam("id") String id) throws Exception {
        if(BeanUtils.isEmpty(id)){
            return new CommonResult<>(false,"请选择数据");
        }

        String result = baseService.createContract(id);
        return new CommonResult(result);
    }

    /**
     * 发起电子合同签订
     * @param threeContrId 第三方合同id
     * @return
     */
    @PostMapping("/sendContract")
    @ApiOperation(value = "M-发起电子合同签订")
    public CommonResult sendContract(@ApiParam(name = "threeContrId",value = "第三方合同id") @RequestParam("threeContrId") String threeContrId) throws Exception {
        if(BeanUtils.isEmpty(threeContrId)){
            return new CommonResult<>(false,"请选择数据");
        }
        baseService.sendContract(threeContrId);
        return new CommonResult(true,"操作成功");
    }


    @ApiImplicitParams({
            @ApiImplicitParam(name = "contractId", value = "文件编号", dataType = "String"),
            @ApiImplicitParam(name = "sn", value = "文件自定义编号", dataType = "String"),
            @ApiImplicitParam(name = "status", value = "文件状态", dataType = "String"),
            @ApiImplicitParam(name = "type", value = "回调类型", dataType = "String"),
            @ApiImplicitParam(name = "contact", value = "联系方式", dataType = "String"),
            @ApiImplicitParam(name = "operatorName", value = "操作人姓名", dataType = "String"),
            @ApiImplicitParam(name = "operatorMobile", value = "操作人手机号", dataType = "String"),
            @ApiImplicitParam(name = "operatorNumber", value = "操作人员工编号", dataType = "String"),
            @ApiImplicitParam(name = "receiverNumber", value = "经办人员工编号", dataType = "String"),
            @ApiImplicitParam(name = "accessKey", value = "ak", dataType = "String"),
            @ApiImplicitParam(name = "secretKey", value = "sk", dataType = "String"),
    })
    @PostMapping("/signingCompleted")
    @ApiOperation(value = "M-签署完成（合同系统回调接口）")
    public void signingCompleted(String contractId,
                                 String sn, String status,String type, String contact,
                                 String operatorName, String operatorMobile,
                                 String operatorNumber,String receiverNumber,
                                 String accessKey, String secretKey) {
        contractCallBackManager.signingCompleted(contractId,sn,status,type,
                contact,operatorName,operatorMobile,operatorNumber,receiverNumber,
                accessKey,secretKey,1) ;
    }

}
