package com.artfess.cgpt.bidding.manager.impl;

import com.artfess.base.exception.BaseException;
import com.artfess.base.feign.SystemConfigFeignService;
import com.artfess.base.manager.BaseManager;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.util.BeanUtils;
import com.artfess.cgpt.bidding.dao.QuotationTemplateDao;
import com.artfess.cgpt.bidding.manager.QuotationTemplateDetailManager;
import com.artfess.cgpt.bidding.manager.QuotationTemplateManager;
import com.artfess.cgpt.bidding.model.QuotationTemplate;
import com.artfess.cgpt.bidding.model.QuotationTemplateDetail;
import com.artfess.cgpt.expert.model.BizExpertDatabaseQualificationLabel;
import com.artfess.cgpt.project.manager.ProjectApprovalManager;
import com.artfess.cgpt.project.model.ProjectApproval;
import com.artfess.cgpt.project.model.UserParticipationQualificationLabel;
import com.artfess.cgpt.purchasing.manager.MatApprovalManager;
import com.artfess.cgpt.purchasing.model.MatApproval;
import com.artfess.cgpt.qualification.model.BizQualificationLabel;
import com.artfess.cgpt.supplier.model.BizSupplierOrgQualificationLabel;
import com.artfess.cgpt.utils.BizUtils;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

/**
 * @Description:
 * @Author: Rong Tao
 * @Date: 2024/3/15 18:15
 */
@Service
public class QuotationTemplateManagerImpl extends BaseManagerImpl<QuotationTemplateDao, QuotationTemplate> implements QuotationTemplateManager {

    @Autowired
    SystemConfigFeignService systemConfigFeignService;
    @Autowired
    private QuotationTemplateDetailManager quotationTemplateDetailManager;
    @Autowired
    private MatApprovalManager matApprovalManager;
    @Autowired
    private ProjectApprovalManager projectApprovalManager;

    @Override
    public PageList<QuotationTemplate> queryAllByPage(QueryFilter<QuotationTemplate> queryFilter) {
        queryFilter.addFilter("IS_DELE_","1", QueryOP.NOT_EQUAL);
        IPage<QuotationTemplate> result = baseMapper.queryAllByPage(convert2IPage(queryFilter.getPageBean()),
                convert2Wrapper(queryFilter, currentModelClass()));

        List<QuotationTemplate> records = result.getRecords();

        if(BeanUtils.isNotEmpty(records) && records.size()>0){
            for(QuotationTemplate data : records){
                //判断是否存在关联数据
                data.setDataQuoteStatus(existDataStatus(data));
            }
        }

        return new PageList<>(result);
    }

    @Override
    @Transactional
    public void saveOrUpdateData(QuotationTemplate data) {
        List<QuotationTemplateDetail> detailList = data.getDetailList();
        if(BeanUtils.isEmpty(data.getId())) {
            //如果是新增，设置编号
            data.setCode(BizUtils.getNextSerialNumber(systemConfigFeignService,"bjmb_bm"));
        }else {
            //更新
            //判断是否存在关联数据，如果存在，不能编辑
            existData(data);

            //清除原有明细
            LambdaQueryWrapper<QuotationTemplateDetail> detailQW = new LambdaQueryWrapper<>();
            detailQW.eq(QuotationTemplateDetail::getTemplateId,data.getId())
                    .eq(QuotationTemplateDetail::getIsDele,"0");
            quotationTemplateDetailManager.remove(detailQW);
        }
        super.saveOrUpdate(data);


        if(BeanUtils.isNotEmpty(detailList) && detailList.size()>0){
            detailList.stream().forEach(s->{
                s.setTemplateId(data.getId());
            });
        }
        quotationTemplateDetailManager.saveOrUpdateBatch(detailList);
    }

    @Override
    public QuotationTemplate dataById(String id) {
        QuotationTemplate result = super.getById(id);
        LambdaQueryWrapper<QuotationTemplateDetail> detailQW = new LambdaQueryWrapper<>();
        detailQW.eq(QuotationTemplateDetail::getTemplateId,id)
                .ne(QuotationTemplateDetail::getIsDele,"1");
        List<QuotationTemplateDetail> detailList = quotationTemplateDetailManager.list(detailQW);
        if(BeanUtils.isNotEmpty(detailList)) result.setDetailList(detailList);
        return result;
    }

    @Override
    public void removeByIds(List<String> ids) {
        List<QuotationTemplate> quotationTemplates = super.listByIds(ids);
        for(QuotationTemplate data : quotationTemplates){
            //判断是否存在关联数据，如果存在，不能删除
            existData(data);
        }
        super.removeByIds(ids);
    }

    /**
     * 校验是否已存在关联数据
     * @param data
     */
    void existData(QuotationTemplate data){
        //查询物资立项表是否有使用该报价模版
        LambdaQueryWrapper<MatApproval> matApprovalQW = new LambdaQueryWrapper<>();
        matApprovalQW.eq(MatApproval::getQuotationTemplate,data.getId())
                .eq(MatApproval::getIsDele,"0");
        if(matApprovalManager.count(matApprovalQW)>0){
            throw new BaseException("模版编码【"+data.getCode()+"】已存在关联数据，无法操作");
        }

        //查询项目立项表是否有使用该报价模版
        LambdaQueryWrapper<ProjectApproval> projectApprovalQW = new LambdaQueryWrapper<>();
        projectApprovalQW.eq(ProjectApproval::getQuotationTemplate,data.getId())
                .eq(ProjectApproval::getIsDele,"0");
        if(projectApprovalManager.count(projectApprovalQW)>0){
            throw new BaseException("模版编码【"+data.getCode()+"】已存在关联数据，无法操作");
        }

    }

    /**
     * 校验是否已存在关联数据
     * @param data
     */
    Integer existDataStatus(QuotationTemplate data){
        //查询物资立项表是否有使用该报价模版
        LambdaQueryWrapper<MatApproval> matApprovalQW = new LambdaQueryWrapper<>();
        matApprovalQW.eq(MatApproval::getQuotationTemplate,data.getId())
                .eq(MatApproval::getIsDele,"0");
        if(matApprovalManager.count(matApprovalQW)>0){
            return 1;
        }

        //查询项目立项表是否有使用该报价模版
        LambdaQueryWrapper<ProjectApproval> projectApprovalQW = new LambdaQueryWrapper<>();
        projectApprovalQW.eq(ProjectApproval::getQuotationTemplate,data.getId())
                .eq(ProjectApproval::getIsDele,"0");
        if(projectApprovalManager.count(projectApprovalQW)>0){
            return 1;
        }

        return 0;
    }

}
