package com.artfess.portal.kh.controller;


import com.alibaba.fastjson.JSON;
import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.annotation.PowerLogInfo;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.context.BaseContext;
import com.artfess.base.controller.BaseController;
import com.artfess.base.enums.LogType;
import com.artfess.base.enums.OperationType;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.valid.AddGroup;
import com.artfess.base.valid.UpdateGroup;
import com.artfess.i18n.util.I18nUtil;
import com.artfess.portal.kh.manager.KhSubjectManager;
import com.artfess.portal.kh.model.KhSubject;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.formula.functions.T;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

/**
 * 考核科目信息 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author min.wu
 * @since 2024-02-20
 */
@Slf4j
@RestController
@Api(tags = "绩效考核-考核科目信息")
@RequestMapping("/kh/subject/")
@ApiGroup(group = {ApiGroupConsts.GROUP_SYSTEM})
public class KhSubjectController extends BaseController<KhSubjectManager, KhSubject> {

    @Autowired
    private BaseContext baseContext;

    @Override
    @PostMapping("/")
    @ApiOperation("添加实体的接口")
    public CommonResult<String> create(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({AddGroup.class}) KhSubject t) {
        String id = baseService.insertTree(t);
        if(t.getSn() == null){
            t.setSn(baseService.getNextSequence(null));
        }
        baseService.updateById(t);
        if(!StringUtils.isNotBlank(id)) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
        return new CommonResult<String>();
    }

    @Override
    @PutMapping("/")
    @ApiOperation("更新实体")
    public CommonResult<String> updateById(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({UpdateGroup.class}) KhSubject t) {
        KhSubject byId = baseService.getById(t.getId());
        t.setCreateCompanyId(baseContext.getCurrentDeptId());
        String id = baseService.updateTree(t, byId.getName());
        if(t.getSn() == null){
            t.setSn(baseService.getNextSequence(null));
        }
        baseService.updateById(t);
        if(!StringUtils.isNotBlank(id)) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
        return new CommonResult<String>();
    }

    @Override
    @DeleteMapping("/{id}")
    @ApiOperation("根据id删除")
    public CommonResult<String> deleteById(@ApiParam(name="id", value="实体id") @PathVariable String id) {
        boolean result = baseService.removeById(id);
        if(!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
        return new CommonResult<String>();
    }

    @Override
    @DeleteMapping("/")
    @ApiOperation("根据id集合批量删除")
    public CommonResult<String> deleteByIds(@ApiParam(name="ids", value="实体集合") @RequestParam String...ids) {
        boolean result = baseService.removeByIds(Arrays.asList(ids));
        if(!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("delete.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }

    @PostMapping("/getTree")
    @ApiOperation("获取科目信息下拉树")
    @PowerLogInfo(logType = LogType.BIZ, operaionType = OperationType.QUERY, description = "获取科目信息下拉树")
    public CommonResult<String> getTree(@ApiParam(name = "model", value = "获取科目信息下拉树") @RequestBody KhSubject entity) {
        log.info("获取科目信息下拉树请求参数:{}", JSON.toJSONString(entity));
        List<KhSubject> list = baseService.getTree(entity);
        return CommonResult.success(list, null);
    }

    @RequestMapping(value = "updateSequence", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "批量修改排序号", notes = "批量修改排序号")
    public CommonResult<String> updateSequence(@ApiParam(name = "params", value = "排序参数：Key：ID，Value：排序号") @RequestBody HashMap<String, Integer> params) throws Exception {
        if(!params.isEmpty()){
            baseService.updateSequence(params);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }

    // List<String> currentAndChildOrgIds = baseContext.getCurrentAndChildOrgIds();
    //        queryFilter.addFilter("CREATE_ORG_ID_", currentAndChildOrgIds ,QueryOP.IN);

    @Override
    @PostMapping(value = "/query", produces = {"application/json; charset=utf-8"})
    @ApiOperation("分页查询结果")
    public PageList<KhSubject> query(@ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<KhSubject> queryFilter) {
        List<String> currentAndChildOrgIds = baseContext.getCurrentAndChildOrgIds();
        queryFilter.addFilter("create_company_id_", currentAndChildOrgIds , QueryOP.IN);
        return baseService.query(queryFilter);
    }
}
