package com.artfess.dataShare.dataShare.manager.impl;

import com.alibaba.fastjson.JSONArray;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.exception.ApplicationException;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.StringUtil;
import com.artfess.dataShare.dataApi.model.BizServiceApi;
import com.artfess.dataShare.dataApi.vo.RespParamVo;
import com.artfess.dataShare.dataShare.manager.BizShareTableManager;
import com.artfess.dataShare.dataShare.model.BizShareApi;
import com.artfess.dataShare.dataShare.dao.BizShareApiDao;
import com.artfess.dataShare.dataShare.manager.BizShareApiManager;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.dataShare.dataShare.model.BizShareTable;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 数据资源-- 接口共享信息 服务实现类
 *
 * @company 阿特菲斯信息技术有限公司
 * @author wubin
 * @since 2024-12-03
 */
@Service
public class BizShareApiManagerImpl extends BaseManagerImpl<BizShareApiDao, BizShareApi> implements BizShareApiManager {
    @Resource
    private RestTemplate restTemplate;

    @Resource
    private BizShareTableManager tableManager;

    @Value("${dataShare-api.url.shareApi.register}")
    private String SHAREAPI_REGISTER_URL;

    @Value("${dataShare-api.url.shareApi.checkSqlValidity}")
    private String SHAREAPI_CHECKSQLVALIDITY_URL;

    @Value("${dataShare-api.url.shareApi.unregister}")
    private String SHAREAPI_UNREGISTER_URL;

    @Value("${dataShare-api.url.shareApi.updateServiceApi}")
    private String SHAREAPI_UPDATESERVICEAPI_URL;

    @Value("${dataShare-api.url.shareApi.query}")
    private String SHAREAPI_QUERY_URL;

    @Value("${dataShare-api.url.shareApi.getInfo}")
    private String SHAREAPI_GETINFO_URL;

    @Value("${dataShare-api.url.shareApi.checkApi}")
    private String SHAREAPI_CHECKAPI_URL;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean register(BizShareApi shareApi) {
        shareApi.setApiUrl("/shareApi"+shareApi.getApiUrl());
        shareApi.setIsDele("0");
        shareApi.setStatus("0");
        boolean status = this.registerToDB(shareApi);
        return status;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean publish(String id) {
        BizShareApi shareApi = this.get(id);
        int count = 0;
        if (BeanUtils.isNotEmpty(shareApi)){
            shareApi.setStatus("1");
            count = baseMapper.updateById(shareApi);
        }
        CommonResult commonResult = new CommonResult<>(ResponseErrorEnums.FAIL_OPTION);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<BizShareApi> request = new HttpEntity<>(shareApi,headers);
        //在容器中注册接口
        commonResult = restTemplate.postForObject(SHAREAPI_REGISTER_URL, request, CommonResult.class);
        return commonResult.getState();
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean soldOut(String id) {
        boolean status = false;
        BizShareApi shareApi = this.get(id);
        shareApi.setStatus("0");
        int count = 0;
        if (BeanUtils.isNotEmpty(shareApi)){
            count = baseMapper.updateById(shareApi);
        }
        CommonResult commonResult = new CommonResult<>(ResponseErrorEnums.FAIL_OPTION);
        if (count > 0){
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
            HttpEntity<BizShareApi> request = new HttpEntity<>(shareApi,headers);
            commonResult = restTemplate.postForObject(SHAREAPI_UNREGISTER_URL, request, CommonResult.class);
        }
        if(!commonResult.getState()){
            status = false;
            throw new ApplicationException(commonResult.getMessage());
        }
        status = commonResult.getState();
        return status;
    }

    @Override
    public boolean checkApiValidity(BizShareApi shareApi) {
        return checkApiInDB(shareApi) && checkApiInSpring(shareApi);
    }

    private boolean checkApiInSpring(BizShareApi shareApi) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<BizShareApi> request = new HttpEntity<>(shareApi,headers);
        CommonResult<String> commonResult = restTemplate.postForObject(SHAREAPI_CHECKAPI_URL, request, CommonResult.class);
        return commonResult.getState();
    }

    public boolean checkApiInDB(BizShareApi serviceApi){
        LambdaQueryWrapper<BizShareApi> queryWrapper = new LambdaQueryWrapper<BizShareApi>();
        queryWrapper.eq(BizShareApi::getApiUrl,serviceApi.getApiUrl());
        queryWrapper.eq(BizShareApi::getApiMode,serviceApi.getApiMode());
        return this.count(queryWrapper) == 0 ? true : false;
    }

    @Override
    public boolean checkSqlValidity(BizShareApi shareApi) throws SQLException {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<BizShareApi> request = new HttpEntity<>(shareApi,headers);
        CommonResult commonResult = restTemplate.postForObject(SHAREAPI_CHECKSQLVALIDITY_URL, request, CommonResult.class);
        return commonResult.getState();
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean unregister(String id) {
        BizShareApi shareApi = this.get(id);
        boolean status = this.unregisterFromDB(shareApi.getId());
        CommonResult commonResult = new CommonResult<>(ResponseErrorEnums.FAIL_OPTION);

        if (status){
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
            HttpEntity<BizShareApi> request = new HttpEntity<>(shareApi,headers);
            commonResult = restTemplate.postForObject(SHAREAPI_UNREGISTER_URL, request, CommonResult.class);
        }

        if(!commonResult.getState()){
            status = false;
            throw new ApplicationException();
        }
        status = commonResult.getState();
        return status;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean updateServiceApi(BizShareApi shareApi) {
        CommonResult commonResult = new CommonResult<>(ResponseErrorEnums.FAIL_OPTION);
        BizShareApi oldShareApi = this.get(shareApi.getId());
        if(!oldShareApi.getApiMode().name().equals(shareApi.getApiMode().name())
                || !oldShareApi.getApiUrl().equals(shareApi.getApiUrl())){
            if (!checkApiValidity(shareApi)){
                throw new ApplicationException("接口已被注册！");
            }
        }
        if(!checkApiCode(shareApi)){
            throw new ApplicationException("接口编码重复！");
        }
        int count = baseMapper.updateById(shareApi);
        return count > 0 ? true : false;
    }

    @Override
    public PageList<BizShareApi> queryApi(QueryFilter<BizShareApi> queryFilter) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<QueryFilter<BizShareApi>> request = new HttpEntity<>(queryFilter,headers);
        PageList pageList = restTemplate.postForObject(SHAREAPI_QUERY_URL, request, PageList.class);
        return pageList;
    }

    @Override
    public BizShareApi getInfo(String id) {
        Map<String,String> param = new HashMap<>();
        param.put("id",id);
        BizShareApi shareApi = restTemplate.getForObject(SHAREAPI_GETINFO_URL, BizShareApi.class, param);
        return shareApi;
    }

    @Override
    public BizShareApi viewApiDocument(String id) {
        Map<String,String> param = new HashMap<>();
        param.put("id",id);
        BizShareApi shareApi = restTemplate.getForObject(SHAREAPI_GETINFO_URL, BizShareApi.class, param);
        String responseExample = shareApi.getResponseExample();
        if (StringUtil.isNotEmpty(responseExample)){
            JSONArray jsonArray = JSONArray.parseArray(responseExample);
            Integer size = jsonArray.size();

            RespParamVo respParamVo = new RespParamVo();
            respParamVo.setResultCode("200");
            respParamVo.setMessage("成功");
            respParamVo.setTotalCount(size.toString());
            respParamVo.setCount(size.toString());
            respParamVo.setRows(jsonArray.toJavaList(Object.class));
            shareApi.setRespParamVo(respParamVo);
        }
        return shareApi;
    }

    @Override
    public List<BizShareApi> queryByTableId(String tableId) {
        List<BizShareApi> list = list(new LambdaQueryWrapper<BizShareApi>()
                .eq(BizShareApi::getIsDele, "0")
                .eq(BizShareApi::getTableId, tableId));
        return list;
    }

    @Override
    public PageList<BizShareApi> queryPageByCatalog(QueryFilter<BizShareApi> queryFilter,String catalogId,String consumerId) {
        IPage<BizShareApi> iPage = baseMapper.queryPageByCatalog(convert2IPage(queryFilter.getPageBean()),
                convert2Wrapper(queryFilter, currentModelClass()),catalogId,consumerId);
        return new PageList<>(iPage);
    }

    @Override
    public boolean checkApiCode(BizShareApi shareApi) {
        LambdaQueryWrapper<BizShareApi> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(BizShareApi::getApiCode,shareApi.getApiCode());
        queryWrapper.and(w -> {
            w.eq(BizShareApi::getApiCode,shareApi.getApiCode());
        });
        if (BeanUtils.isNotEmpty(shareApi.getId())){
            queryWrapper.ne(BizShareApi::getId,shareApi.getId());
        }
        int count = count(queryWrapper);
        return count > 0 ? false : true;
    }

    /**
     * 从数据库中删除接口信息
     * @param id
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public boolean unregisterFromDB(String id){
        return this.removeById(id);
    }

    /**
     * 持久化接口信息
     * @param shareApi
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public boolean registerToDB(BizShareApi shareApi) {
        if(!checkApiValidity(shareApi)){
            throw new ApplicationException("接口已被注册！");
        }
        if(!checkApiCode(shareApi)){
            throw new ApplicationException("接口编码重复！");
        }
        return this.save(shareApi);
    }
}
