package com.artfess.cqxy.projectApproval.manager.impl;

import cn.afterturn.easypoi.excel.ExcelExportUtil;
import cn.afterturn.easypoi.excel.ExcelImportUtil;
import cn.afterturn.easypoi.excel.entity.ExportParams;
import cn.afterturn.easypoi.excel.entity.ImportParams;
import com.artfess.base.context.BaseContext;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.query.PageBean;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.BeanUtils;
import com.artfess.cqxy.projectApproval.dao.ProjectEstablishmentDao;
import com.artfess.cqxy.projectApproval.manager.ProjectEstablishmentManager;
import com.artfess.cqxy.projectApproval.model.ProjectEstablishment;
import com.artfess.cqxy.projectManagement.enums.ProjectStatusEnum;
import com.artfess.cqxy.projectManagement.manager.ProjectManagementManager;
import com.artfess.cqxy.projectManagement.model.ProjectManagement;
import com.artfess.cqxy.search.enums.FunctionEnum;
import com.artfess.cqxy.search.manager.GlobalRetrievalManager;
import com.artfess.cqxy.search.model.GlobalRetrieval;
import com.artfess.cqxy.statistics.vo.StatisticsVo;
import com.artfess.cqxy.universal.manager.AccessoryManager;
import com.artfess.cqxy.universal.model.Accessory;
import com.artfess.cqxy.utils.BizUtils;
import com.artfess.poi.util.ExcelUtil;
import com.artfess.sysConfig.persistence.manager.SysDictionaryManager;
import com.artfess.sysConfig.persistence.param.DictModel;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.Workbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStream;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * 项目立项表(BizProjectEstablishment)表服务实现类
 * @author 黎沐华
 * @since 2022-02-14 15:30:50
 */
@Service
public class ProjectEstablishmentManagerImpl extends BaseManagerImpl<ProjectEstablishmentDao, ProjectEstablishment> implements ProjectEstablishmentManager {

    @Autowired
    private AccessoryManager accessoryManager;

    @Autowired
    private ProjectManagementManager pmm;

    @Autowired
    private BaseContext baseContext;

    @Autowired
    private GlobalRetrievalManager grm;

    @Autowired
    private SysDictionaryManager sdm;

    @Override
    public boolean saveOrUpdate(ProjectEstablishment entity) {
        boolean save = StringUtils.isEmpty(entity.getId());
        boolean savedMain = super.saveOrUpdate(entity);
        // 处理附件信息
        List<Accessory> flag = entity.getAccessoryInfo();//处理空指针
        List<Accessory> accessoryList = null == flag ? new ArrayList<>():flag;
        for (Accessory ele : accessoryList) {
            ele.setSourceId(entity.getId());
            ele.setProjectId(entity.getProjectId());
            ele.setDirectory(ProjectStatusEnum.two.getCode());
            ele.setGroup("ProjectEstablishment");
            ele.setNode(ProjectStatusEnum.two.getCode());
            ele.setCreateBy(baseContext.getCurrentUserId());
            ele.setCreateName(baseContext.getCurrentUserName());
            ele.setCreateTime(LocalDateTime.now());
        }
        // 先清空该ID下的所有附件信息再添加
        accessoryManager.removeBySourceId(entity.getId());
        // 如果没有附件则返回true，不进入添加
        boolean saveAcc = accessoryList.size() == 0 || accessoryManager.saveAccess(accessoryList);
        //更新项目状态
        pmm.updateStatusById(entity.getProjectId(), Integer.valueOf(ProjectStatusEnum.two.getCode()));
        // 同步到检索表
        GlobalRetrieval globalRetrieval= grm.getByBizId(entity.getId());
        handleRetrieval(save || BeanUtils.isEmpty(globalRetrieval) ?new GlobalRetrieval():globalRetrieval,entity);

        return savedMain && saveAcc;
    }

    private void handleRetrieval(GlobalRetrieval globalRetrieval, ProjectEstablishment entity){
        ProjectManagement projectManagement = pmm.getById(entity.getProjectId());
        globalRetrieval.setProjectId(entity.getProjectId());
        globalRetrieval.setProjectName(projectManagement.getProjectName());
        globalRetrieval.setPersonCharge(projectManagement.getProjectManager());
        globalRetrieval.setArchivesType(1);
        globalRetrieval.setBizDataId(entity.getId());
        globalRetrieval.setFunctionCode(FunctionEnum.three.getCode());
        globalRetrieval.setFunctionName(FunctionEnum.three.getName());
        globalRetrieval.setBizTableName(FunctionEnum.three.getTableName());
        globalRetrieval.setDetailsRoteUrl(FunctionEnum.three.getTableRoteUrl());
        globalRetrieval.setTableRoteUrl(FunctionEnum.three.getTableRoteUrl());
        globalRetrieval.setTableApiUrl(FunctionEnum.three.getTableApiUrl());
        globalRetrieval.setDetailsApiUrl(FunctionEnum.three.getDetailsApiUrl());
        globalRetrieval.setFunctionPath(FunctionEnum.three.getFunctionPath());
        globalRetrieval.setDocumentNumber(entity.getDocumentNumber());
        globalRetrieval.setName(entity.getEstablishmentName());
        globalRetrieval.setDocumentUnit(entity.getEstablishmentApprovalUnit());
        globalRetrieval.setSearchTitle(entity.getEstablishmentName()+"_"+entity.getDocumentNumber()+"_"+entity.getEstablishmentApprovalUnit()+"_"+
                entity.getEstablishmentScale()+"_"+entity.getEstablishmentRemarks());
        grm.saveOrUpdate(globalRetrieval);
    }

    @Override
    public boolean deleteByIds(List<String> ids) {
        // 同步删除检索表中的信息
        for(String ele:ids){
            grm.remove(1,ele);
        }
        return removeByIds(ids);
    }

    @Override
    public ProjectEstablishment getById(String id) {
        ProjectEstablishment result = baseMapper.getById(id);
        result.setAccessoryInfo(accessoryManager.getAccessoryBySourceId(id));
        return result;
    }

    @Override
    public PageList<ProjectEstablishment> queryAllByPage(QueryFilter<ProjectEstablishment> queryFilter) {
        BizUtils.handleFilter(queryFilter,"bpe" ,"pm");
        IPage<ProjectEstablishment> result =
                baseMapper.queryAllByPage(
                        convert2IPage(queryFilter.getPageBean()),
                        convert2Wrapper(queryFilter, currentModelClass()));
        // 附件信息处理
        List<ProjectEstablishment> records = result.getRecords();
        for (ProjectEstablishment ele: records ) {
            ele.setAccessoryInfo(accessoryManager.getAccessoryBySourceId(ele.getId()));
        }

        return new PageList<>(result);
    }

    @Override
    public void importExcelData(MultipartFile file, String projectId) {
        Assert.notNull(projectId,"项目ID不能为空");
//        List<DictModel> establishmentName = sdm.queryDictListItemsByCode("xmgl-jclx-jsxz");
//        List<DictModel> establishmentType = sdm.queryDictListItemsByCode("lxba-wjlx");
        List<DictModel> unit = sdm.queryDictListItemsByCode("xmgl-lxspjg");
        ProjectManagement project = pmm.get(projectId);
        try(InputStream inputStream = file.getInputStream();) {
            // 获取数据
            List<ProjectEstablishment> data = ExcelImportUtil.importExcel(inputStream,ProjectEstablishment.class,new ImportParams());
            for(ProjectEstablishment ele:data){
                ele.setProjectId(projectId);
//                ele.setEstablishmentName(BizUtils.getDicCodeByValue(establishmentName, ele.getEstablishmentName()));
//                ele.setEstablishmentType(BizUtils.getDicCodeByValue(establishmentType, ele.getEstablishmentType()));
                ele.setEstablishmentApprovalUnit(BizUtils.getDicCodeByValue(unit, ele.getEstablishmentApprovalUnit()));
                ele.setPcode(project.getProjectCode());
                // 保存数据 & 同步到检索表
                save(ele);
                handleRetrieval(new GlobalRetrieval(),ele);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void exportDatatoExcel(QueryFilter<ProjectEstablishment> queryFilter, HttpServletResponse response) throws IOException {
        String fileName = "项目立项-导出结果";

        // 获取数据
        BizUtils.handleFilter(queryFilter, "bpe", "pm");
        List<ProjectEstablishment> data = baseMapper.queryAllByPage(
                    convert2IPage(new PageBean(0,-1,false)),
                    convert2Wrapper(queryFilter, currentModelClass())).
                getRecords();
        if(null==data||data.size()==0){
            throw new RuntimeException("没有要导出的的数据！");
        }

        // EasyPoi 导出参数、样式、表格格式设置
        ExportParams exportParams = BizUtils.getExportParams(fileName);
        fileName += ".xlsx";

        // 导出下载excel文件
        Workbook workbook = ExcelExportUtil.exportExcel(exportParams, ProjectEstablishment.class, data);
        ExcelUtil.downloadExcel(workbook, fileName, response);
    }

    @Override
    public void updateProjectIdByProiectId(String oldProjectIds, String newProjectId) {
        UpdateWrapper<ProjectEstablishment> wrapper = new UpdateWrapper<ProjectEstablishment>()
                .set(StringUtils.isNotBlank(newProjectId), "PROJECT_ID_", newProjectId)
                .in("PROJECT_ID_", Arrays.asList(oldProjectIds.split(",")));
        this.update(wrapper);
    }

    @Override
    public List<Map<String, Object>> queryInvestment(StatisticsVo statisticsVo) {
        List<Map<String, Object>> list = this.baseMapper.queryInvestment(statisticsVo);
        if(null != list){
            List<DictModel> projectType = sdm.queryDictListItemsByCode("xmgl-xmlx");
            for(Map<String, Object> map : list){
                String type = String.valueOf(map.get("type"));
                map.put("typeName", BizUtils.getDicValueByCode(projectType, type));
            }
        }
        return list;
    }

    //根据项目id查询总投资额
    @Override
    public String getMoneyByPid(String id){
        return baseMapper.getMoneyByPid(id);
    }

}

