package com.artfess.form.controller;

import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.feign.SystemConfigFeignService;
import com.artfess.base.feign.WorkflowFeignService;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.JsonUtil;
import com.artfess.base.util.StringUtil;
import com.artfess.form.model.FormPrintTemplate;
import com.artfess.form.persistence.manager.FormPrintTemplateManager;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.fasterxml.jackson.databind.node.ObjectNode;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.util.Optional;

@RestController
@RequestMapping("/form/printTemplate/v1/")
@Api(tags = "套打模板")
@ApiGroup(group = {ApiGroupConsts.GROUP_FORM})
public class FormPrintTemplateController extends BaseController<FormPrintTemplateManager, FormPrintTemplate> {

    @Resource
    WorkflowFeignService workflowFeignService;
    @Resource
    FormPrintTemplateManager formPrintTemplateManager;
    @Resource
    SystemConfigFeignService systemConfigFeignService;

    @RequestMapping(value = "getPrintList", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "打印模板列表", httpMethod = "POST", notes = "打印模板列表")
    @ResponseBody
    public PageList<FormPrintTemplate> getPrintList(@ApiParam(name = "queryFilter", value = "通用查询对象") @RequestBody QueryFilter<FormPrintTemplate> queryFilter) throws Exception {
        Page<FormPrintTemplate> list = (Page<FormPrintTemplate>) baseService.getPrintList(queryFilter);
        return new PageList<FormPrintTemplate>(list);
    }

    @PostMapping("/save")
    @ApiOperation("添加实体的接口")
    public CommonResult<String> save(@ApiParam(name = "model", value = "打印模板") @RequestBody FormPrintTemplate formPrintTemplate) {
        baseService.saveFormPrintTemplate(formPrintTemplate);
        return new CommonResult<>("上传模板成功");
    }

    @RequestMapping(value = "removes", method = RequestMethod.DELETE, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "批量删除bo定义", httpMethod = "DELETE", notes = "批量删除bo定义")
    public CommonResult<String> batchRemove(
            @ApiParam(name = "ids", value = "bo主键集合", required = true) @RequestParam String... ids) throws Exception {
        formPrintTemplateManager.removeByIds(ids);
        return new CommonResult<>("删除模板成功");
    }

    @RequestMapping(value = "setDefaultVersion", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "设置为主版本", httpMethod = "GET", notes = "设置为主版本")
    public CommonResult<String> setDefaultVersion(
            @ApiParam(name = "id", value = "主键", required = true) @RequestParam String id,
            @ApiParam(name = "formKey", value = "表单key", required = true) @RequestParam String formKey,
            @ApiParam(name = "printType", value = "打印类型", required = true) @RequestParam String printType) throws Exception {
        baseService.setDefaultVersion(formKey, id, printType);
        return new CommonResult<>("设置主版本成功");
    }

    @RequestMapping(value = "print", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "打印模板", httpMethod = "GET", notes = "打印模板")
    public CommonResult<String> print(
            @ApiParam(name = "defId", value = "流程实例Id", required = true) @RequestParam String defId,
            @ApiParam(name = "nodeId", value = "节点ID", required = true) @RequestParam Optional<String> nodeId,
            @ApiParam(name = "procInstId", value = "流程实例Id", required = true) @RequestParam String procInstId) throws Exception {
        ObjectNode objectNode = workflowFeignService.printBoAndFormKey(defId, nodeId.orElse(""), procInstId);
        String formKey = objectNode.get("formKey").asText();
        String formName = objectNode.get("formName").asText();
        FormPrintTemplate formPrintTemplate = baseService.getMainFormPrintTemplate(formKey);
        if (BeanUtils.isEmpty(formPrintTemplate)) {
            return new CommonResult<>(true, formName + "表单没有设置打印模板", "");
        } else {
            String fileId = formPrintTemplate.getFileId();
            objectNode.put("scriptStr", formPrintTemplate.getScriptStr());
            objectNode.put("fileId", fileId);
            String id = systemConfigFeignService.wordPrint(objectNode);
            // 新增打印记录
            try {
                ObjectNode record = JsonUtil.getMapper().createObjectNode();
                record.put("procInstId", procInstId);
                record.put("nodeId", nodeId.orElse(""));
                record.put("procDefId", defId);
                record.put("formKey", formKey);
                record.put("templateId", formPrintTemplate.getId());
                record.put("templateName", formPrintTemplate.getFileName());
                record.put("fileId", id);
                workflowFeignService.addPrintLog(record);
            } catch (Exception e) {
                System.out.println("新增打印记录失败：" + e.getMessage());
            }
            return new CommonResult<>(true, "", id);
        }
    }

    @RequestMapping(value = "wordPrint", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "打印word模板", httpMethod = "GET", notes = "打印word模板")
    public CommonResult<String> wordPrint(@ApiParam(name = "id", value = "id", required = true) @RequestParam String id,
                                          @ApiParam(name = "defId", value = "流程实例Id", required = true) @RequestParam String defId,
                                          @ApiParam(name = "nodeId", value = "节点ID", required = true) @RequestParam Optional<String> nodeId,
                                          @ApiParam(name = "procInstId", value = "流程实例Id", required = true) @RequestParam String procInstId) throws Exception {
        ObjectNode objectNode = workflowFeignService.printBoAndFormKey(defId, nodeId.orElse(""), procInstId);
        FormPrintTemplate formPrintTemplate = baseService.get(id);
        if (BeanUtils.isEmpty(formPrintTemplate)) {
            return new CommonResult<>(false, "请选择正确的打印模板");
        }

        String fileId = formPrintTemplate.getFileId();
        objectNode.put("fileId", fileId);
        objectNode.put("defId", defId);
        objectNode.put("nodeId", nodeId.orElse(""));
        objectNode.put("scriptStr", StringUtil.isNotEmpty(formPrintTemplate.getScriptStr()) ? formPrintTemplate.getScriptStr() : "");
        String printId = systemConfigFeignService.wordPrint(objectNode);
        return new CommonResult<>(true, "", printId);
    }

    @RequestMapping(value = "getFormKey", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "获取表单Key", httpMethod = "GET", notes = "获取表单Key")
    public ObjectNode getFormKey(
            @ApiParam(name = "defId", value = "流程实例Id", required = true) @RequestParam String defId,
            @ApiParam(name = "nodeId", value = "节点ID", required = true) @RequestParam Optional<String> nodeId,
            @ApiParam(name = "procInstId", value = "流程实例Id", required = true) @RequestParam String procInstId) throws Exception {
        return workflowFeignService.printBoAndFormKey(defId, nodeId.orElse(""), procInstId);
    }

    @RequestMapping(value = "getMainTemlate", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "通过表单key获取列表", httpMethod = "GET", notes = "通过表单key获取列表")
    public FormPrintTemplate getMainTemlate(
            @ApiParam(name = "formKey", value = "表单key", required = true) @RequestParam String formKey,
            @ApiParam(name = "printType", value = "打印类型", required = true) @RequestParam String printType) throws Exception {
        return baseService.getMailPrintTemplates(formKey, printType);
    }

}
