package com.artfess.application.jms.impl;

import com.artfess.activemq.model.JmsMessage;
import com.artfess.base.conf.SaaSConfig;
import com.artfess.base.context.BaseContext;
import com.artfess.base.feign.UCFeignService;
import com.artfess.base.jms.JmsActor;
import com.artfess.base.jms.NoticeMessageType;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.JsonUtil;
import com.artfess.base.util.StringUtil;
import com.artfess.mail.model.MailLing;
import com.artfess.mail.model.MailSetting;
import com.artfess.mail.util.MailUtil;
import com.artfess.application.jms.JmsHandler;
import com.artfess.application.persistence.manager.MessageLogManager;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;

/**
 * 邮件消息处理器
 *
 * <pre>
 * @author zhaoxy
 * @company 广州宏天软件股份有限公司
 * @email zhxy@jee-soft.cn
 * @date 2018-06-06 14:23
 * </pre>
 */
@Service
public class MailHandler implements JmsHandler {
    private static final Logger logger = LoggerFactory.getLogger(MailHandler.class);
    @Resource
    MessageLogManager messageLogManager;

    @Value("${mail.username}")
    private String username;
    @Value("${mail.host}")
    private String sendHost;
    @Value("${mail.port}")
    private String sendPort;
    @Value("${mail.password}")
    private String password;
    @Resource
    BaseContext baseContext;
    @Resource
    UCFeignService ucFeignService;

    @Override
    public String getType() {
        return NoticeMessageType.MAIL.key();
    }

    @Override
    public boolean send(JmsMessage jmsMessage) {
        try {
            List<JmsActor> recievers = jmsMessage.getReceivers();

            String strReceiver = getMailAddress(recievers);
            if (StringUtil.isEmpty(strReceiver))
                return false;
            MailSetting mailSetting = new MailSetting();
            mailSetting.setSendHost(sendHost);
            mailSetting.setSendPort(sendPort);
            mailSetting.setMailAddress(username);
            mailSetting.setPassword(password);
            MailLing mail = new MailLing();
            mail.setTo(strReceiver);
            mail.setFrom(username);
            mail.setSubject(jmsMessage.getSubject());
            mail.setContent(jmsMessage.getContent());
            MailUtil mailUtil = null;
            String tenantId = baseContext.getCurrentTenantId();
            ObjectNode infoNode = ucFeignService.getTenantMailServiceInfo(tenantId);
            if (BeanUtils.isNotEmpty(infoNode)) {
                MailSetting seting = getBymailSetting(infoNode);
                mailUtil = new MailUtil(seting);
            } else {
                mailUtil = new MailUtil(mailSetting);
            }
            mailUtil.sendEmail(mail);
            messageLogManager.handLogByMsgHander(jmsMessage, true, "");
            return true;
        } catch (Exception e) {
            logger.error(e.getMessage());
            messageLogManager.handLogByMsgHander(jmsMessage, false, e.getMessage());
            return false;
        }
    }

    /**
     * 根据mailSetting对象，得到MailSeting对象<br/>
     * 注：mailSetting对象中的密码字段应为使用EncryptUtil.encrypt方法加密后的密文
     *
     * @param infoNode
     * @return
     * @throws Exception
     */
    private MailSetting getBymailSetting(ObjectNode infoNode) throws Exception {
        MailSetting seting = new MailSetting();
        String userName = JsonUtil.getString(infoNode, "userName", "");
        String protocal = JsonUtil.getString(infoNode, "mailType", "");
        seting.setProtocal(protocal);
        seting.setMailAddress(userName);
        seting.setPassword(JsonUtil.getString(infoNode, "mailPass", ""));
        seting.setNickName(JsonUtil.getString(infoNode, "nickName", ""));
        String useSsl = JsonUtil.getString(infoNode, "useSsl", "");
        seting.setSSL(!"0".equals(useSsl));
        seting.setValidate(true);
        seting.setIsDeleteRemote(false);
        seting.setIsHandleAttach(true);
        seting.setMailType(JsonUtil.getString(infoNode, "mailType", ""));
        seting.setReceiveHost(JsonUtil.getString(infoNode, "mailHost", ""));
        seting.setReceivePort(JsonUtil.getString(infoNode, "mailPort", ""));
        String s = userName.substring(userName.indexOf('@') + 1, userName.length()).trim();
        seting.setSmtpHost("smtp." + s);
        seting.setSmtpPort("25");
        seting.setSendHost(seting.getSmtpHost());
        seting.setSendPort(seting.getSmtpPort());
        return seting;
    }

    private String getMailAddress(List<JmsActor> recievers) {
        if (BeanUtils.isEmpty(recievers)) return "";
        StringBuilder toUserEmail = new StringBuilder();
        for (JmsActor iUser : recievers) {
            String email = iUser.getEmail();
            if (StringUtil.isEmpty(email)) continue;
            toUserEmail.append(",");
            toUserEmail.append(email);
        }
        String result = toUserEmail.toString();
        result = result.replaceFirst(",", "");
        return result;
    }

    @Override
    public String getTitle() {
        return "邮件";
    }

    @Override
    public boolean getIsDefault() {
        return false;
    }

    @Override
    public boolean getSupportHtml() {
        return true;
    }
}
