package com.artfess.uc.controller;

import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.FieldRelation;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.StringUtil;
import com.artfess.uc.manager.MatrixColDefManager;
import com.artfess.uc.manager.MatrixManager;
import com.artfess.uc.model.Matrix;
import com.artfess.uc.model.MatrixColDef;
import com.artfess.uc.model.User;
import com.artfess.uc.util.ContextUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;

/**
 * 
 * <pre> 
 * 描述：矩阵管理 控制器类
 * 构建组：x7
 * 作者:pangq
 * 邮箱:pangq@jee-soft.cn
 * 日期:2020-06-05 10:24:44
 * 版权：广州宏天软件股份有限公司
 * </pre>
 */
@RestController
@RequestMapping(value="/uc/matrix/v1")
@Api(tags="matrixController")
@ApiGroup(group= {ApiGroupConsts.GROUP_UC})
public class MatrixController extends BaseController<MatrixManager,Matrix>{
	@Resource
	MatrixManager matrixManager;
	
	@Resource
	MatrixColDefManager matrixColDefManager;
	
	/**
	 * 矩阵管理列表(分页条件查询)数据
	 * @param request
	 * @return
	 * @throws Exception 
	 * PageJson
	 * @exception 
	 */
	@PostMapping("/listJson")
	@ApiOperation(value="矩阵管理数据列表", httpMethod = "POST", notes = "获取矩阵管理列表")
	public PageList<Matrix> list(@ApiParam(name="queryFilter",value="查询对象")@RequestBody QueryFilter<Matrix> queryFilter) throws Exception{
		User user = ContextUtil.getCurrentUser();
		if (!user.isAdmin()){
			queryFilter.addFilter("createBy", user.getUserId(), QueryOP.EQUAL, FieldRelation.OR, "manager");
			queryFilter.addFilter("managerId", user.getUserId(), QueryOP.EQUAL, FieldRelation.OR, "manager");
		}
		return matrixManager.query(queryFilter);
	}
	/**
	 * 矩阵管理列表(分页条件查询)数据：包含列信息
	 * @param request
	 * @return
	 * @throws Exception 
	 * PageJson
	 * @exception 
	 */
	@PostMapping("/listDetail")
	@ApiOperation(value="矩阵管理数据列表", httpMethod = "POST", notes = "获取矩阵管理列表")
	public PageList<Matrix> listDetail(@ApiParam(name="queryFilter",value="查询对象")@RequestBody QueryFilter<Matrix> queryFilter) throws Exception{
		PageList<Matrix> query = matrixManager.query(queryFilter);
		if(query.getRows()!=null){
			for (Matrix matrix : query.getRows()) {
				List<MatrixColDef> condList = matrixColDefManager.getCondList(matrix.getId());
				List<MatrixColDef> roleList = matrixColDefManager.getRoleList(matrix.getId());
				matrix.setCondList(condList);
				matrix.setRoleList(roleList);
			}
		}
		return query;
	}
	
	/**
	 * 矩阵管理明细页面
	 * @param id
	 * @return
	 * @throws Exception 
	 * ModelAndView
	 */
	@GetMapping(value="/getJson")
	@ApiOperation(value="矩阵管理数据详情",httpMethod = "GET",notes = "矩阵管理数据详情")
	public Matrix get(@ApiParam(name="id",value="业务对象主键", required = true)@RequestParam String id) throws Exception{
		Matrix matrix = matrixManager.getDetail(id);
		return matrix;
	}
	/**
	 * 根据code获取实体
	 * @param code
	 * @return
	 * @throws Exception
	 */
	@RequestMapping(value="getByCode", method=RequestMethod.GET, produces = { "application/json; charset=utf-8" })
	@ApiOperation(value = "根据编码/别名获取矩阵", httpMethod = "GET", notes = "明细页面")
	public @ResponseBody Matrix getByAlias(@ApiParam(name="code", value="code", required = true)@RequestParam String code) throws Exception {
		if (BeanUtils.isEmpty(code)) {
			return null;
		}
		Matrix matrix = matrixManager.getByCode(code);
		if(BeanUtils.isNotEmpty(matrix)){
			List<MatrixColDef> condList = matrixColDefManager.getCondList(matrix.getId());
			List<MatrixColDef> roleList = matrixColDefManager.getRoleList(matrix.getId());
			matrix.setCondList(condList);
			matrix.setRoleList(roleList);
		}
		return matrix ;
	}
	
    /**
	 * 新增矩阵管理
	 * @param matrix
	 * @throws Exception 
	 * @return
	 * @exception 
	 */
	@PostMapping(value="save")
	@ApiOperation(value = "新增,更新矩阵管理数据", httpMethod = "POST", notes = "新增,更新矩阵管理数据")
	public CommonResult<String> save(@ApiParam(name="matrix",value="矩阵管理业务对象", required = true)@RequestBody Matrix matrix) throws Exception{
		String msg = "保存矩阵管理成功";
		Boolean isCodeExist = matrixManager.isCodeExist(matrix.getCode(),matrix.getId());
		if(isCodeExist){
			return new CommonResult<>(false,"别名已存在");
		}
		matrixManager.saveMatrix(matrix);
		
		return new CommonResult<String>(msg);
	}
	
	/**
	 * 批量删除矩阵管理记录
	 * @param ids
	 * @throws Exception 
	 * @return
	 * @exception 
	 */
	@DeleteMapping(value="/remove")
	@ApiOperation(value = "批量删除矩阵管理记录", httpMethod = "DELETE", notes = "批量删除矩阵管理记录")
	public CommonResult<String> removes(@ApiParam(name="ids",value="业务主键数组,多个业务主键之间用逗号分隔", required = true)@RequestParam String...ids) throws Exception{
		matrixManager.deleLogical(ids);
		return new CommonResult<String>(true, "批量删除成功");
	}
	
	/**
	 * 发布
	 * @param id
	 * @return
	 * @throws Exception 
	 * ModelAndView
	 */
	@GetMapping(value="/publish")
	@ApiOperation(value="发布",httpMethod = "GET",notes = "publish")
	public CommonResult<String>  publish(@ApiParam(name="id",value="业务对象主键", required = true)@RequestParam String id) throws Exception{
		matrixManager.publish(id);
		return new CommonResult<String>(true, "发布成功");
	}
	/**
	 * 保存管理员
	 * @param id
	 * @return
	 * @throws Exception 
	 * ModelAndView
	 */
	@GetMapping(value="/saveManager")
	@ApiOperation(value="保存管理员",httpMethod = "GET",notes = "saveManager")
	public CommonResult<String>  saveManager(@ApiParam(name="id",value="业务对象主键", required = true)@RequestParam String id,
			@ApiParam(name="userId",value="管理员userId", required = true)@RequestParam String userId,
			@ApiParam(name="fullname",value="管理员名称", required = true)@RequestParam String fullname) throws Exception{
		Matrix matrix = matrixManager.get(id);
		matrix.setManagerId(userId);
		matrix.setManagerName(fullname);
		matrixManager.update(matrix);
		return new CommonResult<String>(true, "保存成功");
	}
	
	@PostMapping("/getUsers")
	@ApiOperation(value="获取矩阵配置数据", httpMethod = "POST", notes = "获取矩阵配置数据")
	public List<Map<String, Object>> getUsers(@RequestBody Map<String, Object> params) throws Exception {
		return matrixManager.getUsers(params);
	}
	
	/**
	 * 获取矩阵配置数据
	 * @param id
	 * @return
	 * @throws Exception 
	 */
	@PostMapping("/data/list/{matrixId}")
	@ApiOperation(value="获取矩阵配置数据", httpMethod = "POST", notes = "获取矩阵配置数据")
	@ResponseBody
	public Object dataList(@ApiParam(name="matrixId",value="矩阵id")@PathVariable String matrixId, @ApiParam(name="queryFilter",value="查询对象")@RequestBody QueryFilter<Matrix> queryFilter) throws Exception{
		Map<String,Object> map = this.matrixManager.getData(matrixId,queryFilter.getPageBean());
		return map;
	}
	
	/**
	 * 根据主键获取某条数据
	 * @param id
	 * @return
	 * @throws Exception 
	 */
	@GetMapping("/data/getOneData")
	@ApiOperation(value="获取某条数据", httpMethod = "GET", notes = "获取某条数据")
	@ResponseBody
	public Object getOneData(@ApiParam(name="pk",value="pk")@RequestParam String pk,@ApiParam(name="matrixId",value="matrixId")@RequestParam String matrixId) throws Exception{
		if(StringUtil.isNotEmpty(pk)){
			Map<String,Object> map = this.matrixManager.getOneData(matrixId,pk);
			return map;
		}else{
			Map<String,Object> map = this.matrixManager.getBlankOneData(matrixId);
			return map;
		}
	}
	
	/**
	 * 保存数据
	 * @param id
	 * @return
	 * @throws Exception 
	 */
	@PostMapping("/data/save/{matrixId}")
	@ApiOperation(value="获取矩阵配置数据", httpMethod = "POST", notes = "获取矩阵配置数据")
	public CommonResult<String> dataSave(@ApiParam(name="matrixId",value="矩阵id")@PathVariable String matrixId,@ApiParam(name="data",value="data")@RequestBody String data) throws Exception{
		this.matrixManager.saveData(matrixId,data);
		return new CommonResult<String>(true, "保存成功");
	}
	/**
	 * 批量删除数据记录
	 * @param ids
	 * @throws Exception 
	 * @return
	 * @exception 
	 */
	@DeleteMapping(value="/data/remove/{matrixId}")
	@ApiOperation(value = "批量删除数据记录", httpMethod = "DELETE", notes = "批量删除数据记录")
	public CommonResult<String> dataRemove(@ApiParam(name="matrixId",value="矩阵id")@PathVariable String matrixId, @ApiParam(name="ids",value="业务主键数组,多个业务主键之间用逗号分隔", required = true)@RequestParam String...ids) throws Exception{
		this.matrixManager.dataRemove(matrixId,ids);
		return new CommonResult<String>(true, "删除成功");
	}
	
}
