package com.artfess.portal.controller;

import cn.hutool.json.JSON;
import cn.hutool.json.JSONUtil;
import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.constants.WebsocketConst;
import com.artfess.base.controller.BaseController;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.webSocket.PushService;
import com.artfess.sysConfig.persistence.manager.SysDictionaryManager;
import com.artfess.sysConfig.persistence.manager.SysTypeManager;
import com.artfess.sysConfig.persistence.model.SysDictionary;
import com.artfess.sysConfig.persistence.model.SysDictionaryDetail;
import com.artfess.sysConfig.persistence.model.SysType;
import com.artfess.sysConfig.vo.DictionaryDetailVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @Program: edp
 * @Date: 2021/3/5
 * @Author: ZQF
 * @Description: 系统字典
 */
@RestController
@RequestMapping("/sys/sysDictionary/v1")
@Api(tags = "系统字典")
@ApiGroup(group = {ApiGroupConsts.GROUP_SYSTEM})
public class SysDictionaryController extends BaseController<SysDictionaryManager, SysDictionary> {

    @Resource
    SysDictionaryManager sysDictionaryManager;

    @Resource
    SysTypeManager sysTypeManager;

    @Resource
    PushService pushService;

    @RequestMapping(value = "listJson", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "系统字典(分页条件查询)", httpMethod = "POST", notes = "系统字典(分页条件查询)")
    public PageList<SysDictionary> listJson(@ApiParam(name = "queryFilter", value = "通用查询对象") @RequestBody QueryFilter<SysDictionary> queryFilter) throws Exception {
        QueryFilter<SysType> filter = QueryFilter.build();
        //过滤掉已近删除的树的字典数据
        List<Object> sysTypeList = sysTypeManager.listObjs();
        if (null != sysTypeList && sysTypeList.size() > 0) {
            queryFilter.addFilter("TREE_ID_", sysTypeList, QueryOP.IN);
            return sysDictionaryManager.query(queryFilter);
        }
        return new PageList<SysDictionary>();
    }

    @RequestMapping(value = "saveDictionary", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "保存字典", httpMethod = "POST", notes = "保存字典")
    public CommonResult<String> saveDictionary(@ApiParam(name = "sysDictionary", value = "字典对象") @RequestBody SysDictionary sysDictionary) throws Exception {
        String msg = sysDictionaryManager.saveDictionary(sysDictionary);


        HashMap<String, String> map = new HashMap<>();
        map.put(WebsocketConst.MSG_CMD, WebsocketConst.CMD_DICT);
        JSON parse = JSONUtil.parse(map);
        pushService.pushMsgToAll(parse.toString());
        return new CommonResult<>(true, msg, sysDictionary.getId());
    }

    @RequestMapping(value = "deleteDictionaries", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "批量删除", httpMethod = "POST", notes = "批量删除")
    public CommonResult<String> deleteDictionaries(@ApiParam(name = "ids", value = "Id集合，以,隔开") @RequestParam String ids) throws Exception {
        sysDictionaryManager.deleteDictionaries(ids);
        //Websocket发送消息

        HashMap<String, String> map = new HashMap<>();
        map.put(WebsocketConst.MSG_CMD, WebsocketConst.CMD_DICT);
        JSON parse = JSONUtil.parse(map);
        pushService.pushMsgToAll(parse.toString());
        return new CommonResult<>(true, "删除成功");
    }

    @RequestMapping(value = "updateDictionariesStatus", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "修改字典状态", httpMethod = "POST", notes = "修改字典状态")
    public CommonResult<String> updateDictionariesStatus(@ApiParam(name = "ids", value = "Id集合，以,隔开") @RequestParam String ids,
                                                         @ApiParam(name = "status", value = "状态") @RequestParam Integer status) throws Exception {
        sysDictionaryManager.updateDictionariesStatus(ids, status);
        //webSocket给前台发送消息，要求前台做字典更新

        HashMap<String, String> map = new HashMap<>();
        map.put(WebsocketConst.MSG_CMD, WebsocketConst.CMD_DICT);
        JSON parse = JSONUtil.parse(map);
        pushService.pushMsgToAll(parse.toString());
        return new CommonResult<>(true, "修改成功");
    }

    @RequestMapping(value = "loadDictionaryById", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "根据ID返回字典详情", httpMethod = "POST", notes = "根据ID返回字典详情")
    public SysDictionary loadDictionaryById(@ApiParam(name = "id", value = "主键ID") @RequestParam String id) throws Exception {
        return sysDictionaryManager.loadDictionaryById(id);
    }

    @RequestMapping(value = "moveDictionaries", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "移动字典", httpMethod = "POST", notes = "移动字典")
    public CommonResult<String> moveDictionaries(@ApiParam(name = "ids", value = "Id集合，以,隔开") @RequestParam String ids,
                                                 @ApiParam(name = "treeId", value = "公共树ID") @RequestParam String treeId) throws Exception {
        sysDictionaryManager.moveDictionaries(ids, treeId);

        return new CommonResult<>(true, "移动成功");
    }

    @RequestMapping(value = "updateDictionariesSequence", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "修改字典排序号", httpMethod = "POST", notes = "修改字典排序号")
    public CommonResult<String> updateDictionariesSequence(@ApiParam(name = "params", value = "排序参数：Key：ID，Value：排序号") @RequestBody HashMap<String, Integer> params) throws Exception {
        sysDictionaryManager.updateDictionariesSequence(params);
        return new CommonResult<>(true, "排序完成");
    }

    @RequestMapping(value = "syncDictionaryRedis", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "同步缓存", httpMethod = "POST", notes = "同步缓存")
    public CommonResult<String> syncDictionaryRedis() throws Exception {
        sysDictionaryManager.syncDictionaryRedis();

        //webSocket给前台发送消息，要求前台做字典更新
        HashMap<String, String> map = new HashMap<>();
        map.put(WebsocketConst.MSG_CMD, WebsocketConst.CMD_DICT);
        JSON parse = JSONUtil.parse(map);
        pushService.pushMsgToAll(parse.toString());
        return new CommonResult<>(true, "缓存已更新");
    }

    @RequestMapping(value = "queryLowerDictionary", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "根据字典值的id级联查询下级字典", httpMethod = "POST", notes = "根据字典值的id级联查询下级字典")
    public List<SysDictionary> queryLowerDictionary(@ApiParam(name = "detailId", value = "Id集合，以,隔开") @RequestParam String detailId) throws Exception {
        return sysDictionaryManager.queryLowerDictionary(detailId);
    }

    @RequestMapping(value = "queryAllDictItems", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "查询所有字典信息（登陆时缓存到前台）", httpMethod = "POST", notes = "查询所有字典信息（登陆时缓存到前台）")
    public Map<String, List<DictionaryDetailVo>> queryAllDictItems() throws Exception {
        return sysDictionaryManager.queryAllDictItems();
    }

    @RequestMapping(value = "queryDictItemsByCode", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "根据codes查询字典信息", httpMethod = "POST", notes = "根据codes查询字典信息")
    public Map<String, List<SysDictionaryDetail>> queryDictItemsByCode(@ApiParam(name = "codes", value = "code集合，以,隔开") @RequestParam String codes) throws Exception {
        return sysDictionaryManager.queryDictItemsByCode(codes);
    }

    @RequestMapping(value = "queryLowerDictionaryDetail", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "通过上级字典项的值与其下级级联字典的id，获取级联的字典项数据", httpMethod = "POST", notes = "通过上级字典项的值与其下级级联字典的id，获取级联的字典项数据")
    public List<SysDictionaryDetail> queryLowerDictionaryDetail(@ApiParam(name = "dictLowerId", value = "下级字典id") @RequestParam String dictLowerId,
                                                                @ApiParam(name = "detailId", value = "上级字典值的id") @RequestParam String detailId) throws Exception {
        return sysDictionaryManager.queryLowerDictionaryDetail(dictLowerId, detailId);
    }

}
