package com.artfess.application.controller;


import com.artfess.application.model.RemindConfig;
import com.artfess.application.persistence.manager.RemindConfigManager;
import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.valid.AddGroup;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.formula.functions.T;
import org.quartz.SchedulerException;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

/**
 * 业务提醒配置 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author min.wu
 * @since 2023-12-21
 */
@Slf4j
@RestController
@RequestMapping(value = "/remindConfig/v1")
@Api(tags = "工作提醒")
@ApiGroup(group = {ApiGroupConsts.GROUP_APPLICATION})
public class RemindConfigController extends BaseController<RemindConfigManager, RemindConfig> {

    @Override
    @PostMapping(value="/query", produces={"application/json; charset=utf-8" })
    @ApiOperation("分页查询结果")
    public PageList<RemindConfig> query(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<RemindConfig> queryFilter) {
        return baseService.findByPage(queryFilter);
    }

    @RequestMapping(value = "validClass", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "验证类", httpMethod = "GET", notes = "验证类")
    public CommonResult validClass(@ApiParam(name = "className", value = "类名", required = true) @RequestParam String className) throws Exception {
        boolean rtn = BeanUtils.validClass(className);
        if (rtn) {
            return new CommonResult(true, "验证类成功!", null);
        } else {
            return new CommonResult(false, "验证类失败!", null);
        }
    }

    @PostMapping("/saveOrUpdate")
    @ApiOperation("添加或数据抽取提醒")
    public CommonResult<String> saveOrUpdate(@ApiParam(name="model", value="数据抽取提醒") @Validated({AddGroup.class}) @RequestBody RemindConfig remindConfig) throws Exception {
        boolean rtn = BeanUtils.validClass(remindConfig.getPlanClass());
        if (rtn) {
            boolean result = baseService.saveOrUpdateInfo(remindConfig);
            if(!result) {
                return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "操作失败!");
            }
        } else {
            return new CommonResult(false, "操作失败!", null);
        }

        return new CommonResult<>();
    }

    @Override
    @DeleteMapping("/{id}")
    @ApiOperation("根据id删除")
    public CommonResult<String> deleteById(@ApiParam(name="id", value="实体id") @PathVariable String id) {
        try {
            boolean result = false;
            result = baseService.deleteRemindConfigById(id);
            if(!result) {
                return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "删除实体失败");
            }
            return new CommonResult<>("删除成功！");
        } catch (Exception e) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION,  e.getMessage());
        }
    }

    @PostMapping(value = "/toggleTriggerRun")
    @ApiOperation(value = "启用或停止工作提醒", httpMethod = "POST", notes = "启用或停止工作提醒")
    public CommonResult<String> start(@ApiParam(name = "id", value = "提醒id", required = false) @RequestParam String id) throws Exception {
        boolean result = baseService.toggleTriggerRun(id);
        if(result == true){
            return new CommonResult<>("操作成功！");
        }else{
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "操作失败！");
        }
    }
    @PostMapping(value = "/executeJob")
    @ApiOperation(value = "执行工作提醒", httpMethod = "POST", notes = "执行工作提醒")
    public CommonResult<String> executeJob(@ApiParam(name = "id", value = "提醒id", required = true) @RequestParam String id) throws SchedulerException {
        boolean result = baseService.executeJob(id);
        if(result == true){
            return new CommonResult<>("执行工作提醒成功！");
        }else{
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "执行工作提醒失败！");
        }
    }

    @Override
    @GetMapping("/{id}")
    @ApiOperation("根据id查询实体")
    public RemindConfig getById(@ApiParam(name = "id", value = "实体id") @PathVariable String id) {
        return baseService.findById(id);
    }


}
