package com.artfess.cgpt.sso.manager.impl;

import com.artfess.base.exception.ApplicationException;
import com.artfess.base.jwt.JwtTokenHandler;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.util.EncryptUtil;
import com.artfess.base.util.JsonUtil;
import com.artfess.base.util.StringUtil;
import com.artfess.cgpt.sso.dao.SsoSystemDao;
import com.artfess.cgpt.sso.manager.SsoSystemManager;
import com.artfess.cgpt.sso.model.SsoSystem;
import com.artfess.uc.manager.OrgManager;
import com.artfess.uc.manager.UserManager;
import com.artfess.uc.params.org.OrgUserVo;
import com.artfess.uc.params.user.UserVo;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.fasterxml.jackson.databind.JsonNode;
import com.gexin.rp.sdk.base.uitls.MD5Util;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.time.LocalDate;
import java.util.List;
import java.util.Random;

/**
 * 单点登录系统配置 服务实现类
 *
 * @company 阿特菲斯信息技术有限公司
 * @author chens
 * @since 2022-12-02
 */
@Service
public class SsoSystemManagerImpl extends BaseManagerImpl<SsoSystemDao, SsoSystem> implements SsoSystemManager {

    @Autowired
    private UserManager userManager;

    @Autowired
    OrgManager orgService;

    @Autowired
    private JwtTokenHandler jwtTokenHandler;


    @Override
    public SsoSystem getEntityByAkAndSk(String ak, String sk) {
        return baseMapper.getByAkAndSk(ak,sk);
    }

    @Override
    @Transactional
    public boolean insertSsoSystem(SsoSystem ssoSystem) throws Exception {
        boolean checkFlag = checkSameCode(ssoSystem.getSysCode(),ssoSystem.getId());
        if(checkFlag){
            throw new ApplicationException("系统编码【"+ssoSystem.getSysCode()+"】重复！");
        }
        String accessKey = null;
        String secretKey = null;
        try {
            accessKey = EncryptUtil.encrypt(MD5Util.getMD5Format(ssoSystem.getSysCode()+"_"+new Random().nextInt(1000)));
            Long time = System.currentTimeMillis();
            secretKey =EncryptUtil.encrypt(MD5Util.getMD5Format(accessKey+time));
        } catch (Exception e) {
            throw new ApplicationException("系统生成AK与SK出错！",e);
        }
        ssoSystem.setAccessKey(accessKey);
        ssoSystem.setSecretKey(secretKey);
        int num = this.baseMapper.insert(ssoSystem);
        UserVo userVo = new UserVo();
        userVo.setAccount(ssoSystem.getSysCode());
        userVo.setFullname(ssoSystem.getSysName());
        userVo.setPassword(ssoSystem.getSysCode()+"@6921");
        userManager.addUser(userVo);
        OrgUserVo orgUserVo = new OrgUserVo();
        orgUserVo.setAccount(ssoSystem.getSysCode());
        orgUserVo.setOrgCode("dsfxt");
        orgService.addOrgUser(orgUserVo);

        if(num>0){
            return true;
        }
        return false;
    }

    @Override
    @Transactional
    public boolean updateSsoSystem(SsoSystem ssoSystem) {
        boolean checkFlag = checkSameCode(ssoSystem.getSysCode(),ssoSystem.getId());
        if(checkFlag){
            throw new ApplicationException("系统编码【"+ssoSystem.getSysCode()+"】重复！");
        }
        SsoSystem old = this.baseMapper.selectById(ssoSystem.getId());
        old.setSysName(ssoSystem.getSysName());
        old.setSysCode(ssoSystem.getSysCode());
        old.setCompanyName(ssoSystem.getCompanyName());
        old.setContact(ssoSystem.getContact());
        old.setPhone(ssoSystem.getPhone());
        old.setIsIpAstrict(ssoSystem.getIsIpAstrict());
        old.setIpAddress(ssoSystem.getIpAddress());
        old.setLoginTime(ssoSystem.getLoginTime());
        int num = this.baseMapper.updateById(old);
        if(num>0){
            return true;
        }
        return false;
    }

    private boolean checkSameCode(String code,String id){
        Assert.hasText(code, "系统编码不能为空！");
        QueryWrapper<SsoSystem> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("sys_code_", code);
        queryWrapper.ne(StringUtil.isNotEmpty(id), "id_", id);
        int num = baseMapper.selectCount(queryWrapper);
        if(num>0){
            return true;
        }else{
            return false;
        }
    }

    public JsonNode getSystemByAKAndSK(String ak, String sk) {
        Assert.hasText(ak, "系统访问key不能为空！");
        Assert.hasText(sk, "系统访问安全密钥不能为空！");
        QueryWrapper<SsoSystem> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("access_key_", ak);
        queryWrapper.eq("secret_key_", sk);
        List<SsoSystem> list = this.baseMapper.selectList(queryWrapper);
        if(!CollectionUtils.isEmpty(list)){
            SsoSystem ssoSystem =list.get(0);
            JsonNode jsonNode = null;
            try {
                jsonNode = JsonUtil.toJsonNode(ssoSystem);
            } catch (IOException e) {
                return null;
            }
            return jsonNode;
        }else{
            return null;
        }
    }

    public void updateLastLoginTime(String ak, String sk) {
        Assert.hasText(ak, "系统访问key不能为空！");
        Assert.hasText(sk, "系统访问安全密钥不能为空！");
        UpdateWrapper<SsoSystem> updateWrapper = new UpdateWrapper<>();
        updateWrapper.set("LOGIN_TIME_", LocalDate.now());
        updateWrapper.eq("access_key_", ak);
        updateWrapper.eq("secret_key_", sk);
        this.baseMapper.update(null,updateWrapper);
    }

    @Override
    public String getSysNameByCode(String sysCode) {
        return baseMapper.getSysNameByCode(sysCode);
    }

    @Override
    public String getCurrentSysCode(HttpServletRequest request) {
        String tokenStr = request.getHeader("authorization");
        String token = tokenStr.substring(tokenStr.indexOf("Bearer")+7);
        String sysCode = jwtTokenHandler.getUsernameFromToken(token);
        return sysCode;
    }
}
