package com.artfess.cgpt.bidding.controller;


import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.exception.BaseException;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.BeanUtils;
import com.artfess.cgpt.bidding.manager.BidDocumentsManager;
import com.artfess.cgpt.bidding.model.BidDocuments;
import com.artfess.cgpt.bidding.model.BizBiddingQuotationHistory;
import com.artfess.cgpt.bidding.vo.QuoVo;
import com.artfess.cgpt.project.manager.ProjectApprovalManager;
import com.artfess.cgpt.project.model.ProjectApproval;
import com.artfess.cgpt.project.model.ProjectApprovalDetails;
import com.artfess.cgpt.purchasing.manager.MatApprovalManager;
import com.artfess.cgpt.purchasing.manager.TenderDocumentManager;
import com.artfess.cgpt.purchasing.model.MatApproval;
import com.artfess.cgpt.purchasing.model.MatApprovalDetails;
import com.artfess.cgpt.purchasing.model.PurchasingApplicationDetails;
import com.artfess.redis.util.RedisUtil;
import com.artfess.uc.api.impl.util.ContextUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import com.artfess.base.controller.BaseController;
import com.artfess.cgpt.bidding.model.BizBiddingQuotation;
import com.artfess.cgpt.bidding.manager.BizBiddingQuotationManager;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.List;

/**
 * 当前报价表 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author 管理员
 * @since 2024-01-17
 */
@RestController
@RequestMapping("/bizBiddingQuotation/v1/")
@Api(tags="当前报价表")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class BizBiddingQuotationController extends BaseController<BizBiddingQuotationManager, BizBiddingQuotation> {

    @Autowired
    RedisUtil redisUtil;
    @Autowired
    private MatApprovalManager matApprovalManager;
    @Autowired
    private TenderDocumentManager tenderDocumentManager;
    @Autowired
    private ProjectApprovalManager projectApprovalManager;
    @Autowired
    private BidDocumentsManager bidDocumentsManager;

    @PostMapping("/page")
    @ApiOperation(value = "M-分页查询数据")
    public CommonResult<PageList<BizBiddingQuotation>> queryAllByPage(@ApiParam(name = "queryfilter", value = "通用查询器") @RequestBody QueryFilter<BizBiddingQuotation> queryFilter) {
        PageList<BizBiddingQuotation> result = baseService.queryAllByPage(queryFilter);
        return new CommonResult<>(true, "查询成功", result);
    }

    @PostMapping("/pageQuoAndMatDetail")
    @ApiOperation(value = "M-分页查询数据（包含物资立项明细）")
    public CommonResult<PageList<BizBiddingQuotation>> pageQuoAndMatDetail(@ApiParam(name = "queryfilter", value = "通用查询器") @RequestBody QueryFilter<BizBiddingQuotation> queryFilter) {
        PageList<BizBiddingQuotation> result = baseService.pageQuoAndMatDetail(queryFilter);
        return new CommonResult<>(true, "查询成功", result);
    }

    @PostMapping("/pageMatDetail")
    @ApiOperation(value = "M-分页查询数据（包含物资立项明细）")
    public CommonResult<PageList<BizBiddingQuotation>> pageMatDetail(@ApiParam(name = "queryfilter", value = "通用查询器") @RequestBody QueryFilter<BizBiddingQuotation> queryFilter) {
        PageList<BizBiddingQuotation> result = baseService.pageMatDetail(queryFilter);
        return new CommonResult<>(true, "查询成功", result);
    }

    @PostMapping("/pageQuoOrg")
    @ApiOperation(value = "M-分页去重查询报价单位及评分")
    public CommonResult<PageList<BizBiddingQuotation>> pageQuoOrg(@ApiParam(name = "queryfilter", value = "通用查询器") @RequestBody QueryFilter<BizBiddingQuotation> queryFilter) {
        PageList<BizBiddingQuotation> result = baseService.pageQuoOrg(queryFilter);
        return new CommonResult<>(true, "查询成功", result);
    }

    @PostMapping("/pageProDetail")
    @ApiOperation(value = "M-分页查询数据（包含项目立项明细）")
    public CommonResult<PageList<BizBiddingQuotation>> pageProDetail(@ApiParam(name = "queryfilter", value = "通用查询器") @RequestBody QueryFilter<BizBiddingQuotation> queryFilter) {
        PageList<BizBiddingQuotation> result = baseService.pageProDetail(queryFilter);
        return new CommonResult<>(true, "查询成功", result);
    }

    @PostMapping("/pageLowestEva")
    @ApiOperation(value = "M-分页查询数据（包含最低价标识）")
    public CommonResult<PageList<BizBiddingQuotation>> pageLowestEva(@ApiParam(name = "queryfilter", value = "通用查询器") @RequestBody QueryFilter<BizBiddingQuotation> queryFilter) {
        PageList<BizBiddingQuotation> result = baseService.pageLowestEva(queryFilter);
        return new CommonResult<>(true, "查询成功", result);
    }

    @PostMapping("/quotationMat")
    @ApiOperation(value = "M-物资报价",httpMethod = "POST")
    public CommonResult quoMat(@RequestBody QuoVo quoVo) {
        //获取运营方设置的原始结束时间
        MatApproval notice = quoVo.getMatApproval();

        MatApproval matApproval = matApprovalManager.getById(notice.getId());
        //如果当前时间小于运营方设置的开始时间，就不能报价
        if(LocalDateTime.now().isBefore(matApproval.getBiddingStartTime())) {
            return new CommonResult(false,"投标报价暂未开始！");
        }
        if(LocalDateTime.now().isAfter(matApproval.getBiddingEndTime())) {
            return new CommonResult(false,"投标报价已结束！");
        }
        if(matApproval.getBidStatus()==5){
            return new CommonResult(false,"已开标，无法继续投标报价！");
        }

        Boolean confirmStatus = tenderDocumentManager.queryConfirmStatus(notice.getId());
        if(!confirmStatus){
            return new CommonResult(false,"投标报价失败，请先确认补遗文件！");
        }

        String offerRedisKey = null;
        if(notice.getProcureType()==3){
            //限时竞价
            List<MatApprovalDetails> detailList = notice.getDetailsList();
            for(MatApprovalDetails detail : detailList){
                String quoPriceRedisKey = "quoPrice:" + detail.getId();
                Object priceObj = redisUtil.get(quoPriceRedisKey);
                if(BeanUtils.isNotEmpty(priceObj)){
                    BigDecimal quoPrice = (BigDecimal)priceObj;
                    if(quoPrice.compareTo(detail.getMyPrice()) < 0){
                        return new CommonResult(false, "标的ID【"+detail.getId()+"】报价金额不能高于当前最低报价,请刷新页面后重试...");
                    }
                }
            }

            offerRedisKey = "offer:" + notice.getDetailsList().get(0).getId();
            Object obj = redisUtil.get(offerRedisKey);
            if(BeanUtils.isNotEmpty(obj)){
                boolean isOffer = (boolean) obj;
                if(isOffer) {
                    return new CommonResult(false, "其他人正在报价中,请稍后尝试...");
                }
            }
            redisUtil.set(offerRedisKey, true, 60);
        }

        //如果当前时间超过了运营方设置的原始结束时间，就不能报价了
//        if(LocalDateTime.now().isAfter(notice.getBiddingEndTime())){
//            return new CommonResult(false,"报价已结束！");
//        }

        try {
            baseService.quoMat(quoVo);
        } catch (Exception e) {
            e.printStackTrace();
            System.out.println("报价失败:"+e.getMessage());
            return new CommonResult(false,"报价失败:"+e.getMessage());
        } finally {
            if(notice.getProcureType()==3){
                redisUtil.set(offerRedisKey, false, 60*60*24);
            }
        }

        if(notice.getProcureType()==3){
            //限时竞价
            List<MatApprovalDetails> detailList = notice.getDetailsList();
            for(MatApprovalDetails detail : detailList){
                String quoPriceRedisKey = "quoPrice:" + detail.getId();
                redisUtil.set(quoPriceRedisKey, detail.getMyPrice(), 60*60*72);
            }
        }

        return new CommonResult(true,"报价成功");
    }

    @PostMapping("/quotationProject")
    @ApiOperation(value = "M-项目报价",httpMethod = "POST")
    public CommonResult quoProject(@RequestBody QuoVo quoVo) {
        //获取运营方设置的原始结束时间
        ProjectApproval notice = quoVo.getProjectApproval();

        ProjectApproval approval = projectApprovalManager.getById(notice.getId());
        //如果当前时间小于运营方设置的开始时间，就不能报价
        if(LocalDateTime.now().isBefore(approval.getBiddingStartTime())) {
            return new CommonResult(false,"投标报价暂未开始！");
        }
        if(LocalDateTime.now().isAfter(approval.getBiddingEndTime())) {
            return new CommonResult(false,"投标报价已结束！");
        }
        if(approval.getBidStatus()==5){
            return new CommonResult(false,"已开标，无法继续投标报价！");
        }

        Boolean confirmStatus = tenderDocumentManager.queryConfirmStatus(notice.getId());
        if(!confirmStatus){
            return new CommonResult(false,"投标报价失败，请先确认补遗文件！");
        }

        String offerRedisKey = null;

        if(approval.getProcureType()==3){
            //限时竞价
            List<ProjectApprovalDetails> detailList = notice.getDetailsList();
            for(ProjectApprovalDetails detail : detailList){
                String quoPriceRedisKey = "quoPrice:" + detail.getId();
                Object priceObj = redisUtil.get(quoPriceRedisKey);
                if(BeanUtils.isNotEmpty(priceObj)){
                    BigDecimal quoPrice = (BigDecimal)priceObj;
                    if(quoPrice.compareTo(detail.getMyPrice()) < 0){
                        return new CommonResult(false, "标的ID【"+detail.getId()+"】报价金额不能高于当前最低报价,请刷新页面后重试...");
                    }
                }
            }

            offerRedisKey = "offer:" + notice.getDetailsList().get(0).getId();
            Object obj = redisUtil.get(offerRedisKey);
            if(BeanUtils.isNotEmpty(obj)){
                boolean isOffer = (boolean) obj;
                if(isOffer) {
                    return new CommonResult(false, "其他人正在报价中,请稍后尝试...");
                }
            }
            redisUtil.set(offerRedisKey, true, 60);
        }

        //如果当前时间超过了运营方设置的原始结束时间，就不能报价了
//        if(LocalDateTime.now().isAfter(notice.getBiddingEndTime())){
//            return new CommonResult(false,"报价已结束！");
//        }

        if(BeanUtils.isNotEmpty(approval) && approval.getIsBidDocumentsUpload()==1){
            //如果立项设置必传投标文件
            //查询投标文件是否全部签署完成
            BidDocuments bidDocuments = new BidDocuments();
            bidDocuments.setProjectApprovalType(2);
            bidDocuments.setProjectApprovalId(approval.getId());
            bidDocuments.setSupplierId(ContextUtil.getCurrentOrgId());
            bidDocuments.setQuotationRounds(BeanUtils.isNotEmpty(approval.getQuotationRoundsNum())?approval.getQuotationRoundsNum():1);
            Integer allSignStatus = bidDocumentsManager.getAllSignStatus(bidDocuments);
            if(allSignStatus==0){
                throw new BaseException("请先上传投标文件后重试【如投标文件为在线签署请签署后重试】");
            }

        }

        try {
            baseService.quoProject(quoVo);
        } catch (Exception e) {
            e.printStackTrace();
            System.out.println("报价失败:"+e.getMessage());
            return new CommonResult(false,"报价失败:"+e.getMessage());
        } finally {
            if(notice.getProcureType()==3){
                redisUtil.set(offerRedisKey, false, 60*60*24);
            }
        }

        if(notice.getProcureType()==3){
            //限时竞价
            List<ProjectApprovalDetails> detailList = notice.getDetailsList();
            for(ProjectApprovalDetails detail : detailList){
                String quoPriceRedisKey = "quoPrice:" + detail.getId();
                redisUtil.set(quoPriceRedisKey, detail.getMyPrice(), 60*60*72);
            }
        }

        return new CommonResult(true,"报价成功");
    }

    @PostMapping("/matBidOpenOrgData")
    @ApiOperation(value = "M-开标管理-物资查询公司报价")
    public CommonResult<MatApproval> matBidOpenOrgData(@ApiParam(name = "queryfilter", value = "通用查询器") @RequestBody QueryFilter<BizBiddingQuotation> queryFilter) {
        MatApproval result = baseService.matBidOpenOrgData(queryFilter);
        return new CommonResult<>(true, "查询成功", result);
    }

    @PostMapping("/proBidOpenOrgData")
    @ApiOperation(value = "M-开标管理-项目查询公司报价")
    public CommonResult<ProjectApproval> proBidOpenOrgData(@ApiParam(name = "queryfilter", value = "通用查询器") @RequestBody QueryFilter<BizBiddingQuotation> queryFilter) {
        ProjectApproval result = baseService.proBidOpenOrgData(queryFilter);
        return new CommonResult<>(true, "查询成功", result);
    }

    @PostMapping("/evaOrgQuoData")
    @ApiOperation(value = "M-评标管理-查询公司总报价")
    public CommonResult<PageList<BizBiddingQuotation>> evaOrgQuoData(@ApiParam(name = "queryfilter", value = "通用查询器") @RequestBody QueryFilter<BizBiddingQuotation> queryFilter) {
        PageList<BizBiddingQuotation> result = baseService.evaOrgQuoData(queryFilter);
        return new CommonResult<>(true, "查询成功", result);
    }

    @PostMapping("/evaOrgRankData")
    @ApiOperation(value = "M-定标管理-查询公司评分排名")
    public CommonResult<List<BizBiddingQuotation>> evaOrgRankData(@RequestParam("noticeId") String noticeId) {
        List<BizBiddingQuotation> result = baseService.evaOrgRankData(noticeId);
        return new CommonResult<>(true, "查询成功", result);
    }

    /**
     * 根据Excel文件转换数据
     * @param file Excel文件
     */
//    @PostMapping("/excelToData")
//    @ApiOperation(value = "M-根据Excel文件转换数据")
//    public CommonResult<List<PurchasingApplicationDetails>> excelToData(@RequestParam("file") MultipartFile file) throws Exception{
//        List<PurchasingApplicationDetails> result = baseService.excelToData(file);
//        return new CommonResult<>(true,"操作成功",result);
//    }

    /**
     * 导出数据到Excel
     * @param queryFilter 通用查询器
     * @param response 响应对象
     * @return 通用结果
     * @throws IOException IO异常
     */
    @PostMapping("/exportDataToExcel")
    @ApiOperation(value = "M-导出数据到Excel",notes = "参数说明：查询器只使到了query组传参，其他都可以不要")
    public CommonResult exportDataToExcel(@RequestBody QueryFilter<BizBiddingQuotation> queryFilter, HttpServletResponse response) throws IOException {
        baseService.exportDataToExcel(queryFilter,response);
        return new CommonResult<>("导出成功");
    }

}
