package com.artfess.cgpt.receipt.manager.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.json.JSONUtil;
import com.artfess.base.exception.BaseException;
import com.artfess.base.feign.SystemConfigFeignService;
import com.artfess.base.util.BeanUtils;
import com.artfess.cgpt.foreignApi.service.ForeignCallService;
import com.artfess.cgpt.foreignApi.vo.ForeignResponseVo;
import com.artfess.cgpt.receipt.manager.BizReconciliationDetailManager;
import com.artfess.cgpt.receipt.model.BizReconciliation;
import com.artfess.cgpt.receipt.dao.BizReconciliationDao;
import com.artfess.cgpt.receipt.manager.BizReconciliationManager;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.cgpt.receipt.model.BizReconciliationDetail;
import com.artfess.cgpt.receipt.vo.BizReconciliationVo;
import com.artfess.cgpt.utils.BizUtils;
import com.artfess.uc.api.impl.util.ContextUtil;
import com.artfess.uc.api.model.IUser;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.lang.reflect.Array;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.List;

/**
 * 对账单表 服务实现类
 *
 * @company 阿特菲斯信息技术有限公司
 * @author 管理员
 * @since 2024-03-20
 */
@Service
public class BizReconciliationManagerImpl extends BaseManagerImpl<BizReconciliationDao, BizReconciliation> implements BizReconciliationManager {

    @Value("${zfpt.purIsdUrl}")
    private String purIsdUrl;

    @Autowired
    private BizReconciliationDetailManager recDetailManager;
    @Autowired
    SystemConfigFeignService systemConfigFeignService;
    @Autowired
    private ForeignCallService foreignCallService;

    @Override
    public BizReconciliation getDetail(String id) {
        BizReconciliation result = super.getById(id);
        List<BizReconciliationDetail> detailList = recDetailManager.getByPid(id);
        result.setIsdDetail(detailList);
        return result;
    }

    @Override
    @Transactional
    public void removeByIds(List<String> ids) {
        recDetailManager.removeByPid(String.join(",",ids));
        super.removeByIds(ids);
    }

    @Override
    @Transactional
    public String saveVo(BizReconciliation entity) {
        List<BizReconciliationDetail> detailList = entity.getIsdDetail();

        if(BeanUtils.isEmpty(entity.getId())){
            //新增
            entity.setIsdCreatedate(LocalDateTime.now());
            if(BeanUtils.isEmpty(entity.getIsdCode())){
                entity.setIsdCode(BizUtils.getNextSerialNumber(systemConfigFeignService,"dzgl_dzdh"));
            }
        }else {
            //更新
            recDetailManager.removeByPid(entity.getId());
            entity.setIsdLastchgdate(LocalDateTime.now());
        }
        super.saveOrUpdate(entity);

        if(BeanUtils.isNotEmpty(detailList) && detailList.size()>0){
            detailList.forEach(s->s.setRecId(entity.getId()));
            recDetailManager.saveBatch(detailList);
        }

        return entity.getId();
    }

    @Override
    @Transactional
    public void updateStatus(String ids, Integer status) {
        List<String> idList = Arrays.asList(ids.split(","));
        LambdaUpdateWrapper<BizReconciliation> uw = new LambdaUpdateWrapper<>();
        uw.set(BizReconciliation::getStatus,status)
                .in(BizReconciliation::getId,idList)
                .set(BizReconciliation::getIsdCreatedate,LocalDateTime.now());

        if(status==4 || status==5){
            //确认对账
            IUser user = ContextUtil.getCurrentUser();
            uw.set(BizReconciliation::getVendorUserId,user.getUserId())
                    .set(BizReconciliation::getVendorUserAccount,user.getAccount())
                    .set(BizReconciliation::getVendorUserName,user.getFullname())
                    .set(BizReconciliation::getVendorPhone,user.getMobile());
        }

        super.update(uw);
    }

    @Override
    @Transactional
    public void pushToZfpt(List<String> ids) {
        List<BizReconciliation> recList = super.listByIds(ids);

        //生成支付平台对账单
        for(BizReconciliation rec : recList){
            if(rec.getStatus()!=4) {
                throw new BaseException("对账单确认后才可推送：单号【"+rec.getIsdCode()+"】");
            }

            rec.setIsdDetail(recDetailManager.getByPid(rec.getId()));

            ForeignResponseVo recResult = foreignCallService.callInterface(purIsdUrl, JSONUtil.parseObj(rec), 2);
            if(BeanUtils.isEmpty(recResult)) {
                throw new BaseException("调用[支付平台]对账单导入失败");
            }
            if(!recResult.getCode().equals("0")) {
                throw new BaseException("调用[支付平台]对账单导入失败:"+ recResult.getMsg());
            }
            rec.setSyncStatus(1);
            super.updateById(rec);
        }

    }


}
