package com.artfess.cgpt.project.manager.impl;

import com.artfess.base.exception.BaseException;
import com.artfess.base.query.*;
import com.artfess.base.util.BeanUtils;
import com.artfess.cgpt.bidding.manager.BizBiddingQuotationManager;
import com.artfess.cgpt.bidding.model.BizBiddingQuotation;
import com.artfess.cgpt.contract.manager.BizContractManager;
import com.artfess.cgpt.order.manager.BizRegistrationOrderManager;
import com.artfess.cgpt.project.manager.ProjectApprovalManager;
import com.artfess.cgpt.project.manager.UserParticipationQualificationLabelManager;
import com.artfess.cgpt.project.model.ProjectApproval;
import com.artfess.cgpt.project.model.UserParticipation;
import com.artfess.cgpt.project.dao.UserParticipationDao;
import com.artfess.cgpt.project.manager.UserParticipationManager;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.cgpt.project.model.UserParticipationQualificationLabel;
import com.artfess.cgpt.purchasing.manager.ApprovalInviteVendorManager;
import com.artfess.cgpt.purchasing.manager.MatApprovalManager;
import com.artfess.cgpt.purchasing.manager.MatApprovalQualificationManager;
import com.artfess.cgpt.purchasing.model.ApprovalInviteVendor;
import com.artfess.cgpt.purchasing.model.MatApproval;
import com.artfess.cgpt.purchasing.model.MatApprovalQualification;
import com.artfess.cgpt.qualification.manager.BizQualificationLabelManager;
import com.artfess.cgpt.qualification.model.BizQualificationLabel;
import com.artfess.cgpt.supplier.manager.BizSupplierEnterpriseManager;
import com.artfess.cgpt.supplier.model.AdmittedSuppliersQualification;
import com.artfess.cgpt.supplier.model.BizSupplierEnterprise;
import com.artfess.cgpt.supplier.model.BizSupplierOrgQualificationLabel;
import com.artfess.cgpt.universal.model.Accessory;
import com.artfess.cgpt.universal.service.AccessoryService;
import com.artfess.uc.api.impl.util.ContextUtil;
import com.artfess.uc.api.model.IGroup;
import com.artfess.uc.api.model.IUser;
import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.lang.reflect.InvocationTargetException;
import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 项目-用户参与表 服务实现类
 *
 * @company 阿特菲斯信息技术有限公司
 * @author 管理员
 * @since 2024-01-17
 */
@Service
public class UserParticipationManagerImpl extends BaseManagerImpl<UserParticipationDao, UserParticipation> implements UserParticipationManager {

    @Autowired
    private AccessoryService accessoryService;
    @Autowired
    private UserParticipationQualificationLabelManager userParticipationQualificationLabelManager;
    @Autowired
    private MatApprovalManager matApprovalManager;
    @Autowired
    private ApprovalInviteVendorManager approvalInviteVendorManager;
    @Autowired
    private BizSupplierEnterpriseManager supplierEnterpriseManager;
    @Autowired
    private BizBiddingQuotationManager quotationManager;
    @Autowired
    private BizContractManager contractManager;
    @Autowired
    private BizRegistrationOrderManager registrationOrderManager;
    @Autowired
    private ProjectApprovalManager projectApprovalManager;
    @Autowired
    private BizQualificationLabelManager qualificationLabelManager;
    @Autowired
    private MatApprovalQualificationManager approvalQualificationManager;

    @Override
    public List<UserParticipation> getNoRefundData() {
        List<UserParticipation> result = baseMapper.getNoRefundData();
        return result;
    }

    @Override
    public List<UserParticipation> queryList(Wrapper<UserParticipation> wrapper) {
        List<UserParticipation> result = baseMapper.queryList(wrapper);
        return result;
    }

    @Override
    public PageList<UserParticipation> queryAllByPage(QueryFilter<UserParticipation> queryFilter) {
        queryFilter.addFilter("IS_DELE_","0", QueryOP.EQUAL);
        IPage<UserParticipation> result = baseMapper.queryAllByPage(convert2IPage(queryFilter.getPageBean()),
                convert2Wrapper(queryFilter, currentModelClass()));
        return new PageList<>(result);
    }

    @Override
    public UserParticipation getDataById(String id) {
        UserParticipation result = super.getById(id);

        //查询资质类型关联表
        LambdaQueryWrapper<MatApprovalQualification> qualificationLabelQW = new LambdaQueryWrapper<>();
        qualificationLabelQW.eq(MatApprovalQualification::getProjectId,result.getNoticeId());
        result.setMatApprovalQualificationList(approvalQualificationManager.list(qualificationLabelQW));

        //查询关联资质库
        QueryFilter<UserParticipationQualificationLabel> qualificationQF = QueryFilter.build();
        qualificationQF.setPageBean(new PageBean(0,-1,false));
        qualificationQF.addFilter("a.USER_PARTICIPATION_ID_",id,QueryOP.EQUAL);
        result.setQualificationLabelList(userParticipationQualificationLabelManager.queryQualificationPage(qualificationQF).getRows());

        //查询附件
        if(BeanUtils.isNotEmpty(result.getQualificationLabelList()) && result.getQualificationLabelList().size()>0){
            result.getQualificationLabelList().forEach(s->{
                s.setAccessoryList(accessoryService.getAccessoryBySourceId(s.getId()));
            });
        }

        return result;
    }

    @Override
    public PageList<UserParticipation> queryByPage(QueryFilter<UserParticipation> queryFilter) {
        queryFilter.addFilter("IS_DELE_","0", QueryOP.EQUAL);
        queryFilter.addFilter("COMPANY_ID_", ContextUtil.getCurrentOrgId(),QueryOP.EQUAL);
        IPage<UserParticipation> result = baseMapper.queryAllByPage(convert2IPage(queryFilter.getPageBean()),
                convert2Wrapper(queryFilter, currentModelClass()));
        return new PageList<>(result);
    }

    @Override
    public PageList<UserParticipation> pageRegistraAndBid(QueryFilter<UserParticipation> queryFilter) {
        Integer round = 1;
        List<QueryField> querys = queryFilter.getQuerys();
        if(BeanUtils.isNotEmpty(querys) && querys.size()>0){
            for(QueryField qf : querys){
                if(qf.getProperty().equals("round")){
                    round = (Integer) qf.getValue();
                    qf.setProperty("1");
                    qf.setValue("1");
                    break;
                }
            }
        }

        queryFilter.addFilter("IS_DELE_","0", QueryOP.EQUAL);
        IPage<UserParticipation> result = baseMapper.queryAllByPage(convert2IPage(queryFilter.getPageBean()),
                convert2Wrapper(queryFilter, currentModelClass()));

        List<UserParticipation> records = result.getRecords();
        if(BeanUtils.isNotEmpty(records) && records.size()>0){
            for(UserParticipation userPart : records){
                //查询报价表是否已报价
                LambdaQueryWrapper<BizBiddingQuotation> quoQW = new LambdaQueryWrapper<>();
                quoQW.eq(BizBiddingQuotation::getQuotationRounds,round)
                        .eq(BizBiddingQuotation::getNoticeId,userPart.getNoticeId())
                        .eq(BizBiddingQuotation::getNoticeType,userPart.getNoticeType())
                        .eq(BizBiddingQuotation::getQuotationCompanyId,userPart.getCompanyId());
                if(quotationManager.count(quoQW)>0){
                    userPart.setIsQuo(1);
                }else {
                    userPart.setIsQuo(0);
                }
            }
        }

        return new PageList<>(result);
    }

    @Override
    @Transactional
    public void participate(UserParticipation entity) {
        //校验是否有参与权限
        verifyPermissions(entity);

        if(entity.getNoticeType()==2){
            //项目
            ProjectApproval notice = projectApprovalManager.getById(entity.getNoticeId());
            if(notice.getHasQualification()==2){
                throw new BaseException("请刷新后重新点击参与并上传资质");
            }
        }else {
            //物资
            MatApproval notice = matApprovalManager.getById(entity.getNoticeId());
            if(notice.getHasQualification()==2){
                throw new BaseException("请刷新后重新点击参与并上传资质");
            }
        }

        if(BeanUtils.isEmpty(entity.getType())){
            entity.setType(1);
        }

        //查询参与表该公司有没有数据
        LambdaQueryWrapper<UserParticipation> hisQW = new LambdaQueryWrapper<>();
        hisQW.eq(UserParticipation::getCompanyId,ContextUtil.getCurrentOrgId())
                .eq(UserParticipation::getNoticeId,entity.getNoticeId())
                .eq(UserParticipation::getNoticeType,entity.getNoticeType())
                .eq(UserParticipation::getType,entity.getType())
                .eq(UserParticipation::getIsDele,"0").last("limit 1");
        UserParticipation history = baseMapper.selectOne(hisQW);

        if(BeanUtils.isEmpty(history)){
            //新增
            IUser user = ContextUtil.getCurrentUser();
            entity.setUserId(user.getUserId());
            entity.setUserAccount(user.getAccount());
            entity.setUserName(user.getFullname());
            if(BeanUtils.isNotEmpty(user.getMobile())) {
                entity.setUserPhone(user.getMobile());
            }
            entity.setParticipationDate(LocalDateTime.now());
            IGroup org = ContextUtil.getCurrentGroup();
            if(BeanUtils.isNotEmpty(org)){
                entity.setCompanyId(org.getGroupId());
                entity.setCompanyCode(org.getGroupCode());
                entity.setCompanyName(org.getName());
            }

            if(BeanUtils.isNotEmpty(entity.getDocumentFee()) && entity.getDocumentFee().compareTo(BigDecimal.ZERO)>0){
                //如果有标书费，生成标书费记录
                UserParticipation docFeeData = new UserParticipation();
                try {
                    BeanUtils.copyNotNullProperties(docFeeData,entity);
                } catch (ReflectiveOperationException e) {
                    e.printStackTrace();
                    throw new BaseException("生成标书费记录失败："+e.getMessage());
                }
                docFeeData.setStatus(3);
                docFeeData.setPrice(entity.getDocumentFee());
                docFeeData.setType(2);
                super.save(docFeeData);
                entity.setDocumentFeeId(docFeeData.getId());

                if(BeanUtils.isEmpty(entity.getPrice()) || entity.getPrice().compareTo(BigDecimal.ZERO)==0){
                    entity.setStatus(4);
                }

            }

            super.save(entity);

        }else {
            //更新
            history.setNoticeTitle(entity.getNoticeTitle());
            history.setStatus(entity.getStatus());
            super.updateById(history);
        }

    }

    @Override
    @Transactional
    public String qualificationsParticipate(UserParticipation entity) {
        //校验是否有参与权限
        verifyPermissions(entity);

        if(BeanUtils.isEmpty(entity.getType())){
            entity.setType(1);
        }

        //设置状态为审核中
        entity.setStatus(2);
        List<BizQualificationLabel> qualificationLabelList = entity.getQualificationLabelList();
        List<Accessory> accessoryList = entity.getAccessoryList();

        //1.查询参与表该公司有没有数据
        LambdaQueryWrapper<UserParticipation> hisQW = new LambdaQueryWrapper<>();
        hisQW.eq(UserParticipation::getCompanyId,ContextUtil.getCurrentOrgId())
                .eq(UserParticipation::getNoticeId,entity.getNoticeId())
                .eq(UserParticipation::getNoticeType,entity.getNoticeType())
                .eq(UserParticipation::getType,entity.getType())
                .eq(UserParticipation::getIsDele,"0").last("limit 1");
        UserParticipation history = baseMapper.selectOne(hisQW);

        if(BeanUtils.isEmpty(history)){
            //新增
            IUser user = ContextUtil.getCurrentUser();
            entity.setUserId(user.getUserId());
            entity.setUserAccount(user.getAccount());
            entity.setUserName(user.getFullname());
            if(BeanUtils.isNotEmpty(user.getMobile())) {
                entity.setUserPhone(user.getMobile());
            }
            entity.setParticipationDate(LocalDateTime.now());
            IGroup org = ContextUtil.getCurrentGroup();
            if(BeanUtils.isNotEmpty(org)){
                entity.setCompanyId(org.getGroupId());
                entity.setCompanyCode(org.getGroupCode());
                entity.setCompanyName(org.getName());
            }

            if(BeanUtils.isNotEmpty(entity.getDocumentFee()) && entity.getDocumentFee().compareTo(BigDecimal.ZERO)>0){
                //如果有标书费，生成标书费记录
                UserParticipation docFeeData = new UserParticipation();
                try {
                    BeanUtils.copyNotNullProperties(docFeeData,entity);
                } catch (ReflectiveOperationException e) {
                    e.printStackTrace();
                    throw new BaseException("生成标书费记录失败："+e.getMessage());
                }
                docFeeData.setPrice(entity.getDocumentFee());
                docFeeData.setType(2);
                super.save(docFeeData);
                entity.setDocumentFeeId(docFeeData.getId());
            }
            super.save(entity);
        }else {
            //更新
            entity.setId(history.getId());
//            entity.setId(history.getId());
            history.setNoticeTitle(entity.getNoticeTitle());
            history.setStatus(entity.getStatus());
            super.updateById(history);

            //清除历史附件
            accessoryService.removeBySourceId(history.getId());

            //清除历史关联资质
            LambdaQueryWrapper<UserParticipationQualificationLabel> partQualificationLabelQW = new LambdaQueryWrapper<>();
            partQualificationLabelQW.eq(UserParticipationQualificationLabel::getUserParticipationId,entity.getId());
            userParticipationQualificationLabelManager.remove(partQualificationLabelQW);

        }

        if(BeanUtils.isNotEmpty(accessoryList) && accessoryList.size()>0){
            accessoryList.stream().forEach(s->s.setSourceId(entity.getId()));
            accessoryService.saveBatch(accessoryList);
        }

        if(BeanUtils.isNotEmpty(qualificationLabelList) && qualificationLabelList.size()>0){
            List<UserParticipationQualificationLabel> participationQualificationLabelList = new ArrayList<>();
            for(BizQualificationLabel label : qualificationLabelList){
                if(BeanUtils.isEmpty(label.getId())){
                    qualificationLabelManager.saveOrUpdateEntity(label);
                }
                UserParticipationQualificationLabel temp = new UserParticipationQualificationLabel();
                temp.setUserParticipationId(entity.getId());
                temp.setQualificationLabelId(label.getId());
                participationQualificationLabelList.add(temp);
            }
            userParticipationQualificationLabelManager.saveBatch(participationQualificationLabelList);
        }

        return entity.getId();
    }

    @Override
    @Transactional
    public void modifyStatus(String ids, String status) {
        if(status.equals("5")){
            //如果是取消

            //查询是否已支付，如果已支付就不能取消
            LambdaQueryWrapper<UserParticipation> participationQW = new LambdaQueryWrapper<>();
            participationQW.select(UserParticipation::getNoticeCode)
                    .in(UserParticipation::getId, Arrays.asList(ids.split(",")))
                    .eq(UserParticipation::getStatus,1)
                    .eq(UserParticipation::getType,1)
                    .gt(UserParticipation::getPrice,0)
                    .eq(UserParticipation::getIsDele,"0");
            List<Object> objectList = super.listObjs(participationQW);
            if(BeanUtils.isNotEmpty(objectList) && objectList.size()>0){
                List<String> collect = objectList.stream().map(s -> String.valueOf(s)).collect(Collectors.toList());
                throw new BaseException("取消失败：项目编号【"+String.join(",",collect)+"】已缴纳保证金，请等待成交后退还保证金");
            }

        }

        LambdaUpdateWrapper<UserParticipation> participationUW = new LambdaUpdateWrapper<>();
        participationUW.set(UserParticipation::getStatus,status)
                .in(UserParticipation::getId, Arrays.asList(ids.split(",")))
                .eq(UserParticipation::getIsDele,"0");
        super.update(participationUW);
    }

    @Override
    @Transactional
    public void modifyStatusByNoticeId(String noticeIds, String status) {
        if(status.equals("5")){
            //如果是取消

            //查询是否已支付，如果已支付就不能取消
            LambdaQueryWrapper<UserParticipation> participationQW = new LambdaQueryWrapper<>();
            participationQW.select(UserParticipation::getNoticeCode)
                    .eq(UserParticipation::getCompanyId,ContextUtil.getCurrentOrgId())
                    .in(UserParticipation::getNoticeId, Arrays.asList(noticeIds.split(",")))
                    .eq(UserParticipation::getStatus,1)
                    .eq(UserParticipation::getType,1)
                    .gt(UserParticipation::getPrice,0)
                    .eq(UserParticipation::getIsDele,"0");
            List<Object> objectList = super.listObjs(participationQW);
            if(BeanUtils.isNotEmpty(objectList) && objectList.size()>0){
                List<String> collect = objectList.stream().map(s -> String.valueOf(s)).collect(Collectors.toList());
                throw new BaseException("取消失败：项目编号【"+String.join(",",collect)+"】已缴纳保证金，请等待成交后退还保证金");
            }

        }

        LambdaUpdateWrapper<UserParticipation> participationUW = new LambdaUpdateWrapper<>();
        participationUW.set(UserParticipation::getStatus,status)
                .eq(UserParticipation::getCompanyId,ContextUtil.getCurrentOrgId())
                .in(UserParticipation::getNoticeId, Arrays.asList(noticeIds.split(",")))
                .eq(UserParticipation::getType,1)
                .eq(UserParticipation::getIsDele,"0");
        super.update(participationUW);

    }

    @Override
    public void refundWinBidBond() {
        //1.查询未退款的成交供应商参与数据
        List<UserParticipation> noRefundData = baseMapper.getNoRefundData();

        if(BeanUtils.isNotEmpty(noRefundData) && noRefundData.size()>0){
            for(UserParticipation userPart : noRefundData){
                //2.查询成交项目合同是否全部签订完成
                Integer noSignContrCount = contractManager.getNoSignContrCount(userPart.getNoticeId(),userPart.getCompanyId());

                if(noSignContrCount==0){
                    if(userPart.getNoticeType()==1){
                        //3.如果全部签订完成，退还保证金
                        MatApproval notice = matApprovalManager.getById(userPart.getNoticeId());
                        if(BeanUtils.isNotEmpty(notice.getBond()) && notice.getBond().compareTo(BigDecimal.ZERO)>0){
                            if(BeanUtils.isEmpty(userPart.getRegistrationOrderId())){
                                continue;
                            }
                            try {
                                //退还保证金
                                registrationOrderManager.refundBond(userPart);
                            }catch (Exception e){
                                e.printStackTrace();
                                System.out.println("退还已签订合同成交供应商的保证金接口执行失败："+e.getMessage());
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * 校验是否有参与权限
     * @param entity
     */
    void verifyPermissions(UserParticipation entity){
        Integer releaseMethod = 1;
        String procureOrgId = "";
        LocalDateTime responseEndTime = null;
        if(entity.getNoticeType()==2){
            //项目
            ProjectApproval notice = projectApprovalManager.getById(entity.getNoticeId());
            releaseMethod = notice.getReleaseMethod();
            procureOrgId = notice.getProcureOrgId();
            responseEndTime = notice.getResponseTimeDeadline();
        }else {
            //物资
            MatApproval notice = matApprovalManager.getById(entity.getNoticeId());
            releaseMethod = notice.getReleaseMethod();
            procureOrgId = notice.getProcureOrgId();
            responseEndTime = notice.getResponseTimeDeadline();
        }

        if(BeanUtils.isNotEmpty(responseEndTime)){
            if(responseEndTime.isBefore(LocalDateTime.now())){
                throw new BaseException("参与失败：响应时间已截止");
            }
        }

        if(releaseMethod==2){
            //如果是邀请，判断当前公司是否为邀请的供应商
            LambdaQueryWrapper<ApprovalInviteVendor> inviteVendorQW = new LambdaQueryWrapper<>();
            inviteVendorQW.eq(ApprovalInviteVendor::getProjectApprovalId,entity.getNoticeId())
                    .eq(ApprovalInviteVendor::getProjectApprovalType,entity.getNoticeType())
                    .eq(ApprovalInviteVendor::getVendorId,ContextUtil.getCurrentOrgId());
            if(approvalInviteVendorManager.count(inviteVendorQW)==0){
                throw new BaseException("该立项只允许被邀请的供应商才能参与");
            }
        }else {
            //如果是公开，判断当前公司是否为采购单位的准入供应商
//            LambdaQueryWrapper<BizSupplierEnterprise> supplierEnterpriseQW = new LambdaQueryWrapper<>();
//            supplierEnterpriseQW.eq(BizSupplierEnterprise::getEnterpriseId,procureOrgId)
//                    .eq(BizSupplierEnterprise::getSupplierId,ContextUtil.getCurrentOrgId())
//                    .eq(BizSupplierEnterprise::getReviewStatus,2)
//                    .eq(BizSupplierEnterprise::getIsDele,"0");
//            if(supplierEnterpriseManager.count(supplierEnterpriseQW)==0){
//                throw new BaseException("请申请成为采购单位的准入供应商或联系采购单位邀请成为准入供应商后重试");
//            }
        }
    }

}
