package com.artfess.portal.controller;

import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.feign.UCFeignService;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.FileUtil;
import com.artfess.base.util.HttpUtil;
import com.artfess.base.util.StringUtil;
import com.artfess.base.util.ZipUtil;
import com.artfess.base.util.time.DateFormatUtil;
import com.artfess.portal.model.PortalNewsNotice;
import com.artfess.portal.persistence.manager.PortalNewsNoticeManager;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 前端控制器
 *
 * @author 赵祥云
 * @company 阿特菲斯信息技术有限公司
 * @since 2020-07-03
 */
@Slf4j
@RestController
@Api(tags = "新闻公告表")
@ApiGroup(group = {ApiGroupConsts.GROUP_SYSTEM})
@RequestMapping("/portalNewsNotice/v1/")
public class PortalNewsNoticeController extends BaseController<PortalNewsNoticeManager, PortalNewsNotice> {

    @Resource
    UCFeignService ucFeignService;

    @PostMapping("/")
    @ApiOperation("添加新闻公告信息的接口")
    public CommonResult<String> create(@ApiParam(name="model", value="机组信息") @RequestBody PortalNewsNotice news) {
        boolean result = baseService.insertNews(news);
        if(!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
        return new CommonResult<>();
    }

    @GetMapping("/getNews")
    @ApiOperation("根据分类id查询实体")
    public List<PortalNewsNotice> getNews(@ApiParam(name = "classifyId", value = "分类id") @RequestParam String classifyId) {
        return baseService.getNews(classifyId);
    }

    @PostMapping(value = "/query", produces = {"application/json; charset=utf-8"})
    @ApiOperation("分页查询结果")
    public PageList<PortalNewsNotice> query(@ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<PortalNewsNotice> queryFilter) {
        PageList<PortalNewsNotice> list = baseService.query(queryFilter);
        Set<String> set = new HashSet<>();
        list.getRows().forEach(item -> {
            set.add(item.getCreateBy());
        });
        Map<String, String> map = new HashMap<>();
        if (BeanUtils.isNotEmpty(set) && set.size() > 0) {
            ArrayNode users = ucFeignService.getUserByIdsOrAccounts(StringUtil.join(set, ","));
            if (BeanUtils.isNotEmpty(users) && users.size() > 0) {
                users.forEach(item -> {
                    map.put(item.get("id").asText(), item.get("fullname").asText());
                });
            }
        }
        if (map.size() > 0) {
            list.getRows().forEach(item -> {
                if (map.containsKey(item.getCreateBy())) {
                    item.setCreateByName(map.get(item.getCreateBy()));
                }
            });
        }
        return list;
    }

    @GetMapping("/getNewsById")
    @ApiOperation("根据分类id查询实体")
    public PortalNewsNotice getNewsById(@ApiParam(name = "classifyId", value = "id") @RequestParam String id) {
        PortalNewsNotice portalNewsNotice = baseService.get(id);
        if (StringUtil.isNotEmpty(portalNewsNotice.getCreateBy())) {
            JsonNode createUser = ucFeignService.getUserById(portalNewsNotice.getCreateBy()).getValue();
            if (BeanUtils.isNotEmpty(createUser) && createUser.has("fullname")) {
                portalNewsNotice.setCreateByName(createUser.get("fullname").asText());
            }
        }
        if (StringUtil.isNotEmpty(portalNewsNotice.getUpdateBy())) {
            JsonNode updateUser = ucFeignService.getUserById(portalNewsNotice.getUpdateBy()).getValue();
            if (BeanUtils.isNotEmpty(updateUser) && updateUser.has("fullname")) {
                portalNewsNotice.setUpdateByName(updateUser.get("fullname").asText());
            }
        }
        return portalNewsNotice;
    }

    @PostMapping("exportNews")
    @ApiOperation(value = "根据id集合导出新闻公告", httpMethod = "POST", notes = "根据id集合导出新闻公告")
    public void exportNews(HttpServletResponse response, HttpServletRequest request, @RequestBody List<String> ids) throws Exception {
        String fileName = "newsNotice_" + DateFormatUtil.format(LocalDateTime.now(), "yyyy_MMdd_HHmm");
        String json = baseService.export(ids);
        HttpUtil.downLoadFile(request, response, json, "newsNotice.json", fileName);
    }

    @PostMapping("importNews")
    @ApiOperation(value = "导入新闻公告", httpMethod = "POST", notes = "导入新闻公告")
    public CommonResult importNews(MultipartHttpServletRequest request, HttpServletResponse response, @RequestParam String classifyId) throws Exception {
        MultipartFile file = request.getFile("file");
        String uzPath = "";
        try {
            String rootRealPath = (FileUtil.getIoTmpdir() + "/attachFiles/unZip/").replace("/", File.separator);
            FileUtil.createFolder(rootRealPath, true);
            String name = file.getOriginalFilename();
            String fileDir = StringUtil.substringBeforeLast(name, ".");
            ZipUtil.unZipFile(file, rootRealPath); // 解压文件
            uzPath = rootRealPath + File.separator + fileDir; // 解压后文件的真正路径
            baseService.importFile(uzPath, classifyId);
            return new CommonResult<>(true, "导入成功");
        } catch (Exception e) {
            return new CommonResult(false, "导入失败：" + e.getMessage());
        } finally {
            if (StringUtil.isNotEmpty(uzPath)) {
                File file1 = new File(uzPath);
                if (file1.exists()) {
                    file1.delete();
                }
            }
        }
    }

    @Override
    @PostMapping(value = "/updateSequence")
    @ApiOperation(value = "修改排序号", httpMethod = "POST", notes = "修改排序号")
    public CommonResult<String> updateSequence(@ApiParam(name = "map", value = "key:新闻id,value:排序号", required = true) @RequestBody HashMap<String,Integer> map) throws Exception {
        baseService.updateSequence(map);
        return new CommonResult<>("修改成功！");
    }


}
