package com.artfess.examine.controller;


import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.context.BaseContext;
import com.artfess.base.controller.BaseController;
import com.artfess.base.enums.QuestionStateEnum;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.exception.RequiredException;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.util.AuthenticationUtil;
import com.artfess.data.vo.UserExamVo;
import com.artfess.examine.manager.ExamUserEvaluationManager;
import com.artfess.examine.manager.ExamUserRecordManager;
import com.artfess.examine.model.ExamUserEvaluation;
import com.artfess.examine.model.ExamUserEvaluationDetail;
import com.artfess.examine.model.ExamUserRecord;
import com.artfess.examine.vo.SettingYearVo;
import com.artfess.poi.util.ExcelUtils;
import com.artfess.poi.util.FileDownloadUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ClassPathResource;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;

/**
 * 年度考生综合评定表 前端控制器
 *
 * @author min.wu
 * @company 阿特菲斯信息技术有限公司
 * @since 2022-11-28
 */
@Slf4j
@RestController
@Api(tags = "基础数据-年度评定")
@RequestMapping("/exam/user/evaluation/")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class ExamUserEvaluationController extends BaseController<ExamUserEvaluationManager, ExamUserEvaluation> {

    @Autowired
    private ExamUserRecordManager userRecordManager;

    @Resource
    BaseContext baseContext;

    @PostMapping(value = "/detailPage", produces = {"application/json; charset=utf-8"})
    @ApiOperation("详情记录-从年度评定分页界面弹出，入参评定id e.id_,其他参数和我的试卷一样")
    public PageList<ExamUserRecord> detailPage(@ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<ExamUserRecord> queryFilter) {
        AtomicReference<Boolean> flag = new AtomicReference<>();
        queryFilter.getQuerys().forEach(queryField -> {
            //如果是模拟训练则只能查看自己创建的试卷
            if ("e.id_".equals(queryField.getProperty()) && null == queryField.getValue()) {
                queryField.setValue("-1");
            }
        });
        queryFilter.addFilter("b.paper_type_", "2", QueryOP.EQUAL);
        queryFilter.addFilter("s.status_", QuestionStateEnum.finish.getType(), QueryOP.EQUAL);

        return userRecordManager.detailPage(queryFilter);
    }

    @GetMapping(value = "/evaluationDetail/{id}", produces = {"application/json; charset=utf-8"})
    @ApiOperation("评定详情列表，年度上报（0：未上报、1：已上报）")
    public CommonResult evaluationDetail(@ApiParam(name = "id", value = "年度评定id") @PathVariable String id) {
        Map<String, Object> map = new HashMap<>();
        List<ExamUserEvaluationDetail> list = this.baseService.evaluationDetail(id);
        map.put("list", list);
        String reportStatus = this.baseService.getReportStatus();
        map.put("reportStatus", reportStatus);
        return CommonResult.success(map, "");

    }


    @Override
    @PostMapping(value = "/query", produces = {"application/json; charset=utf-8"})
    @ApiOperation("分页查询结果")
    public PageList<ExamUserEvaluation> query(@ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<ExamUserEvaluation> queryFilter) {
        return baseService.findByPage(queryFilter);
    }

    @PostMapping(value = "/myYearPage", produces = {"application/json; charset=utf-8"})
    @ApiOperation("我的年度评定")
    public PageList<ExamUserEvaluation> myYearPage(@ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<ExamUserEvaluation> queryFilter) {
        queryFilter.addFilter("u.id_", AuthenticationUtil.getCurrentUserId(), QueryOP.EQUAL);
        queryFilter.getQuerys().removeIf(s -> "s.paper_id_".equals(s.getProperty()));
        List<String> currentAndChildOrgIds = baseContext.getCurrentAndChildOrgIds();
        queryFilter.addFilter("e.create_org_id_", currentAndChildOrgIds, QueryOP.IN);
        return baseService.myYearPage(queryFilter);
    }

    @PostMapping("/addSubject")
    @ApiOperation("考生添加年度考核课目")
    public CommonResult<String> addSubject(@ApiParam(name = "model", value = "实体信息") @RequestBody ExamUserEvaluation t) {
        log.info("考生添加年度考核课目请求参数:{}", JSON.toJSONString(t));
        boolean result = baseService.addSubject(t);
        if (!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "更新实体失败");
        }
        return new CommonResult<>();
    }

    @PostMapping("/delSubject")
    @ApiOperation("考生删除年度考核课目")
    public CommonResult<String> delSubject(@ApiParam(name = "model", value = "实体信息") @RequestBody ExamUserEvaluation t) {
        log.info("考生删除年度考核课目请求参数:{}", JSON.toJSONString(t));
        boolean result = baseService.delSubject(t);
        if (!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "更新实体失败");
        }
        return new CommonResult<>();
    }

    @PostMapping("/settingYear")
    @ApiOperation("设置年度最终成绩")
    public CommonResult<String> settingYear(@ApiParam(name = "model", value = "实体信息") @RequestBody SettingYearVo t) {
        log.info("设置年度最终成绩请求参数:{}", JSON.toJSONString(t));
        boolean result = baseService.settingYear(t);
        if (!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "更新实体失败");
        }
        return new CommonResult<>();
    }

    /**
     * todo
     * @param file
     * @param orgId
     * @return
     */
    @ApiOperation(value = "成绩导入")
    @PostMapping("/importExcel")
    public CommonResult<String> importExcel(@RequestParam("file") MultipartFile file,@RequestParam(value = "orgId", required = false) String orgId) {
        try {
            ExcelUtils<ExamUserEvaluationDetail> excelUtil = new ExcelUtils<>(ExamUserEvaluationDetail.class);
            List<ExamUserEvaluationDetail> list = excelUtil.importExcel(null,file.getInputStream());
            boolean result = baseService.importExcel(list, orgId);
            if(!result) {
                throw new IllegalArgumentException("导入失败");
            }
            return new CommonResult<>();
        } catch (Exception e) {
            throw new IllegalArgumentException("导入失败," + e.getMessage());
        }
    }

    /**
     * 下载导入模板
     *
     * @param response
     * @return
     */
    @ApiOperation(value = "下载导入模板")
    @GetMapping("/downModel")
    public void downTemplate(HttpServletResponse response) {
        try {
            ClassPathResource classPathResource = new ClassPathResource("model/userEvaluationDetail.xlsx");
            FileDownloadUtil.fileDownload(response, classPathResource.getInputStream(), "线下成绩导入模板.xlsx");
        } catch (Exception e) {
            response.setCharacterEncoding("utf-8");
            throw new RequiredException("你所下载的资源不存在");
        }
    }

    @ApiOperation(value = "成绩导出")
    @GetMapping("/export")
    public void export(HttpServletRequest request,HttpServletResponse response, @ApiParam(name = "model", value = "实体信息") @RequestParam(required = false) String orgId) throws Exception {
        String date = new SimpleDateFormat("yyyyMMddHHmmss").format(new Date());
        String fileName = String.format("年度成绩信息-%s", date);
        baseService.export(orgId, response,fileName);
    }


    @PostMapping("/batchSettingYear")
    @ApiOperation("批量设置年度最终成绩")
    public CommonResult<String> batchSettingYear(@ApiParam(name = "model", value = "实体信息") @RequestBody SettingYearVo t) {
        log.info("设置年度最终成绩请求参数:{}", JSON.toJSONString(t));
        boolean result = baseService.batchSettingYear(t);
        if (!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "更新实体失败");
        }
        return new CommonResult<>();
    }

    @PostMapping(value = "/getSubjectInfo", produces = {"application/json; charset=utf-8"})
    @ApiOperation("获取当年该岗位要考核的课目（传岗位id就可以）")
    public CommonResult getSubjectInfo(@ApiParam(name = "model", value = "实体信息") @RequestBody SettingYearVo t) {
        Assert.hasText(t.getPositionId(), "请先选择岗位");
        List<JSONObject> list = this.baseService.getSubjectInfo(t);
        return CommonResult.success(list, "");
    }

    @PostMapping(value = "/getPositionPaper", produces = {"application/json; charset=utf-8"})
    @ApiOperation("获取当前年度该课目下所有年度考核所有试卷（传课目id就可以）")
    public CommonResult getPositionPaper(@ApiParam(name = "model", value = "实体信息") @RequestBody SettingYearVo t) {
        Assert.hasText(t.getSubjectId(), "请先选择课目");
        List<JSONObject> list = this.baseService.getPositionPaper(t);
        return CommonResult.success(list, "");
    }

    @PostMapping(value = "/userQualityAnalyse", produces = {"application/json; charset=utf-8"})
    @ApiOperation("人员素质情况分析")
    public PageList<UserExamVo> userQualityAnalyse(@ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<ExamUserEvaluation> queryFilter) {
        return baseService.userQualityAnalyse(queryFilter);
    }

    /**
     * 设置考生职业教育分数以及总体评价
     */
    @PostMapping(value = "/setUserWorkEvaluation", produces = {"application/json; charset=utf-8"})
    @ApiOperation("设置考生职业教育分数以及总体评价")
    public CommonResult setUserWorkEvaluation(@ApiParam(name = "model", value = "实体信息") @RequestBody ExamUserEvaluation t) {
        log.info("设置考生职业教育分数以及总体评价请求参数:{}", JSON.toJSONString(t));
        boolean result = baseService.setUserWorkEvaluation(t);
        if (!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "更新实体失败");
        }
        return new CommonResult<>();
    }

}
