package com.artfess.dataShare.dataShare.manager.impl;

import com.artfess.base.exception.ApplicationException;
import com.artfess.base.util.BeanUtils;
import com.artfess.dataShare.dataApi.model.BizServiceCatalogs;
import com.artfess.dataShare.dataShare.manager.BizShareTableManager;
import com.artfess.dataShare.dataShare.model.BizShareCatalogs;
import com.artfess.dataShare.dataShare.dao.BizShareCatalogsDao;
import com.artfess.dataShare.dataShare.manager.BizShareCatalogsManager;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.dataShare.dataShare.model.BizShareTable;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.List;

/**
 * 数据资产-共享资源目录 服务实现类
 *
 * @company 阿特菲斯信息技术有限公司
 * @author 伍彬
 * @since 2024-12-03
 */
@Service
public class BizShareCatalogsManagerImpl extends BaseManagerImpl<BizShareCatalogsDao, BizShareCatalogs> implements BizShareCatalogsManager {

    @Resource
    private BizShareTableManager tableManager;

    @Override
    @Transactional
    public String upTree(BizShareCatalogs shareCatalogs){
        if (!validateCatalog(shareCatalogs)){
            throw new ApplicationException("目录名字【" + shareCatalogs.getName() + "】或目录编号【" + shareCatalogs.getCode() + "】重复！");
        }
        BizShareCatalogs old = this.get(shareCatalogs.getId());
        shareCatalogs.setHasChildren(old.getHasChildren());
        shareCatalogs.setIsDele(old.getIsDele());
        return this.updateTree(shareCatalogs,old.getName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean deleteById(String id) {
        BizShareCatalogs shareCatalogs = this.get(id);
        Integer count = this.getHasChildren(shareCatalogs);
        Assert.state(count == 0,"该目录下有子目录，删除失败");
        //判断该目录下有没有管理接口
        int tableCount = tableManager.count(new LambdaQueryWrapper<BizShareTable>()
                .eq(BizShareTable::getIsDele, "0")
                .eq(BizShareTable::getCatalogId, id));
        if (tableCount > 0){
            throw new ApplicationException("该目录下有资源管理，删除失败！");
        }
        boolean result = this.removeById(id);
        //更新父目录的子节点
        if(result){
            refreshHasChildren(shareCatalogs);
        }
        return result;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean removeTree(String id, String parentId) {
        boolean result = false;
        BizShareCatalogs oldCatalog = get(id);
        BizShareCatalogs oldParent = get(oldCatalog.getParentId());
        BizShareCatalogs newCatalog = get(id);
        newCatalog.setParentId(parentId);
        if (!validateCatalog(newCatalog)){
            throw new ApplicationException("目录名字【" + newCatalog.getName() + "】或目录编号【" + newCatalog.getCode() + "】重复！");
        }

        BizShareCatalogs newParent = get(parentId);
        newCatalog.buildFullIdAndName(newParent);
        List<BizShareCatalogs> children = baseMapper.selectList(new LambdaQueryWrapper<BizShareCatalogs>()
                .eq(BizShareCatalogs::getIsDele, "0")
                .eq(BizShareCatalogs::getParentId, newCatalog.getId()));
        for (BizShareCatalogs child : children) {
            child.buildFullIdAndName(newCatalog);
            updateById(child);
        }
        if (this.updateTree(newCatalog,oldCatalog.getName()) != null){
            oldParent.setHasChildren(oldParent.getHasChildren() - 1);
            result = updateById(oldParent);;
        }
        return result;
    }

    @Override
    public String saveTree(BizShareCatalogs shareCatalogs) {
        if (!validateCatalog(shareCatalogs)){
            throw new ApplicationException("目录名字【" + shareCatalogs.getName() + "】或目录编号【" + shareCatalogs.getCode() + "】重复！");
        }
        return this.insertTree(shareCatalogs);
    }

    /**
     * 验证重复目录
     * @param serviceCatalogs
     * @return
     */
    private boolean validateCatalog(BizShareCatalogs serviceCatalogs){
        LambdaQueryWrapper<BizShareCatalogs> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(BizShareCatalogs::getParentId,serviceCatalogs.getParentId());
        queryWrapper.and(w -> {
            w.eq(BizShareCatalogs::getName,serviceCatalogs.getName())
                    .or()
                    .eq(BizShareCatalogs::getCode,serviceCatalogs.getCode());
        });
        if (BeanUtils.isNotEmpty(serviceCatalogs.getId())){
            queryWrapper.ne(BizShareCatalogs::getId,serviceCatalogs.getId());
        }
        int count = count(queryWrapper);
        return count > 0 ? false : true;
    }

    /**
     * 更新父目录的子节点
     * @param shareCatalogs 当前目录
     */
    @Transactional(rollbackFor = Exception.class)
    public void refreshHasChildren(BizShareCatalogs shareCatalogs){
        //父目录
        BizShareCatalogs parent = this.get(shareCatalogs.getParentId());
        if (parent == null){
            return;
        }
        //获取子目录数量
        Integer hasChildren = getHasChildren(parent);
        parent.setHasChildren(hasChildren);
        //更新子目录
        this.update(parent);
        refreshHasChildren(parent);
    }

    /**
     * 获取子目录的数量
     * @param shareCatalogs
     * @return
     */
    private Integer getHasChildren(BizShareCatalogs shareCatalogs){
        LambdaQueryWrapper<BizShareCatalogs> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(BizShareCatalogs::getParentId,shareCatalogs.getId());
        Integer count = this.baseMapper.selectCount(queryWrapper);
        return count;
    }

}
