package com.artfess.dataShare.dataResource.ods.manager.impl;

import com.artfess.base.exception.ApplicationException;
import com.artfess.base.exception.BaseException;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.DataSourceUtil;
import com.artfess.base.util.StringUtil;
import com.artfess.dataShare.dataResource.ods.dao.BizOdsFieldDao;
import com.artfess.dataShare.dataResource.ods.manager.BizOdsFieldManager;
import com.artfess.dataShare.dataResource.ods.manager.BizOdsTableManager;
import com.artfess.dataShare.dataResource.ods.model.BizOdsField;
import com.artfess.dataShare.dataResource.ods.model.BizOdsTable;
import com.artfess.dataShare.dataResource.ods.vo.BizOdsFieldVo;
import com.artfess.dataShare.util.Constants;
import com.artfess.dataShare.util.DorisUtils;
import com.artfess.dataShare.util.FieldUtil;
import com.artfess.poi.util.ExcelTool;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 数据资源-- ODS字段信息 服务实现类
 *
 * @company 阿特菲斯信息技术有限公司
 * @author WH
 * @since 2024-11-13
 */
@Service
public class BizOdsFieldManagerImpl extends BaseManagerImpl<BizOdsFieldDao, BizOdsField> implements BizOdsFieldManager {

    @Resource
    BizOdsTableManager tableManager;

    @Resource
    private BizOdsFieldManager fieldManager;

    @Override
    public List<BizOdsField> queryFieldByTableId(String tableId) {
        QueryWrapper<BizOdsField> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("table_id_", tableId);
        queryWrapper.orderByAsc("sn_");
        return this.list(queryWrapper);
    }

    @Transactional(rollbackFor = Exception.class )
    public void export(MultipartFile file, String tableId) throws Exception{
        BizOdsTable table = tableManager.get(tableId);
        if(null==table){
            throw new ApplicationException("未查询到实体表");
        }
        String tableName=table.getTableNameEn();
        ExcelTool<BizOdsFieldVo> util = new ExcelTool<>(BizOdsFieldVo.class);
        List<BizOdsFieldVo> list = util.importExcel(file.getInputStream(),2,2);
        list = list.stream().filter(vo-> StringUtil.isNotEmpty(vo.getFieldName())).collect(Collectors.toList());

        List<BizOdsField> oldFields = fieldManager.queryFieldByTableId(tableId);
        Set<String> manageFields = new HashSet<>();
        if (BeanUtils.isNotEmpty(oldFields) && oldFields.size() > 0){
            manageFields = oldFields.stream()
                    .filter(entity -> "1".equals(entity.getFieldType()))
                    .map(entity -> entity.getFieldName())
                    .collect(Collectors.toSet());
        }

        for(int i=1;i<list.size();i++){
            BizOdsFieldVo vo = list.get(i);
            if (manageFields.contains(vo.getFieldName())){
                continue;
            }
            if(checkExistsField(tableId,vo.getFieldName())){
                throw new BaseException("表【"+table.getTableNameCh()+"中存在字段("+vo.getName()+")】请修改后在导入");
            }
        }
        List<String> disFieldList=list.stream().map(p->p.getFieldName()).distinct().collect(Collectors.toList());
        if(disFieldList.size()<list.size()){
            throw new BaseException("表"+table.getTableNameCh()+"中存在重复字段名请修改后在保存");
        }
        for(int i=1;i<list.size();i++){
            BizOdsFieldVo vo = list.get(i);
            if(StringUtil.isEmpty(vo.getFieldName())){
                continue;
            }
            if (manageFields.contains(vo.getFieldName())){
                continue;
            }
            BizOdsField field = vo.parse(vo);
            field.setTableId(tableId);
            //field.setDataType(vo.getDataType());
            //field.setCode(field.getName());
            //field.setDataType(FieldUtil.getColumnType(field.getDataType()));
            if("管理字段".equals(vo.getFieldType())){
                field.setFieldType("1");
            }else if("业务字段".equals(vo.getFieldType())){
                field.setFieldType("2");
            }
            //创建字段
            Integer attrLength=field.getAttrLength()==null ? 0 : field.getAttrLength();
            Integer decimalLen=field.getDecimalLen()==null ? 4 : field.getDecimalLen();
            String fieldSq="ALTER TABLE `"+tableName+"` ADD COLUMN `"+field.getFieldName()+"` "+ FieldUtil.getColumnType(field.getDataType(), attrLength, attrLength, decimalLen) +" NULL COMMENT '"+field.getDesc()+"' ";
            field.setDecimalLen(decimalLen);
            field.setFlSql(fieldSq);
            field.setCode(field.getFieldName());
            field.setMouldName(table.getName());
            field.setMouldCode(table.getCode());
            field.setIsDele("0");
            this.create(field);
        }


    }

    @Override
    public void saveOrUpdates(List<BizOdsField> field) {
        List<String> disFieldList=field.stream().map(p->p.getFieldName()).distinct().collect(Collectors.toList());
        if(disFieldList.size()<field.size()){
            throw new BaseException("表中存在重复字段名请修改后在保存");
        }
        String tableId = field.get(0).getTableId();
        BizOdsTable table = tableManager.get(tableId);
        if(null==table){
            throw new ApplicationException("未查询到实体表");
        }
        String tableName=table.getTableNameEn();
        List<BizOdsField> oldFields = this.queryFieldByTableId(tableId);
        if (BeanUtils.isNotEmpty(oldFields) && oldFields.size() > 0){
            removeByIds(oldFields.stream().filter(p->"2".equals(p.getFieldType())).map(entity -> entity.getId()).collect(Collectors.toList()));
        }
        //表里面没有字段的时候才加默认的管理字段
        if(oldFields.size()==0){
            addManageField(oldFields);
        }
        for(BizOdsField item : field){
            if(StringUtil.isEmpty(item.getId())){
                if(checkExistsField(tableId,item.getFieldName())){
                    throw new BaseException("表中存在字段("+item.getFieldName()+")请修改后在导入");
                }
            }else{
                if(checkExistsField(tableId,item.getFieldName(),item.getId())){
                    throw new BaseException("表中存在字段("+item.getFieldName()+"请修改后在导入");
                }
            }
            item.setTableId(tableId);
            item.setFieldName(item.getCode());
            item.setId(null);
            Integer attrLength=item.getAttrLength()==null ? 0 : item.getAttrLength();
            Integer decimalLen=item.getDecimalLen()==null ? 4 : item.getDecimalLen();
            String fieldSq="ALTER TABLE `"+tableName+"` ADD COLUMN `"+item.getFieldName()+"` "+ FieldUtil.getColumnType(item.getDataType(), attrLength, attrLength, decimalLen) +" NULL COMMENT '"+item.getDesc()+
                    "' ";
            item.setFlSql(fieldSq);
        }
        if(field.size()>0) {
            this.saveBatch(field);
        }
    }

    private void addManageField(List<BizOdsField> fieldList){
        BizOdsField createBy = new BizOdsField();
        createBy.setFieldName("CREATE_BY_");
        createBy.setName("创建人");
        createBy.setCode("CREATE_BY_");
        createBy.setDesc("创建人");
        createBy.setDataType("varchar");
        createBy.setAttrLength(200);
        createBy.setIsPk("0");
        createBy.setIsRequired("0");
        createBy.setIsEncrypt("0");
        createBy.setDecimalLen(0);
        createBy.setFieldType("1");
        createBy.setIsDele("0");

        BizOdsField createOrgId = new BizOdsField();
        createOrgId.setFieldName("CREATE_ORG_ID_");
        createOrgId.setName("创建单位");
        createOrgId.setCode("CREATE_ORG_ID_");
        createOrgId.setDesc("创建单位");
        createOrgId.setDataType("varchar");
        createOrgId.setAttrLength(200);
        createOrgId.setIsPk("0");
        createOrgId.setIsRequired("0");
        createOrgId.setIsEncrypt("0");
        createOrgId.setDecimalLen(0);
        createOrgId.setFieldType("1");
        createOrgId.setIsDele("0");

        BizOdsField createTime = new BizOdsField();
        createTime.setFieldName("CREATE_TIME_");
        createTime.setName("创建时间");
        createTime.setCode("CREATE_TIME_");
        createTime.setDesc("创建时间");
        createTime.setDataType("date");
        createTime.setAttrLength(0);
        createTime.setIsPk("0");
        createTime.setIsRequired("0");
        createTime.setIsEncrypt("0");
        createTime.setDecimalLen(0);
        createTime.setFieldType("1");
        createTime.setIsDele("0");

        BizOdsField id = new BizOdsField();
        id.setFieldName("ID_");
        id.setName("主键");
        id.setCode("ID_");
        id.setDesc("主键");
        id.setDataType("varchar");
        id.setAttrLength(500);
        id.setIsPk("1");
        id.setIsRequired("0");
        id.setIsEncrypt("0");
        id.setDecimalLen(0);
        id.setFieldType("1");
        id.setIsDele("0");
        id.setSn(-1);

        Set<String> set = fieldList.stream().map(entity -> entity.getCode()).collect(Collectors.toSet());
        if (!set.contains(createBy.getCode())){
            fieldList.add(createBy);
        }
        if (!set.contains(createOrgId.getCode())){
            fieldList.add(createOrgId);
        }
        if (!set.contains(createTime.getCode())){
            fieldList.add(createTime);
        }
        if (!set.contains(id.getCode())){
            fieldList.add(id);
        }
    }

    public boolean checkExistsField(String tableId,String fieldName){
        QueryWrapper<BizOdsField> queryWrapper=new QueryWrapper<>();
        queryWrapper.eq("TABLE_ID_",tableId);
        queryWrapper.eq("FIELD_NAME_",fieldName);
        int count =this.count(queryWrapper);
        if(count>0){
            return true;
        }
        return false;
    }

    public boolean checkExistsField(String tableId,String fieldName,String id){
        QueryWrapper<BizOdsField> queryWrapper=new QueryWrapper<>();
        queryWrapper.eq("TABLE_ID_",tableId);
        queryWrapper.eq("FIELD_NAME_",fieldName);
        queryWrapper.ne("ID_",id);
        int count =this.count(queryWrapper);
        if(count>0){
            return true;
        }
        return false;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveOrUpdateFields(List<BizOdsField> fields) throws Exception {
        if(fields.size()==0){
            throw  new ApplicationException("没有修改的字段");
        }
        String tableId = fields.get(0).getTableId();
        BizOdsTable table = tableManager.get(tableId);
        List<String> disFieldList=fields.stream().map(p->p.getCode()).distinct().collect(Collectors.toList());
        if(disFieldList.size()<fields.size()){
            throw new BaseException("表中存在重复字段名请修改后在保存");
        }
        String tableName=table.getTableNameEn();
        JdbcTemplate template = DataSourceUtil.getJdbcTempByDsAlias(Constants.ODS_DATA_SOURCE_ALIAS);

        for(BizOdsField field : fields){
            field.setIsCreate("1");
            //创建字段
            Integer attrLength=field.getAttrLength()==null ? 0 : field.getAttrLength();
            Integer decimalLen=field.getDecimalLen()==null ? 4 : field.getDecimalLen();
            field.setDecimalLen(decimalLen);
            field.setFieldName(field.getCode());
            //新镇字段
            if(StringUtil.isEmpty(field.getId())){
                field.setMouldCode(fields.get(0).getMouldCode());
                field.setMouldName(fields.get(0).getMouldName());
                if(field.getSn()==null){
                    Map<String,Object> map=new HashMap<>();
                    map.put("TABLE_ID_",tableId);
                    field.setSn(this.getNextSequence(map));
                }
                String fieldSq="ALTER TABLE `"+tableName+"` ADD COLUMN `"+field.getFieldName()+"` "+ FieldUtil.getColumnType(field.getDataType(), attrLength, attrLength, decimalLen) +" NULL COMMENT '"+field.getDesc()+"' ";
                field.setFlSql(fieldSq);
                template.execute(fieldSq);
            }else if(fieldEquals(field)){
                BizOdsField oldField = this.getById(field.getId());
                //修改字段名doris与mysql语法不一样，ODS，DWD，APP都是doris
                if(!field.getFieldName().equals(oldField.getFieldName())){
                    //ALTER TABLE app_user_info CHANGE COLUMN `USER_PHONE_` `USER_PHONE_1_` varchar COMMENT '移动手机号';
                    String fieldSq="ALTER TABLE `"+tableName+"` RENAME COLUMN `"+oldField.getFieldName()+"` `"+ field.getFieldName() +"` ";
       /*             String fieldSq="ALTER TABLE `"+tableName+"` CHANGE `"+oldField.getFieldName()+"`  `"+ field.getFieldName() +"` "+FieldUtil.getColumnType(field.getDataType(), attrLength, attrLength,
                            decimalLen)+" DEFAULT NULL COMMENT '"+field.getDesc()+"' ";*/
                    template.execute(fieldSq);
                }
                //
                if(!field.getDesc().equals(oldField.getDesc())){
                    //ALTER TABLE app_user_info MODIFY COLUMN `USER_PHONE_` varchar COMMENT '移动手机号';
                    String fieldSq="ALTER TABLE `"+tableName+"` MODIFY COLUMN `"+field.getFieldName()+"`  COMMENT '"+field.getDesc()+"' ";
                    template.execute(fieldSq);
                }
            }
        }
        this.saveOrUpdateBatch(fields);
        //更新建表语句
        String createSql = DorisUtils.getOdsTableSql(table.getTableNameEn(),table.getTableDesc(), fields);
        table.setCreateTableSql(createSql);
        table.setFieldTotal(fields.size());
        this.tableManager.updateById(table);
    }

    //判断传过来的字段是不是修改了
    public boolean fieldEquals(BizOdsField field){
        BizOdsField oldField = this.getById(field.getId());
        if(!field.getFieldName().equals(oldField.getFieldName())
                || !field.getDesc().equals(oldField.getDesc())
        ){
            return true;
        }
        return false;
    }
}
