package com.artfess.device.base.controller;


import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.model.CommonResult;
import com.artfess.base.valid.AddGroup;
import com.artfess.base.valid.UpdateGroup;
import com.artfess.device.base.manager.DeviceBaseCompanyManager;
import com.artfess.device.base.manager.DeviceBaseParamsConfManager;
import com.artfess.device.base.manager.DeviceBaseUserManager;
import com.artfess.device.base.model.DeviceBaseCompany;
import com.artfess.device.base.model.DeviceBaseUser;
import com.artfess.device.base.vo.CompanyAndUserVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Arrays;
import java.util.List;

/**
 * 对单位（管理单位、运维单位、设备厂商、供应商）信息进行管理 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author min.wu
 * @since 2022-07-12
 */
@RestController
@Slf4j
@Api(tags = "设施设备-单位管理")
@RequestMapping("/device/base/company/")
@ApiGroup(group = {ApiGroupConsts.DEVICE_BIZ})
public class DeviceBaseCompanyController extends BaseController<DeviceBaseCompanyManager, DeviceBaseCompany> {

    @Autowired
    private DeviceBaseParamsConfManager deviceBaseParamsConfManager;

    @Autowired
    private DeviceBaseUserManager deviceBaseUserManager;

    @PostMapping(value="/findAll", produces={"application/json; charset=utf-8" })
    @ApiOperation("获取单位列表（不分页）")
    public List<DeviceBaseCompany> findAll(@ApiParam(name="model", value="查询条件") @RequestBody DeviceBaseCompany company) {
        return baseService.findAll(company);
    }

    @Override
    @DeleteMapping("/{id}")
    @ApiOperation("根据id删除")
    public CommonResult<String> deleteById(@ApiParam(name="id", value="实体id") @PathVariable String id) {
        List<DeviceBaseUser> list = deviceBaseUserManager.findByCompanyIds(Arrays.asList(id));
        if(!CollectionUtils.isEmpty(list)){
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION.getCode(), "该单位下还有人员，请先删除人员");
        }
        boolean result = baseService.removeById(id);
        if(!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "删除实体失败");
        }
        return new CommonResult<>();
    }

    @Override
    @DeleteMapping("/")
    @ApiOperation("根据id集合批量删除")
    public CommonResult<String> deleteByIds(@ApiParam(name="ids", value="实体集合") @RequestParam String...ids) {
        List<DeviceBaseUser> list = deviceBaseUserManager.findByCompanyIds(Arrays.asList(ids));
        if(!CollectionUtils.isEmpty(list)){
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION.getCode(), "该单位下还有人员，请先删除人员");
        }
        boolean result = baseService.removeByIds(Arrays.asList(ids));
        if(!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "删除实体失败");
        }
        return new CommonResult<>();
    }

    @Override
    @PostMapping("/")
    @ApiOperation("添加实体的接口")
    public CommonResult<String> create(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({AddGroup.class}) DeviceBaseCompany t) {
        boolean result = baseService.save(t);
        if(!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
        deviceBaseParamsConfManager.saveParamsValue(t.getParamsValueList(), t.getId());
        return new CommonResult<>();
    }


    @Override
    @PutMapping("/")
    @ApiOperation("更新实体")
    public CommonResult<String> updateById(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({UpdateGroup.class})  DeviceBaseCompany t) {
        boolean result = baseService.updateById(t);
        if(!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "更新实体失败");
        }
        deviceBaseParamsConfManager.saveParamsValue(t.getParamsValueList(), t.getId());
        return new CommonResult<>();
    }

    @PostMapping(value="/getCompanyAndUser", produces={"application/json; charset=utf-8" })
    @ApiOperation(value = "获取单位人员2级下拉", response = CompanyAndUserVo.class)
    public List<CompanyAndUserVo> getCompanyAndUser(@ApiParam(name="model", value="查询条件") @RequestBody DeviceBaseCompany company) {
        return baseService.getCompanyAndUser(company);
    }

}
