package net.zstong.awump.device.web;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.extern.slf4j.Slf4j;
import net.zstong.awump.base.service.AwIntensiveFrameService;
import net.zstong.awump.device.repository.DeviceInsanceRepository;
import net.zstong.awump.device.web.response.CountDeviceData;
import net.zstong.awump.device.web.response.DeviceInfo;
import org.hswebframework.web.api.crud.entity.QueryNoPagingOperation;
import org.hswebframework.web.authorization.annotation.Authorize;
import org.hswebframework.web.authorization.annotation.Resource;
import org.jetlinks.community.device.service.LocalDeviceInstanceService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import java.util.Comparator;
import java.util.Map;

@RestController
@RequestMapping({"/aw/dev/overview"})
@Resource(id = "aw-dev-overview", name = "设备监测接口")
@Tag(name = "设备监测接口")
@Slf4j
public class DevViewController {

    @Autowired
    private LocalDeviceInstanceService deviceInstanceService;

    @Autowired
    private AwIntensiveFrameService awIntensiveFrameService;


    @GetMapping("/{productId}/{deviceId}")
    @QueryNoPagingOperation(summary = "获取设备信息")
    @Authorize(merge = false)
    public Mono<DeviceInfo> getDeviceInfo(@PathVariable @Parameter(description = "产品分类") String productId,
                                          @PathVariable @Parameter(description = "设备ID") String deviceId) {
        return Mono.just(productId).flatMap(p -> {
            if ("MJJ".equals(p)) {
                return awIntensiveFrameService
                    .createQuery()
                    .where("id", deviceId)
                    .fetchOne()
                    .map(d -> DeviceInfo.of(d));
            } else {
                return this.deviceInstanceService
                    .createQuery()
                    .where("id", deviceId)
                    .fetchOne()
                    .map(d -> DeviceInfo.of(d));
            }
        });
    }

    @Autowired
    private DeviceInsanceRepository deviceInsanceRepository;

    @PostMapping("/count")
    @Operation(summary = "统计设备情况")
    @Authorize(merge = false)
    public Flux<CountDeviceData> countDevice(@RequestBody Map params) {
        return deviceInsanceRepository.countByWarehouse().flatMap(w -> {
            if (w.getWarehouse() == null) {
                w.setWarehouse("unknow");
                w.setWarehouseName("未知");
                return deviceInsanceRepository.countByNonWarehouse().collectList().map(l -> {
                    w.setDetail(l);
                    return w;
                });
            } else {
                return deviceInsanceRepository.countByWarehouse(w.getWarehouse()).collectList().map(l -> {
                    w.setDetail(l);
                    return w;
                });
            }
        }).sort(Comparator.comparing(CountDeviceData::getWarehouseName))
            .mergeWith(Mono.just(new CountDeviceData("all", "全部", null, 0, null))
                .flatMap(a -> {
                    return deviceInsanceRepository.countByState().collectList().map(l -> {
                        a.setDetail(l);
                        l.forEach(x -> {
                            a.setTotal(a.getTotal() + x.getTotal());
                        });
                        return a;
                    });
                }));
    }

}
