export interface GetSystemInfoSuccessResult {
    /**
     * 用户设置字体大小
     */
    readonly fontSizeSetting: number;
    /**
     * 系统版本
     */
    readonly system: string;
    /**
     * 版本号
     */
    readonly version: string;
    /**
     * 手机品牌
     */
    readonly brand: string;
    /**
     * 当前电量百分比
     */
    readonly currentBattery: string;
    /**
     * 窗口高度
     */
    readonly windowHeight: number;
    /**
     * 设备像素比
     */
    readonly pixelRatio: number;
    /**
     * 系统名
     */
    readonly platform: 'Android' | 'iOS' | 'iPhone OS';
    /**
     * 屏幕高度
     */
    readonly screenHeight: number;
    /**
     * 状态栏高度
     */
    readonly statusBarHeight: number;
    /**
     * 设置的语言
     */
    readonly language: string;
    /**
     * 设备磁盘容量
     */
    readonly storage: string;
    /**
     * 当前运行的客户端，当前是支付宝则有效值是"alipay"
     */
    readonly app: string;
    /**
     * 标题栏高度
     */
    readonly titleBarHeight: number;
    /**
     * 手机型号
     */
    readonly model: string;
    /**
     * 屏幕宽度
     */
    readonly screenWidth: number;
    /**
     * 窗口宽度
     */
    readonly windowWidth: number;
}
export interface OpenBluetoothAdapterSuccessResult {
    /**
     * 是否支持 BLE
     */
    readonly isSupportBLE: boolean;
}
export interface OpenBluetoothAdapterFailResult {
    /**
     * - 12：蓝牙未打开
     * - 13：与系统服务的链接暂时丢失
     * - 14：未授权使用蓝牙功能
     * - 15：未知错误
     */
    readonly error: 12 | 13 | 14 | 15;
}
export interface StartBluetoothDevicesDiscoveryOptions {
    /**
     * 蓝牙设备主 service 的 uuid 列表
     */
    services?: string[];
    /**
     * 是否允许重复上报同一设备， 如果允许重复上报，则 onBluetoothDeviceFound 方法会多次上报同一设备，但是 RSSI 值会有不同
     */
    allowDuplicatesKey?: boolean;
    /**
     * 上报设备的间隔，默认为0，意思是找到新设备立即上报，否则根据传入的间隔上报
     */
    interval?: number;
    complete?(res: any): void;
}
export interface BluetoothDevice {
    /**
     * 蓝牙设备名称，某些设备可能没有
     */
    name: string;
    /**
     * 值与 name 一致
     *
     * @deprecated
     */
    deviceName: string;
    /**
     * 广播设备名称
     */
    localName: string;
    /**
     * 设备 Id
     */
    deviceId: string;
    /**
     * 设备信号强度
     */
    RSSI: number;
    /**
     * 设备的广播内容
     */
    advertisData: string;
    /**
     * 设备的manufacturerData
     */
    manufacturerData: string;
}
export interface GetBluetoothDevicesSuccessResult {
    /**
     * 已发现的设备列表
     */
    readonly devices: ReadonlyArray<Readonly<BluetoothDevice>>;
}
export interface WriteBLECharacteristicValueOptions {
    /**
     * 蓝牙设备 id，参考 device 对象
     */
    deviceId: string;
    /**
     * 蓝牙特征值对应 service 的 uuid
     */
    serviceId: string;
    /**
     * 蓝牙特征值的 uuid
     */
    characteristicId: string;
    /**
     * 蓝牙设备特征值对应的值，16进制字符串，限制在20字节内
     */
    value: string;
    complete?(res: any): void;
}
export interface ReadBLECharacteristicValueSuccessResult {
    /**
     * 设备特征值信息
     */
    readonly characteristic: {
        /**
         * 蓝牙设备特征值的 uuid
         */
        readonly characteristicId: string;
        /**
         * 蓝牙设备特征值对应服务的 uuid
         */
        readonly serviceId: string;
        /**
         * 蓝牙设备特征值的value
         */
        readonly value: string;
    };
}
export interface ReadBLECharacteristicValueOptions {
    /**
     * 蓝牙设备 id，参考 device 对象
     */
    deviceId: string;
    /**
     * 蓝牙特征值对应 service 的 uuid
     */
    serviceId: string;
    /**
     * 蓝牙特征值的 uuid
     */
    characteristicId: string;
    complete?(res: any): void;
}
export interface NotifyBLECharacteristicValueChangeOptions {
    /**
     * 蓝牙设备 id，参考 device 对象
     */
    deviceId: string;
    /**
     * 蓝牙特征值对应 service 的 uuid
     */
    serviceId: string;
    /**
     * 蓝牙特征值的 uuid
     */
    characteristicId: string;
    /**
     * notify 的 descriptor 的 uuid （只有android 会用到，非必填，默认值00002902-0000-10008000-00805f9b34fb）
     */
    descriptorId?: string;
    /**
     * 是否启用notify或indicate
     */
    state?: boolean;
    complete?(res: any): void;
}
export interface GetBLEDeviceServicesSuccessResult {
    /**
     * 设备service 对象列表，详见下表特征值信息
     */
    readonly services: ReadonlyArray<{
        /**
         * 蓝牙设备服务的 uuid
         */
        readonly serviceId: string;
        /**
         * 该服务是否为主服务
         */
        readonly isPrimary: boolean;
    }>;
}
export interface GetBLEDeviceCharacteristicsSuccessResult {
    /**
     * 设备特征值列
     */
    readonly characteristics: ReadonlyArray<{
        /**
         * 蓝牙设备特征值的 uuid
         */
        readonly characteristicId: string;
        /**
         * 蓝牙设备特征值对应服务的 uuid
         */
        readonly serviceId: string;
        /**
         * 蓝牙设备特征值对应的16进制值
         */
        readonly value: string;
        /**
         * 该特征值支持的操作类型
         */
        readonly properties: {
            /**
             * 该特征值是否支持 read 操作
             */
            readonly read: boolean;
            /**
             * 该特征值是否支持 write 操作
             */
            readonly write: boolean;
            /**
             * 该特征值是否支持 notify 操作
             */
            readonly notify: boolean;
            /**
             * 该特征值是否支持 indicate 操作
             */
            readonly indicate: boolean;
        };
    }>;
}
export interface GetBLEDeviceCharacteristicsOptions {
    /**
     * 蓝牙设备 id，参考 device 对象
     */
    deviceId: string;
    /**
     * 蓝牙特征值对应 service 的 uuid
     */
    serviceId: string;
    complete?(res: any): void;
}
export interface OnBLECharacteristicValueChangeEvent {
    /**
     * 蓝牙设备 id，参考 device 对象
     */
    readonly deviceId: string;
    /**
     * 蓝牙特征值对应 service 的 uuid
     */
    readonly serviceId: string;
    /**
     * 蓝牙特征值的 uuid
     */
    readonly characteristicId: string;
    /**
     * 特征值最新的16进制值
     */
    readonly value: string;
}
export default interface Device {
    getSystemInfo(options: {
        complete?(): void;
    }): Promise<GetSystemInfoSuccessResult>;
    getSystemInfoSync(): GetSystemInfoSuccessResult;
    setClipboard(options: {
        text: string;
        complete?(): void;
    }): Promise<{
        success: boolean;
    }>;
    getClipboard(options: {
        complete?(): void;
    }): Promise<{
        text: string;
    }>;
    /**
     * 初始化小程序蓝牙模块，生效周期为调用 [dd.openBluetoothAdapter](#dd.openBluetoothAdapter) 至调用 [dd.closeBluetoothAdapter](#dd.closeBluetoothAdapter) 或小程序被销毁为止。
     * 在小程序蓝牙适配器模块生效期间，开发者可以正常调用下面的小程序API，并会收到蓝牙模块相关的 on 事件回调。
     */
    openBluetoothAdapter(options?: {
        /**
         * 不传的话默认是true，表示是否在离开当前页面时自动断开蓝牙(仅对android有效)
         */
        autoClose?: boolean;
        complete?(res: OpenBluetoothAdapterSuccessResult | OpenBluetoothAdapterFailResult): void;
    }): Promise<OpenBluetoothAdapterSuccessResult>;
    /**
     * 关闭本机蓝牙模块。
     *
     * Bug & Tip
     * 1. tip: 调用该方法将断开所有已建立的蓝牙连接并释放系统资源。
     * 2. tip: 建议在结束小程序蓝牙流程时调用，与 dd.openBluetoothAdapter 成对调用。
     * 3. tip: 调用 dd.closeBluetoothAdapter 释放资源为异步操作，不建议使用 dd.closeBluetoothAdapter 和 dd.openBluetoothAdapter 作为异常处理流程（相当于先关闭再开启，重新初始化，效率低，易发生线程同步问题）
     */
    closeBluetoothAdapter(options?: {
        complete?(res: any): void;
    }): Promise<any>;
    getBluetoothAdapterState(options?: {
        complete?(res: any): void;
    }): Promise<{
        readonly available: boolean;
        readonly discovering: boolean;
    }>;
    /**
     * 开始搜寻附近的蓝牙外围设备。搜索结果将在 [dd.onBluetoothDeviceFound](#dd.onBluetoothDeviceFound) 事件中返回。
     *
     * Bug & Tip
     * - tip: 该操作比较耗费系统资源，请在搜索并连接到设备后调用 stop 方法停止搜索。
     */
    startBluetoothDevicesDiscovery(options?: StartBluetoothDevicesDiscoveryOptions): Promise<any>;
    /**
     * 停止搜寻附近的蓝牙外围设备。
     */
    stopBluetoothDevicesDiscovery(options?: {
        complete?(res: any): void;
    }): Promise<any>;
    /**
     * 获取所有已发现的蓝牙设备，包括已经和本机处于连接状态的设备。
     *
     * Bug & Tip
     * - tip: 模拟器可能无法获取 advertisData 及 RSSI，请使用真机调试
     * - tip: 开发者工具和 Android 上获取到的deviceId为设备 MAC 地址，iOS 上则为设备 uuid; 因此 deviceId 不能硬编码到代码中，需要分平台处理，iOS可根据设备属性（localName/advertisData/manufacturerData 等属性）进行动态匹配。
     */
    getBluetoothDevices(options?: {
        complete?(res: any): void;
    }): Promise<GetBluetoothDevicesSuccessResult>;
    /**
     * 获取处于已连接状态的设备。
     *
     * Bug & Tip
     * - tip: 如果传递的 services 为空，则返回所有的已经连接的设备。
     * - tip: Android 上获取到的deviceId为设备 MAC 地址，iOS 上则为设备 uuid。因此deviceId不能硬编码到代码中，需要区分处理。
     */
    getConnectedBluetoothDevices(options?: {
        services?: string[];
        complete?(res: any): void;
    }): Promise<{
        readonly devices: ReadonlyArray<Readonly<BluetoothDevice>>;
    }>;
    /**
     * 连接低功耗蓝牙设备。
     *
     * Bug & Tip
     * - tip: 若小程序在之前已有搜索过某个蓝牙设备，可直接传入之前搜索获取的 deviceId 直接尝试连接该设备，无需进行搜索操作。
     * - tip: 若指定的蓝牙设备已经连接，重复连接直接返回成功。
     */
    connectBLEDevice(options: {
        deviceId: string;
        complete?(res: any): void;
    }): Promise<any>;
    /**
     * 断开与低功耗蓝牙设备的连接。
     *
     * Bug & Tip
     * - tip: 蓝牙连接随时可能断开，建议监听 [dd.onBLEConnectionStateChanged](#dd.onBLEConnectionStateChanged) 回调事件，当蓝牙设备断开时按需执行重连操作
     * - tip: 若对未连接的设备或已断开连接的设备调用数据读写操作的接口，会返回10006错误，详见错误码，建议进行重连操作
     */
    disconnectBLEDevice(options: {
        deviceId: string;
        complete?(res: any): void;
    }): Promise<any>;
    /**
     * 向低功耗蓝牙设备特征值中写入数据。
     *
     * Bug & Tip
     * - tip: 设备的特征值必须支持 write 才可以成功调用，具体参照 characteristic 的 properties 属性。
     * - tip: 写入的二进制数据需要进行 hex 编码。
     */
    writeBLECharacteristicValue(options: WriteBLECharacteristicValueOptions): Promise<any>;
    /**
     * 读取低功耗蓝牙设备特征值中的数据。调用后在 [dd.onBLECharacteristicValueChange()](#dd.onBLECharacteristicValueChange) 事件中接收数据返回。
     *
     * Bug & Tip
     * - tip: 设备的特征值必须支持read才可以成功调用，具体参照 characteristic 的 properties 属性
     * - tip: 并行多次调用读写接口存在读写失败的可能性。
     * - tip: 如果读取超时，错误码 10015，[dd.onBLECharacteristicValueChange](#dd.onBLECharacteristicValueChange) 接口之后可能返回数据，需要接入方酌情处理。
     */
    readBLECharacteristicValue(options: ReadBLECharacteristicValueOptions): Promise<ReadBLECharacteristicValueSuccessResult>;
    /**
     * 启用低功耗蓝牙设备特征值变化时的 notify 功能。注意：设备的特征值必须支持 notify/indicate 才可以成功调用，具体参照 characteristic 的 properties 属性 另外，必须先启用 notify 才能监听到设备 characteristicValueChange 事件。
     *
     * Bug & Tip
     * - tip: 订阅操作成功后需要设备主动更新特征值的 value，才会触发 dd.onBLECharacteristicValueChange 。
     * - tip: 订阅方式效率比较高，推荐使用订阅代替 read 方式。
     */
    notifyBLECharacteristicValueChange(options: NotifyBLECharacteristicValueChangeOptions): Promise<any>;
    /**
     * 获取蓝牙设备所有 service（服务）
     *
     * Bug & Tip
     * - tip: 建立连接后先执行 dd.getBLEDeviceServices 与 dd.getBLEDeviceCharacteristics 后再进行与蓝牙设备的数据交互。
     */
    getBLEDeviceServices(options: {
        deviceId: string;
        complete?(res: any): void;
    }): Promise<GetBLEDeviceServicesSuccessResult>;
    /**
     * 获取蓝牙设备所有 characteristic（特征值）
     *
     * Bug & Tip
     * - tip: 建立连接后先执行 dd.getBLEDeviceServices 与 dd.getBLEDeviceCharacteristics 后再进行与蓝牙设备的数据交互。
     */
    getBLEDeviceCharacteristics(options: GetBLEDeviceCharacteristicsOptions): Promise<GetBLEDeviceCharacteristicsSuccessResult>;
    /**
     * 搜索到新的蓝牙设备时触发此事件。
     *
     * Bug & Tip
     * - tip: 模拟器可能无法获取 advertisData 及 RSSI ，请使用真机调试。
     * - tip: 开发者工具和 Android 上获取到的deviceId为设备 MAC 地址，iOS 上则为设备 uuid。因此deviceId不能硬编码到代码中，需要分平台处理，iOS可根据设备属性（localName/advertisData/manufacturerData等）进行动态匹配。
     * - tip: 若在 d.onBluetoothDeviceFound 回调中包含了某个蓝牙设备，则此设备会添加到 dd.getBluetoothDevices 接口获取到的数组中。
     */
    onBluetoothDeviceFound(callback: (res: {
        readonly devices: ReadonlyArray<Readonly<BluetoothDevice>>;
    }) => void): void;
    /**
     * 移除寻找到新的蓝牙设备事件的监听。
     *
     * Bug & Tip
     * - tip: 为防止多次注册事件监听导致一次事件多次回调，建议每次调用on方法监听事件之前，先调用off方法，关闭之前的事件监听。
     */
    offBluetoothDeviceFound(): void;
    /**
     * 监听低功耗蓝牙设备的特征值变化的事件。
     */
    onBLECharacteristicValueChange(callback: (res: OnBLECharacteristicValueChangeEvent) => void): void;
    /**
     * 移除低功耗蓝牙设备的特征值变化事件的监听。
     */
    offBLECharacteristicValueChange(): void;
    /**
     * 监听低功耗蓝牙连接的错误事件，包括设备丢失，连接异常断开等。
     *
     * Bug & Tip
     * - tip: 为防止多次注册事件监听导致一次事件多次回调，建议每次调用on方法监听事件之前，先调用off方法，关闭之前的事件监听。
     */
    onBLEConnectionStateChanged(callback: (res: {
        readonly deviceId: string;
        readonly connected: string;
    }) => void): void;
    /**
     * 移除低功耗蓝牙连接状态变化事件的监听。
     *
     * Bug & Tip
     * - tip: 为防止多次注册事件监听导致一次事件多次回调，建议每次调用on方法监听事件之前，先调用off方法，关闭之前的事件监听。
     */
    offBLEConnectionStateChanged(): void;
    /**
     * 监听本机蓝牙状态变化的事件。
     */
    onBluetoothAdapterStateChange(callback: (res: {
        readonly available: boolean;
        readonly discovering: boolean;
    }) => void): void;
    /**
     * 移除本机蓝牙状态变化的事件的监听。
     *
     * Bug & Tip
     * - tip: 为防止多次注册事件监听导致一次事件多次回调，建议每次调用on方法监听事件之前，先调用off方法，关闭之前的事件监听。
     */
    offBluetoothAdapterStateChange(): void;
    stopPullDownRefresh(): void;
    hideShareMenu(options?: {
        complete?(res: any): void;
    }): Promise<void>;
}
